package com.esv.freight.notice.module.sms.controller;

import com.alibaba.fastjson.JSONObject;
import com.esv.freight.notice.BaseTestController;
import com.esv.freight.notice.common.response.ECode;
import com.esv.freight.notice.module.sms.em.CaptchaEnum;
import lombok.extern.slf4j.Slf4j;
import org.junit.Assert;
import org.junit.FixMethodOrder;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.MethodSorters;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.http.MediaType;
import org.springframework.test.context.junit4.SpringRunner;
import org.springframework.test.web.servlet.MvcResult;
import org.springframework.test.web.servlet.request.MockMvcRequestBuilders;
import org.springframework.test.web.servlet.result.MockMvcResultHandlers;
import org.springframework.test.web.servlet.result.MockMvcResultMatchers;

/**
 * @description:
 * @project: notice-service
 * @name: com.esv.freight.notice.module.sms.controller.SmsCaptchaControllerTest
 * @author: 黄朝斌
 * @email: huangchaobin@esvtek.com
 * @createTime: 2020/04/16 15:14
 * @version:1.0
 */
@RunWith(SpringRunner.class)
@SpringBootTest
@Slf4j
@FixMethodOrder(MethodSorters.NAME_ASCENDING)
public class SmsCaptchaControllerTest extends BaseTestController {

    private static String PHONE = "13912345678";

    @Value("${sms.captcha.frequency:60}")
    private long SMS_CAPTCHA_FREQUENCY;

    @Value("${sms.captcha.total:20}")
    private int SMS_CAPTCHA_TOTAL;

    /**
     * 获取短信验证码成功
     **/
    @Test
    public void a1_sendSmsCaptcha_success_test() throws Exception {
        String url = "/sms/captcha/send";

        // 构造数据
        JSONObject reqJson = new JSONObject();
        reqJson.put("phone", PHONE);
        reqJson.put("type", CaptchaEnum.LOGIN.getType());

        MvcResult mvcResult = this.getMockMvc().perform(MockMvcRequestBuilders.post(url)
                .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
                .content(reqJson.toJSONString()))
                .andDo(MockMvcResultHandlers.print())
                .andExpect(MockMvcResultMatchers.status().isOk())
                .andReturn();

        String responseStr = mvcResult.getResponse().getContentAsString();
        log.info(responseStr);

        JSONObject result = JSONObject.parseObject(responseStr);
        Assert.assertEquals(ECode.SUCCESS.code(), result.getIntValue("code"));
    }

    /**
     * 获取短信验证码失败：type未定义
     **/
    @Test
    public void b1_sendSmsCaptcha_param_type_error_test() throws Exception {
        String url = "/sms/captcha/send";

        // 构造数据
        JSONObject reqJson = new JSONObject();
        reqJson.put("phone", PHONE);
        reqJson.put("type", "test");

        MvcResult mvcResult = this.getMockMvc().perform(MockMvcRequestBuilders.post(url)
                .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
                .content(reqJson.toJSONString()))
                .andDo(MockMvcResultHandlers.print())
                .andExpect(MockMvcResultMatchers.status().isOk())
                .andReturn();

        String responseStr = mvcResult.getResponse().getContentAsString();
        log.info(responseStr);

        JSONObject result = JSONObject.parseObject(responseStr);
        Assert.assertEquals(ECode.PARAM_ERROR.code(), result.getIntValue("code"));
    }

    /**
     * 获取短信验证码失败：无效的手机号
     **/
    @Test
    public void b2_sendSmsCaptcha_param_phone_error_test() throws Exception {
        String url = "/sms/captcha/send";

        // 构造数据
        JSONObject reqJson = new JSONObject();
        reqJson.put("phone", "1391234567x");
        reqJson.put("type", CaptchaEnum.LOGIN.getType());

        MvcResult mvcResult = this.getMockMvc().perform(MockMvcRequestBuilders.post(url)
                .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
                .content(reqJson.toJSONString()))
                .andDo(MockMvcResultHandlers.print())
                .andExpect(MockMvcResultMatchers.status().isOk())
                .andReturn();

        String responseStr = mvcResult.getResponse().getContentAsString();
        log.info(responseStr);

        JSONObject result = JSONObject.parseObject(responseStr);
        Assert.assertEquals(1001, result.getIntValue("code"));
    }

    /**
     * 获取短信验证码失败：超频
     **/
    @Test
    public void b3_sendSmsCaptcha_frequency_error_test() throws Exception {
        String url = "/sms/captcha/send";

        // 构造数据
        JSONObject reqJson = new JSONObject();
        reqJson.put("phone", PHONE);
        reqJson.put("type", CaptchaEnum.LOGIN.getType());

        MvcResult mvcResult = this.getMockMvc().perform(MockMvcRequestBuilders.post(url)
                .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
                .content(reqJson.toJSONString()))
                .andDo(MockMvcResultHandlers.print())
                .andExpect(MockMvcResultMatchers.status().isOk())
                .andReturn();

        String responseStr = mvcResult.getResponse().getContentAsString();
        log.info(responseStr);

        JSONObject result = JSONObject.parseObject(responseStr);
        Assert.assertEquals(1002, result.getIntValue("code"));
    }

    /**
     * 获取短信验证码失败：超量
     **/
    @Test
    public void b4_sendSmsCaptcha_total_error_test() throws Exception {
        String url = "/sms/captcha/send";

        // 构造数据
        JSONObject reqJson = new JSONObject();
        reqJson.put("phone", "13012345678");
        reqJson.put("type", CaptchaEnum.LOGIN.getType());
        for (int i = 0; i < SMS_CAPTCHA_TOTAL; i++) {
            MvcResult mvcResult = this.getMockMvc().perform(MockMvcRequestBuilders.post(url)
                    .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
                    .content(reqJson.toJSONString()))
                    .andDo(MockMvcResultHandlers.print())
                    .andExpect(MockMvcResultMatchers.status().isOk())
                    .andReturn();

            String responseStr = mvcResult.getResponse().getContentAsString();
            log.info("第{}次发送短信验证码", (i+1));
            log.info(responseStr);
            Thread.sleep(SMS_CAPTCHA_FREQUENCY * 1000);
        }

        MvcResult mvcResult = this.getMockMvc().perform(MockMvcRequestBuilders.post(url)
                .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
                .content(reqJson.toJSONString()))
                .andDo(MockMvcResultHandlers.print())
                .andExpect(MockMvcResultMatchers.status().isOk())
                .andReturn();

        String responseStr = mvcResult.getResponse().getContentAsString();
        log.info("第{}次发送短信验证码", (SMS_CAPTCHA_TOTAL+1));
        log.info(responseStr);

        JSONObject result = JSONObject.parseObject(responseStr);
        Assert.assertEquals(1003, result.getIntValue("code"));
    }

    /**
     * 校验短信验证码成功
     **/
    @Test
    public void c1_checkSmsCaptcha_success_test() throws Exception {
        String url = "/sms/captcha/check";

        // 构造数据
        JSONObject reqJson = new JSONObject();
        reqJson.put("phone", PHONE);
        reqJson.put("captcha", "325922");
        reqJson.put("type", CaptchaEnum.LOGIN.getType());

        MvcResult mvcResult = this.getMockMvc().perform(MockMvcRequestBuilders.post(url)
                .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
                .content(reqJson.toJSONString()))
                .andDo(MockMvcResultHandlers.print())
                .andExpect(MockMvcResultMatchers.status().isOk())
                .andReturn();

        String responseStr = mvcResult.getResponse().getContentAsString();
        log.info(responseStr);

        JSONObject result = JSONObject.parseObject(responseStr);
        Assert.assertEquals(ECode.SUCCESS.code(), result.getIntValue("code"));
    }

    /**
     * 校验短信验证码失败：验证码不合法
     **/
    @Test
    public void c1_checkSmsCaptcha_param_captcha_error_test() throws Exception {
        String url = "/sms/captcha/check";

        // 构造数据
        JSONObject reqJson = new JSONObject();
        reqJson.put("phone", PHONE);
        reqJson.put("captcha", "");
        reqJson.put("type", CaptchaEnum.LOGIN.getType());

        MvcResult mvcResult = this.getMockMvc().perform(MockMvcRequestBuilders.post(url)
                .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
                .content(reqJson.toJSONString()))
                .andDo(MockMvcResultHandlers.print())
                .andExpect(MockMvcResultMatchers.status().isOk())
                .andReturn();

        String responseStr = mvcResult.getResponse().getContentAsString();
        log.info(responseStr);

        JSONObject result = JSONObject.parseObject(responseStr);
        Assert.assertEquals(ECode.PARAM_ERROR.code(), result.getIntValue("code"));
    }

    /**
     * 校验短信验证码失败：验证码不合法
     **/
    @Test
    public void c1_checkSmsCaptcha_captcha_not_exits_error_test() throws Exception {
        String url = "/sms/captcha/check";

        // 构造数据
        JSONObject reqJson = new JSONObject();
        reqJson.put("phone", PHONE);
        reqJson.put("captcha", "325922");
        reqJson.put("type", CaptchaEnum.LOGIN.getType());

        MvcResult mvcResult = this.getMockMvc().perform(MockMvcRequestBuilders.post(url)
                .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
                .content(reqJson.toJSONString()))
                .andDo(MockMvcResultHandlers.print())
                .andExpect(MockMvcResultMatchers.status().isOk())
                .andReturn();

        String responseStr = mvcResult.getResponse().getContentAsString();
        log.info(responseStr);

        JSONObject result = JSONObject.parseObject(responseStr);
        Assert.assertEquals(1003, result.getIntValue("code"));
    }
}
