package com.esv.freight.notice.common.util;

import com.alibaba.fastjson.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.client.CookieStore;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLContextBuilder;
import org.apache.http.conn.ssl.TrustStrategy;
import org.apache.http.impl.client.*;
import org.springframework.http.*;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponentsBuilder;

import javax.net.ssl.SSLContext;
import java.net.URI;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.X509Certificate;
import java.util.Map;

@Slf4j
public class HttpUtils {

    /**
     * 从连接池中获取连接的超时时间（单位：毫秒）
     */
    public static final int CONNECTION_REQUEST_TIMEOUT_DEFAULT = 1000;

    /**
     * 与服务器连接超时时间（单位：毫秒）：httpclient会创建一个异步线程用以创建socket连接，此处设置该socket的连接超时时间
     */
    public static final int CONNECT_TIMEOUT_DEFAULT = 3000;

    public static final int CONNECT_TIMEOUT_10 = 10000;

    /**
     * socket读数据超时时间（单位：毫秒）：从服务器获取响应数据的超时时间
     */
    public static final int SOCKET_TIMEOUT_DEFAULT = 2000;

    public static String get(String url) throws Exception {
        return get(url, null);
    }

    public static String get(String url, Map<String, String> headerMap) throws Exception {
        HttpComponentsClientHttpRequestFactory factory = new HttpComponentsClientHttpRequestFactory();
        factory.setConnectionRequestTimeout(CONNECTION_REQUEST_TIMEOUT_DEFAULT);
        int connectTimeout = CONNECT_TIMEOUT_DEFAULT;
        factory.setConnectTimeout(connectTimeout);
        factory.setReadTimeout(SOCKET_TIMEOUT_DEFAULT);
        RestTemplate restTemplate = new RestTemplate(factory);

        // 特殊字符处理
        UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(url);
        URI uri = builder.build().encode().toUri();

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        // 设置Header
        if (null != headerMap) {
            for (Map.Entry<String, String> entry : headerMap.entrySet()) {
                headers.set(entry.getKey(), entry.getValue());
            }
        }

        HttpEntity<String> requestEntity = new HttpEntity<>(headers);

        ResponseEntity<String> result = restTemplate.exchange(uri, HttpMethod.GET, requestEntity, String.class);
        return result.getBody();
    }

    public static String post(String url, JSONObject bodyJson) {
        return post(url, null, bodyJson);
    }

    public static String post(String url, Map<String, String> headerMap, JSONObject bodyJson) {
        log.info("url:{}", url);
        log.info("headerMap:{}", headerMap.toString());
        log.info("body:{}", bodyJson.toJSONString());
        HttpComponentsClientHttpRequestFactory factory = new HttpComponentsClientHttpRequestFactory();
        factory.setConnectionRequestTimeout(CONNECTION_REQUEST_TIMEOUT_DEFAULT);
        int connectTimeout = CONNECT_TIMEOUT_DEFAULT;
        factory.setConnectTimeout(connectTimeout);
        factory.setReadTimeout(SOCKET_TIMEOUT_DEFAULT);
        RestTemplate restTemplate = new RestTemplate(factory);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        // 设置Header
        if (null != headerMap) {
            for (Map.Entry<String, String> entry : headerMap.entrySet()) {
                headers.set(entry.getKey(), entry.getValue());
            }
        }

        HttpEntity<String> requestEntity = new HttpEntity<>(bodyJson.toString(), headers);
        String result = restTemplate.postForObject(url, requestEntity, String.class);
        log.info("response:{}", result);
        return result;
    }

    private static CloseableHttpClient createSSLInsecureClient() {
        int connectTimeout = CONNECT_TIMEOUT_DEFAULT;
        return createSSLInsecureClient(CONNECTION_REQUEST_TIMEOUT_DEFAULT, connectTimeout, SOCKET_TIMEOUT_DEFAULT);
    }

    private static CloseableHttpClient createSSLInsecureClient(int connectionRequestTimeout, int connectTimeout, int socketTimeout) {
        CookieStore cookieStore = new BasicCookieStore();

        RequestConfig requestConfig = RequestConfig.custom()
                // 从连接池中获取连接的超时时间
                .setConnectionRequestTimeout(connectionRequestTimeout)
                // 与服务器连接超时时间：httpclient会创建一个异步线程用以创建socket连接，此处设置该socket的连接超时时间
                .setConnectTimeout(connectTimeout)
                // socket读数据超时时间：从服务器获取响应数据的超时时间
                .setSocketTimeout(socketTimeout)
                .build();

        try {
            SSLContext sslContext = new SSLContextBuilder()
                    .loadTrustMaterial(null, new TrustStrategy() {
                        //信任所有
                        @Override
                        public boolean isTrusted(X509Certificate[] chain, String authType) {
                            return true;
                        }
                    }).build();
            SSLConnectionSocketFactory sslsf = new SSLConnectionSocketFactory(sslContext);
            return HttpClients.custom()
                    .setSSLSocketFactory(sslsf)
                    .setKeepAliveStrategy(new DefaultConnectionKeepAliveStrategy())
                    .setRedirectStrategy(new DefaultRedirectStrategy())
                    .setDefaultRequestConfig(requestConfig)
                    .setDefaultCookieStore(cookieStore)
                    .build();
        } catch (KeyManagementException e) {
            log.error(e.getMessage(), ThrowableUtils.toString(e));
        } catch (NoSuchAlgorithmException e) {
            log.error(e.getMessage(), ThrowableUtils.toString(e));
        } catch (KeyStoreException e) {
            log.error(e.getMessage(), ThrowableUtils.toString(e));
        }
        return HttpClients.createDefault();
    }

}
