package com.esv.freight.notice.module.msg.service;

import com.alibaba.fastjson.JSONObject;
import com.esv.freight.notice.common.exception.EException;
import com.esv.freight.notice.common.response.ECode;
import com.esv.freight.notice.common.util.HttpUtils;
import com.esv.freight.notice.common.util.ThrowableUtils;
import com.esv.freight.notice.module.msg.MsgConstants;
import com.esv.freight.notice.module.msg.form.DevicePushForm;
import com.esv.freight.notice.module.msg.pojo.JPushObjectInfo;
import com.esv.freight.notice.module.msg.pojo.JPushResult;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;

import java.util.*;

/**
 * @description:
 * @project: freight-notice-service
 * @name: com.esv.freight.notice.module.msg.service.JPushService
 * @author: 黄朝斌
 * @email: huangchaobin@esvtek.com
 * @createTime: 2020/06/08 11:07
 * @version:1.0
 */
@Service("jPushService")
@RefreshScope
@Slf4j
public class JPushService {

    @Value("${msg.jpush.driver.app-key}")
    private String driverAppKey;

    @Value("${msg.jpush.driver.master-secret}")
    private String driverMasterSecret;

    @Value("${msg.jpush.goodsowner.app-key}")
    private String goodsOwnerAppKey;

    @Value("${msg.jpush.goodsowner.master-secret}")
    private String goodsOwnerMasterSecret;

    /**
     * description 获取推送消息
     * param [form]
     * return com.esv.freight.notice.module.msg.pojo.JPushObjectInfo
     * author HuangChaobin
     * createTime 2020/06/09 17:20
     **/
    public JPushObjectInfo getPushObjectInfo(DevicePushForm form) {
        JPushObjectInfo jPushObjectInfo = new JPushObjectInfo();
        BeanUtils.copyProperties(form, jPushObjectInfo);
        if (1 == form.getAppType()) {
            jPushObjectInfo.setAppKey(goodsOwnerAppKey);
            jPushObjectInfo.setMasterSecret(goodsOwnerMasterSecret);
        } else {
            jPushObjectInfo.setAppKey(driverAppKey);
            jPushObjectInfo.setMasterSecret(driverMasterSecret);
        }

        return jPushObjectInfo;
    }

    /**
     * description 推送消息到指定设备
     * param [jPushObjectInfo]
     * return com.esv.freight.notice.module.msg.pojo.JPushResult
     * author HuangChaobin
     * createTime 2020/06/09 11:14
     **/
    public JPushResult pushMessage(JPushObjectInfo jPushObjectInfo) {
        JSONObject jPushJsonObject = new JSONObject();

        // 推送平台设置
        if (1 == jPushObjectInfo.getDeviceType()) {
            jPushJsonObject.put("platform", "android");
        } else {
            jPushJsonObject.put("platform", "ios");
        }

        // 推送设备指定
        List<String> registrationIdList = new ArrayList<>(1);
        registrationIdList.add(jPushObjectInfo.getRegistrationId());
        JSONObject registrationIdJson = new JSONObject();
        registrationIdJson.put("registration_id", registrationIdList);
        jPushJsonObject.put("audience", registrationIdJson);

        // 设置消息内容体
        jPushJsonObject.put("message", jPushObjectInfo.getMessage());

        // 推送消息
        Map<String, String> headerMap = new HashMap<>(1);
        headerMap.put("Authorization", this.getPushAuthorization(jPushObjectInfo.getAppType()));
        String pushResult;
        try {
            pushResult = HttpUtils.post(MsgConstants.PUSH_API_V3, headerMap, jPushJsonObject);
        } catch (Exception e) {
            log.error("调用极光推送时发生错误：{}. {}", e.getMessage(), ThrowableUtils.toString(e));
            throw new EException(ECode.THIRD_PARTY_ERROR.code(), "请求极光推送服务时发生错误");
        }

        // 判断推送结果
        JSONObject pushResultJson = JSONObject.parseObject(pushResult);
        JPushResult jPushResult = new JPushResult();
        jPushResult.setPushContent(jPushJsonObject);
        if (pushResultJson.containsKey("error")) {
            // 推送失败
            jPushResult.setResult(false);
            jPushResult.setCode(pushResultJson.getInteger("code"));
            jPushResult.setMessage(pushResultJson.getString("message"));
        } else {
            // 推送成功
            jPushResult.setResult(true);
            jPushResult.setSendNO(pushResultJson.getInteger("sendno"));
            jPushResult.setMsgId(pushResultJson.getString("msg_id"));
        }

        return jPushResult;
    }

    /**
     * description 获取HTTP Header（头）Authorization
     * param [appType]
     * return java.lang.String
     * author HuangChaobin
     * createTime 2020/06/08 17:27
     **/
    private String getPushAuthorization(Integer appType) {
        String authorization;
        String authInfo;
        // APP类型：1-货主、2-司机
        if (1 == appType) {
            authInfo = goodsOwnerAppKey + ":" + goodsOwnerMasterSecret;
        } else {
            authInfo = driverAppKey + ":" + driverMasterSecret;
        }
        authorization = "Basic " + Base64.getEncoder().encodeToString(authInfo.getBytes());
        return authorization;
    }

}
