package com.esv.freight.notice.module.sms.service.impl;

import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.esv.freight.notice.common.component.RedisComponent;
import com.esv.freight.notice.common.em.DbDeletedEnum;
import com.esv.freight.notice.common.exception.EException;
import com.esv.freight.notice.common.util.DateUtils;
import com.esv.freight.notice.common.util.SmsCaptchaUtils;
import com.esv.freight.notice.module.sms.dao.SmsCaptchaDao;
import com.esv.freight.notice.module.sms.entity.SmsCaptchaEntity;
import com.esv.freight.notice.module.sms.form.CaptchaForm;
import com.esv.freight.notice.module.sms.service.SmsCaptchaService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.concurrent.TimeUnit;


@Service("smsCaptchaService")
@RefreshScope
@Slf4j
public class SmsCaptchaServiceImpl extends ServiceImpl<SmsCaptchaDao, SmsCaptchaEntity> implements SmsCaptchaService {

    @Value("${spring.application.name}")
    private String applicationName;

    /**
     * 短信验证码频次，默认60秒
     **/
    @Value("${sms.captcha.frequency:60}")
    private long SMS_CAPTCHA_FREQUENCY;

    /**
     * 短信验证码总数，默认20
     **/
    @Value("${sms.captcha.total:20}")
    private int SMS_CAPTCHA_TOTAL;

    /**
     * 短信验证码有效期，默认10分钟
     **/
    @Value("${sms.captcha.period:10}")
    private long SMS_CAPTCHA_PERIOD;

    private RedisComponent redisComponent;

    @Autowired
    public SmsCaptchaServiceImpl(RedisComponent redisComponent) {
        this.redisComponent = redisComponent;
    }

    @Override
    public void sendSmsCaptcha(CaptchaForm form) throws EException {
        String phone = form.getPhone();
        // 1.校验短信验证码
        this.checkSendSmsCaptcha(form);

        // 2.获取短信验证码随机数
        String smsCaptcha = SmsCaptchaUtils.getCaptcha();

        // 3.TODO，调用第三方服务发送短信验证码

        // 4.将短信验证码放入缓存
        // 频次
        redisComponent.set(this.getFrequencyCacheKey(phone), "yes", SMS_CAPTCHA_FREQUENCY);
        // 值
        redisComponent.set(this.getValueCacheKey(form), smsCaptcha, SMS_CAPTCHA_PERIOD, TimeUnit.MINUTES);
        // 总数量
        int total;
        if (redisComponent.hasKey(this.getTotalCacheKey(phone))) {
            total = (int) redisComponent.get(this.getTotalCacheKey(phone)) + 1;
        } else {
            total = 1;
        }
        redisComponent.set(this.getTotalCacheKey(phone), total, 1, TimeUnit.DAYS);

        // 5.DB存储短信验证码
        SmsCaptchaEntity entity = new SmsCaptchaEntity();
        entity.setPhone(phone);
        entity.setCaptcha(smsCaptcha);
        entity.setCaptchaType(form.getType());
        entity.setInvalidTime(DateUtils.plusMinutes(new Date(), Integer.parseInt(String.valueOf(SMS_CAPTCHA_PERIOD))));
        this.baseMapper.insert(entity);
    }

    @Override
    public void checkSmsCaptcha(CaptchaForm form) throws EException {
        // 1.获取短信验证码
        String rightCaptcha = null;
        String captchaCacheKey = this.getValueCacheKey(form);
        if (redisComponent.hasKey(captchaCacheKey)) {
            // 从缓存获取验证码
            rightCaptcha = redisComponent.get(captchaCacheKey).toString();
        } else {
            // 从DB查询短信验证码
            SmsCaptchaEntity queryEntity = new SmsCaptchaEntity();
            queryEntity.setPhone(form.getPhone());
            queryEntity.setCaptchaType(form.getType());
            SmsCaptchaEntity smsCaptchaEntity = this.baseMapper.selectLatestCaptcha(queryEntity);
            if (null == smsCaptchaEntity) {
                throw new EException(1003, "验证码不存在或已失效");
            }
            Date InvalidTime = smsCaptchaEntity.getInvalidTime();
            if (0 <= InvalidTime.compareTo(new Date())) {
                rightCaptcha = smsCaptchaEntity.getCaptcha();
            }
        }

        // 2.校验短信验证码
        if (!rightCaptcha.equals(form.getCaptcha())) {
            throw new EException(1002, "验证码错误");
        }

        // 3.短信验证码正确，失效操作
        if (redisComponent.hasKey(captchaCacheKey)) {
            redisComponent.del(captchaCacheKey);
        }
        SmsCaptchaEntity updateEntity = new SmsCaptchaEntity();
        updateEntity.setDeleted(DbDeletedEnum.YES.getCode());
        updateEntity.setVerifyTime(new Date());
        updateEntity.setPhone(form.getPhone());
        updateEntity.setCaptchaType(form.getType());
        this.baseMapper.updateLatestCaptcha(updateEntity);
    }

    /**
     * description 校验短信验证码
     * param [form]
     * return void
     * author Administrator
     * createTime 2020/04/16 9:29
     **/
    private void checkSendSmsCaptcha(CaptchaForm form) throws EException {
        String phone = form.getPhone();
        // 校验频次
        if (redisComponent.hasKey(this.getFrequencyCacheKey(phone))) {
            throw new EException(1002, "获取短信验证码过于频繁，请稍后再试");
        }
        // 校验总数
        Object total = redisComponent.get(this.getTotalCacheKey(phone));
        if (null != total && SMS_CAPTCHA_TOTAL <= Integer.parseInt(total.toString())) {
            throw new EException(1003, "当日短信验证码总数已达上限，请明天再试");
        }
    }

    /**
     * 获取短信验证码频次缓存key
     **/
    private String getFrequencyCacheKey(String phone) {
        StringBuffer sb = new StringBuffer();
        sb.append(applicationName);
        sb.append("::sms::captcha::frequency::");
        sb.append(phone);
        return sb.toString();
    }

    /**
     * 获取短信验证码总数缓存key
     **/
    private String getTotalCacheKey(String phone) {
        StringBuffer sb = new StringBuffer();
        sb.append(applicationName);
        sb.append("::sms::captcha::total::");
        sb.append(DateUtils.getSysdateStr(DateUtils.DATE_FORMAT2));
        sb.append("::");
        sb.append(phone);
        return sb.toString();
    }

    /**
     * 获取短信验证码内容缓存key
     **/
    private String getValueCacheKey(CaptchaForm form) {
        StringBuffer sb = new StringBuffer();
        sb.append(applicationName);
        sb.append("::sms::captcha::value::");
        sb.append(form.getPhone());
        sb.append("::");
        sb.append(form.getType());
        return sb.toString();
    }
}