package com.esv.datacenter.iot.module.devicemodel.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.esv.datacenter.iot.common.component.MqttAcl;
import com.esv.datacenter.iot.common.component.MqttClientAuthComponent;
import com.esv.datacenter.iot.common.exception.EException;
import com.esv.datacenter.iot.common.response.ECode;
import com.esv.datacenter.iot.common.vo.PageResultVO;
import com.esv.datacenter.iot.module.datamodel.entity.DataModelEntity;
import com.esv.datacenter.iot.module.devicemodel.dao.DeviceInstanceDao;
import com.esv.datacenter.iot.module.devicemodel.dto.DeviceInstanceDto;
import com.esv.datacenter.iot.module.devicemodel.dto.DeviceInstanceFullMapDto;
import com.esv.datacenter.iot.module.devicemodel.entity.DeviceInstanceEntity;
import com.esv.datacenter.iot.module.devicemodel.form.DeviceInstanceForm;
import com.esv.datacenter.iot.module.devicemodel.service.DeviceDataMapService;
import com.esv.datacenter.iot.module.devicemodel.service.DeviceInstanceService;
import com.esv.datacenter.iot.module.devicemodel.vo.DeviceInstanceAccessInfoVO;
import com.esv.datacenter.iot.module.devicemodel.vo.DeviceInstanceMqttTopicInfoVO;
import com.esv.datacenter.iot.module.devicemodel.vo.DeviceInstanceVO;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;


@Service("deviceInstanceService")
public class DeviceInstanceServiceImpl extends ServiceImpl<DeviceInstanceDao, DeviceInstanceEntity> implements DeviceInstanceService {

    @Autowired
    MqttClientAuthComponent mqttClientAuthComponent;

    @Autowired
    DeviceDataMapService deviceDataMapService;

    @Override
    public int getInstanceCountByTypeId(Long deviceTypeId) {
        return this.getBaseMapper().selectCount(new LambdaQueryWrapper<DeviceInstanceEntity>()
                .eq(DeviceInstanceEntity::getDeviceTypeId, deviceTypeId));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long insertDeviceInstance(DeviceInstanceForm form) {
        int count = this.getBaseMapper().selectCount(new LambdaQueryWrapper<DeviceInstanceEntity>()
                .eq(DeviceInstanceEntity::getName, form.getName()));
        if (0 < count) {
            throw new EException(ECode.BIZ_PARAM_ERROR.code(), "设备名称[" + form.getName() + "]已存在");
        }

        DeviceInstanceEntity entity = new DeviceInstanceEntity();
        BeanUtils.copyProperties(form, entity);
        String communicationId = UUID.randomUUID().toString().replaceAll("-", "");
        entity.setCommunicationId(communicationId);
        this.getBaseMapper().insert(entity);
        Long deviceInstanceId = entity.getId();

        // 保存client Mqtt Topic发布/订阅权限
        this.saveInstanceMqttTopicAuth(form.getDeviceTypeId(), deviceInstanceId, communicationId);

        return deviceInstanceId;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteInstance(Long id) {
        // 获取实例信息
        DeviceInstanceEntity deviceInstanceEntity = this.getInstanceById(id);

        // 删除实例
        this.getBaseMapper().deleteById(id);

        // 删除设备Mqtt Topic权限记录
        mqttClientAuthComponent.deleteClientAcl(deviceInstanceEntity.getCommunicationId());
    }

    @Override
    public void updateInstance(DeviceInstanceForm form) {
        if (Objects.nonNull(StringUtils.trimToNull(form.getName()))) {
            int count = this.getBaseMapper().selectCount(new LambdaQueryWrapper<DeviceInstanceEntity>()
                    .ne(DeviceInstanceEntity::getId, form.getId())
                    .eq(DeviceInstanceEntity::getName, form.getName()));
            if (0 < count) {
                throw new EException(ECode.BIZ_PARAM_ERROR.code(), "设备名称[" + form.getName() + "]已存在");
            }
        }

        DeviceInstanceEntity entity = new DeviceInstanceEntity();
        BeanUtils.copyProperties(form, entity);
        this.getBaseMapper().updateById(entity);
    }

    @Override
    public Boolean isValidId(Long id) {
        DeviceInstanceEntity entity = this.getInstanceById(id);
        if (Objects.isNull(entity)) {
            return false;
        } else {
            return true;
        }
    }

    @Override
    public DeviceInstanceEntity getInstanceById(Long id) {
        return this.getBaseMapper().selectById(id);
    }

    @Override
    public PageResultVO getInstance4Page(DeviceInstanceForm form) {
        IPage<DeviceInstanceDto> page = new Page<>(form.getPageNum(), form.getPageSize());
        this.baseMapper.select4Page(page, form);

        List<DeviceInstanceDto> entityList = page.getRecords();
        List<DeviceInstanceVO> voList = new ArrayList<>();
        for (DeviceInstanceDto entity : entityList) {
            DeviceInstanceVO vo = new DeviceInstanceVO();
            BeanUtils.copyProperties(entity, vo);
            vo.setCreateTime(entity.getCreateTime().getTime());
            vo.setUpdateTime(entity.getUpdateTime().getTime());
            voList.add(vo);
        }

        return new PageResultVO(page, voList);
    }

    @Override
    public List<DeviceInstanceVO> getInstanceListByTypeId(Long deviceTypeId) {
        List<DeviceInstanceEntity> entityList = this.getBaseMapper().selectList(new LambdaQueryWrapper<DeviceInstanceEntity>()
                .eq(DeviceInstanceEntity::getDeviceTypeId, deviceTypeId)
                .orderByAsc(DeviceInstanceEntity::getName));

        List<DeviceInstanceVO> voList = new ArrayList<>();
        for (DeviceInstanceEntity entity : entityList) {
            DeviceInstanceVO vo = new DeviceInstanceVO();
            BeanUtils.copyProperties(entity, vo);
            vo.setCreateTime(entity.getCreateTime().getTime());
            vo.setUpdateTime(entity.getUpdateTime().getTime());
            voList.add(vo);
        }

        return voList;
    }

    @Override
    public Map<String, Long> getInstanceCountByType(Long deviceTypeId) {
        Map<String, Object> queryObj = new HashMap<>(1);
        if (Objects.nonNull(deviceTypeId)) {
            queryObj.put("deviceTypeId", deviceTypeId);
        }
        List<Map<String, Long>> instanceCountMapList = this.getBaseMapper().selectInstanceCountByType(queryObj);

        Map<String, Long> instanceCountMap = new HashMap<>(instanceCountMapList.size());
        for (Map<String, Long> map : instanceCountMapList) {
            instanceCountMap.put(String.valueOf(map.get("device_type_id")), map.get("count"));
        }

        return instanceCountMap;
    }

    @Override
    public DeviceInstanceEntity getInstanceByCommunicationId(String communicationId) {
        return this.getBaseMapper().selectOne(new LambdaQueryWrapper<DeviceInstanceEntity>()
                .eq(DeviceInstanceEntity::getCommunicationId, communicationId));
    }

    @Override
    public List<DeviceInstanceDto> get4List(DeviceInstanceDto dto) {
        return this.getBaseMapper().select4List(dto);
    }

    @Override
    public void changeDeviceOnlineState(List<String> communicationIdList) {
        // 将在线设备状态修改为1
        UpdateWrapper<DeviceInstanceEntity> updateWrapper = new UpdateWrapper<>();
        updateWrapper.isNotNull("communication_id");
        updateWrapper.in("communication_id", communicationIdList);
        updateWrapper.ne("online_state", 1);
        DeviceInstanceEntity entity = new DeviceInstanceEntity();
        entity.setOnlineUpdateTime(new Date());
        entity.setOnlineState(1);
        baseMapper.update(entity, updateWrapper);
        // 将不在线设备状态修改为0
        updateWrapper = new UpdateWrapper<>();
        updateWrapper.isNotNull("communication_id");
        updateWrapper.notIn("communication_id", communicationIdList);
        updateWrapper.ne("online_state", 0);
        entity.setOnlineState(0);
        baseMapper.update(entity, updateWrapper);
    }

    @Override
    public DeviceInstanceVO deviceInstanceDetail(Long deviceInstanceId) {
        DeviceInstanceEntity deviceInstanceEntity = baseMapper.selectById(deviceInstanceId);
        if (deviceInstanceEntity == null) {
            return null;
        }
        DeviceInstanceVO deviceInstanceVO = new DeviceInstanceVO();
        BeanUtils.copyProperties(deviceInstanceEntity, deviceInstanceVO);
        deviceInstanceVO.setCreateTime(deviceInstanceEntity.getCreateTime().getTime());
        deviceInstanceVO.setUpdateTime(deviceInstanceEntity.getUpdateTime().getTime());
        return deviceInstanceVO;
    }

    @Override
    public void saveInstanceMqttTopicAuth(Long deviceTypeId, Long deviceInstanceId, String clientId) {
        // 获取设备对应的数据模型
        List<DataModelEntity> dataModelEntityList = deviceDataMapService.getDataModelByDeviceTypeId(deviceTypeId);
        if (Objects.isNull(dataModelEntityList) || 0 == dataModelEntityList.size()) {
            return;
        }

        List<MqttAcl> mqttAclList = new ArrayList<>();
        for (DataModelEntity dataModelEntity : dataModelEntityList) {
            MqttAcl mqttAcl = new MqttAcl();
            mqttAcl.setAllow(1);
            mqttAcl.setClientid(clientId);
            mqttAcl.setAccess(2);
            StringBuffer sb = new StringBuffer();
            sb.append("$esv/iot/")
                    .append(dataModelEntity.getId()).append("/")
                    .append(deviceTypeId).append("/")
                    .append(deviceInstanceId).append("/data/upload");
            mqttAcl.setTopic(sb.toString());
            mqttAclList.add(mqttAcl);
        }
        mqttClientAuthComponent.saveClientAcl(mqttAclList);
    }

    @Override
    public List<DeviceInstanceFullMapDto> getDeviceInstanceFullMap(Long instanceId) {
        return this.getBaseMapper().selectDeviceInstanceFullMap(instanceId);
    }

    @Override
    public DeviceInstanceAccessInfoVO getAccessInfoByInstanceId(Long id) {
        // 获取设备实例的设备类型、数据模型映射
        List<DeviceInstanceFullMapDto> deviceInstanceFullMapDtoList = this.getDeviceInstanceFullMap(id);

        // 获取设备对应的Mqtt Topic信息
        String clientId  = deviceInstanceFullMapDtoList.get(0).getCommunicationId();
        List<MqttAcl> mqttAclList = mqttClientAuthComponent.getClientTopicInfo(clientId);
        Map<Long, String> mqttTopicMap = new HashMap<>(32);
        for (MqttAcl mqttAcl : mqttAclList) {
            Long dataModelId = Long.parseLong(mqttAcl.getTopic().replaceFirst("\\$esv/iot/", "").substring(0, 1));
            mqttTopicMap.put(dataModelId, mqttAcl.getTopic());
        }

        // 构造返回数据
        DeviceInstanceAccessInfoVO vo = new DeviceInstanceAccessInfoVO();
        vo.setInstanceId(deviceInstanceFullMapDtoList.get(0).getInstanceId());
        vo.setInstanceName(deviceInstanceFullMapDtoList.get(0).getInstanceName());
        vo.setCommunicationId(deviceInstanceFullMapDtoList.get(0).getCommunicationId());
        vo.setDeviceTypeId(deviceInstanceFullMapDtoList.get(0).getDeviceTypeId());
        vo.setDeviceTypeName(deviceInstanceFullMapDtoList.get(0).getDeviceTypeName());

        List<DeviceInstanceMqttTopicInfoVO> mqttTopicInfoVOList = new ArrayList<>();
        for (DeviceInstanceFullMapDto dto : deviceInstanceFullMapDtoList) {
            DeviceInstanceMqttTopicInfoVO topicInfoVO = new DeviceInstanceMqttTopicInfoVO();
            if (Objects.isNull(dto.getDataModelId())) {
                continue;
            }
            topicInfoVO.setDataModelId(dto.getDataModelId());
            topicInfoVO.setDataModelName(dto.getDataModelName());
            topicInfoVO.setMqttTopic(mqttTopicMap.get(dto.getDataModelId()));
            mqttTopicInfoVOList.add(topicInfoVO);
        }
        vo.setMqttTopicInfoVOList(mqttTopicInfoVOList);

        return vo;
    }
}