package com.esv.datacenter.iot.module.devicemodel.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.esv.datacenter.iot.common.exception.EException;
import com.esv.datacenter.iot.common.response.ECode;
import com.esv.datacenter.iot.common.vo.PageResultVO;
import com.esv.datacenter.iot.module.devicemodel.dao.DeviceInstanceDao;
import com.esv.datacenter.iot.module.devicemodel.dto.DeviceInstanceDto;
import com.esv.datacenter.iot.module.devicemodel.entity.DeviceInstanceEntity;
import com.esv.datacenter.iot.module.devicemodel.form.DeviceInstanceForm;
import com.esv.datacenter.iot.module.devicemodel.service.DeviceInstanceService;
import com.esv.datacenter.iot.module.devicemodel.vo.DeviceInstanceVO;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;


@Service("deviceInstanceService")
public class DeviceInstanceServiceImpl extends ServiceImpl<DeviceInstanceDao, DeviceInstanceEntity> implements DeviceInstanceService {

    @Override
    public int getInstanceCountByTypeId(Long deviceTypeId) {
        return this.getBaseMapper().selectCount(new LambdaQueryWrapper<DeviceInstanceEntity>()
                .eq(DeviceInstanceEntity::getDeviceTypeId, deviceTypeId));
    }

    @Override
    public Long insertDeviceInstance(DeviceInstanceForm form) {
        int count = this.getBaseMapper().selectCount(new LambdaQueryWrapper<DeviceInstanceEntity>()
                .eq(DeviceInstanceEntity::getName, form.getName()));
        if (0 < count) {
            throw new EException(ECode.BIZ_PARAM_ERROR.code(), "设备名称[" + form.getName() + "]已存在");
        }

        DeviceInstanceEntity entity = new DeviceInstanceEntity();
        BeanUtils.copyProperties(form, entity);
        entity.setCommunicationId(UUID.randomUUID().toString().replaceAll("-", ""));
        this.getBaseMapper().insert(entity);

        return entity.getId();
    }

    @Override
    public void deleteInstance(Long id) {
        this.getBaseMapper().deleteById(id);
    }

    @Override
    public void updateInstance(DeviceInstanceForm form) {
        if (Objects.nonNull(StringUtils.trimToNull(form.getName()))) {
            int count = this.getBaseMapper().selectCount(new LambdaQueryWrapper<DeviceInstanceEntity>()
                    .ne(DeviceInstanceEntity::getId, form.getId())
                    .eq(DeviceInstanceEntity::getName, form.getName()));
            if (0 < count) {
                throw new EException(ECode.BIZ_PARAM_ERROR.code(), "设备名称[" + form.getName() + "]已存在");
            }
        }

        DeviceInstanceEntity entity = new DeviceInstanceEntity();
        BeanUtils.copyProperties(form, entity);
        this.getBaseMapper().updateById(entity);
    }

    @Override
    public Boolean isValidId(Long id) {
        DeviceInstanceEntity entity = this.getInstanceById(id);
        if (Objects.isNull(entity)) {
            return false;
        } else {
            return true;
        }
    }

    @Override
    public DeviceInstanceEntity getInstanceById(Long id) {
        return this.getBaseMapper().selectById(id);
    }

    @Override
    public PageResultVO getInstance4Page(DeviceInstanceForm form) {
        IPage<DeviceInstanceDto> page = new Page<>(form.getPageNum(), form.getPageSize());
        this.baseMapper.select4Page(page, form);

        List<DeviceInstanceDto> entityList = page.getRecords();
        List<DeviceInstanceVO> voList = new ArrayList<>();
        for (DeviceInstanceDto entity : entityList) {
            DeviceInstanceVO vo = new DeviceInstanceVO();
            BeanUtils.copyProperties(entity, vo);
            vo.setCreateTime(entity.getCreateTime().getTime());
            vo.setUpdateTime(entity.getUpdateTime().getTime());
            voList.add(vo);
        }

        return new PageResultVO(page, voList);
    }

    @Override
    public List<DeviceInstanceVO> getInstanceListByTypeId(Long deviceTypeId) {
        List<DeviceInstanceEntity> entityList = this.getBaseMapper().selectList(new LambdaQueryWrapper<DeviceInstanceEntity>()
                .eq(DeviceInstanceEntity::getDeviceTypeId, deviceTypeId)
                .orderByAsc(DeviceInstanceEntity::getName));

        List<DeviceInstanceVO> voList = new ArrayList<>();
        for (DeviceInstanceEntity entity : entityList) {
            DeviceInstanceVO vo = new DeviceInstanceVO();
            BeanUtils.copyProperties(entity, vo);
            vo.setCreateTime(entity.getCreateTime().getTime());
            vo.setUpdateTime(entity.getUpdateTime().getTime());
            voList.add(vo);
        }

        return voList;
    }

    @Override
    public Map<String, Long> getInstanceCountByType(Long deviceTypeId) {
        Map<String, Object> queryObj = new HashMap<>(1);
        if (Objects.nonNull(deviceTypeId)) {
            queryObj.put("deviceTypeId", deviceTypeId);
        }
        List<Map<String, Long>> instanceCountMapList = this.getBaseMapper().selectInstanceCountByType(queryObj);

        Map<String, Long> instanceCountMap = new HashMap<>(instanceCountMapList.size());
        for (Map<String, Long> map : instanceCountMapList) {
            instanceCountMap.put(String.valueOf(map.get("device_type_id")), map.get("count"));
        }

        return instanceCountMap;
    }

    @Override
    public DeviceInstanceEntity getInstanceByCommunicationId(String communicationId) {
        return this.getBaseMapper().selectOne(new LambdaQueryWrapper<DeviceInstanceEntity>()
                .eq(DeviceInstanceEntity::getCommunicationId, communicationId));
    }

    @Override
    public List<DeviceInstanceDto> get4List() {
        return this.getBaseMapper().select4List();
    }

    @Override
    public void changeDeviceOnlineState(List<String> communicationIdList) {
        // 将在线设备状态修改为1
        UpdateWrapper<DeviceInstanceEntity> updateWrapper = new UpdateWrapper<>();
        updateWrapper.isNotNull("communication_id");
        updateWrapper.in("communication_id", communicationIdList);
        updateWrapper.ne("online_state", 1);
        DeviceInstanceEntity entity = new DeviceInstanceEntity();
        entity.setOnlineUpdateTime(new Date());
        entity.setOnlineState(1);
        baseMapper.update(entity, updateWrapper);
        // 将不在线设备状态修改为0
        updateWrapper = new UpdateWrapper<>();
        updateWrapper.isNotNull("communication_id");
        updateWrapper.notIn("communication_id", communicationIdList);
        updateWrapper.ne("online_state", 0);
        entity.setOnlineState(0);
        baseMapper.update(entity, updateWrapper);
    }

}