package com.esv.datacenter.iot.common.util;

import com.alibaba.fastjson.JSONObject;
import com.esv.datacenter.iot.common.constants.CommonConstants;
import com.esv.datacenter.iot.common.exception.EException;
import com.esv.datacenter.iot.common.response.ECode;
import com.esv.gateway.common.GatewayHeaders;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.Map;

/**
 * @description: Http请求工具类
 * @author: 黄朝斌
 * @email: huangchaobin@esvtek.com
 * @createTime: 2020/05/26 19:09
 * @version:1.0
 */
@Slf4j
public class ReqUtils {

    public static String getRequestHeader(String headerKey) {
        ServletRequestAttributes servletRequestAttributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
        if (null == servletRequestAttributes) {
            Object value = ThreadLocalCacheUtils.getByKey(headerKey);
            return null == value ? null : value.toString();
        } else {
            HttpServletRequest request = servletRequestAttributes.getRequest();
            return request.getHeader(headerKey);
        }
    }

    /**
     * description 获取当前请求的数据权限
     * param []
     * return java.lang.String
     * author Administrator
     * createTime 2020/05/25 13:55
     **/
    public static String getRequestDataPerm() {
        return getRequestHeader(GatewayHeaders.DATA_PERM);
    }

    /**
     * description 获取当前请求的用户ID
     * param []
     * return java.lang.Long
     * author Administrator
     * createTime 2020/04/28 14:08
     **/
    public static Long getRequestUserId() {
        String userId = getRequestHeader(GatewayHeaders.USER_ID);
        if (StringUtils.isNotBlank(userId)) {
            return Long.parseLong(userId);
        } else {
            return null;
        }
    }

    /**
     * description 获取当前请求的租户ID
     * param []
     * return java.lang.Long
     * author Administrator
     * createTime 2020/05/25 13:55
     **/
    public static Long getRequestTenantId() {
        String tenantId = getRequestHeader(GatewayHeaders.TENANT_ID);
        if (StringUtils.isNotBlank(tenantId)) {
            return Long.parseLong(tenantId);
        } else {
            return null;
        }
    }

    /**
     * @description 获取当前请求的部门ID
     * @return java.lang.Long
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/06 15:56
     **/
    public static Long getRequestDepartmentId() {
        String departmentId = getRequestHeader(GatewayHeaders.DEPARTMENT_ID);
        if (StringUtils.isNotBlank(departmentId)) {
            return Long.parseLong(departmentId);
        } else {
            return null;
        }
    }

    /**
     * description 获取当前请求的用户帐号
     * param []
     * return java.lang.String
     * author Administrator
     * createTime 2020/04/28 14:12
     **/
    public static String getRequestUserAccount() {
        return getRequestHeader(GatewayHeaders.USER_ACCOUNT);
    }

    /**
     * 获得Http客户端的ip
     * @param req
     * @return
     */
    public static String getHttpClientIp(HttpServletRequest req){
        String ip = req.getHeader(CommonConstants.HTTP_HEADER_X_FORWARDED_FOR);
        if(ip == null || ip.length() == 0 || CommonConstants.UNKNOWN_STRING.equalsIgnoreCase(ip)) {
            ip = req.getHeader(CommonConstants.HTTP_HEADER_PROXY_CLIENT_IP);
        }
        if(ip == null || ip.length() == 0 || CommonConstants.UNKNOWN_STRING.equalsIgnoreCase(ip)) {
            ip = req.getHeader(CommonConstants.HTTP_HEADER_WL_PROXY_CLIENT_IP);
        }
        if(ip == null || ip.length() == 0 || CommonConstants.UNKNOWN_STRING.equalsIgnoreCase(ip)) {
            ip = req.getRemoteAddr();
        }
        return ip;
    }

    /**
     * description 校验参数不能为空：key存在、value存在且不为空字符串
     * param [reqJson, params]
     * return void
     * author Administrator
     * createTime 2020/04/17 17:23
     **/
    public static void checkParamsNotBlank(JSONObject reqJson, String[] params) throws EException {
        for (int i = 0; i < params.length; i++) {
            if (StringUtils.isBlank(reqJson.getString(params[i]))) {
                throw new EException(ECode.PARAM_ERROR.code(), "参数["+params[i]+"]不能为空");
            }
        }
    }

    /**
     * description 获取Post请求Body
     * param [req]
     * return java.lang.String
     * author HuangChaobin
     * createTime 2020/06/02 19:45
     **/
    public static String getPostBody(HttpServletRequest req) {
        String reqBody = null;

        String method = req.getMethod();
        if (!CommonConstants.HTTP_REQUEST_METHOD_POST.equalsIgnoreCase(method)) {
            return reqBody;
        }
        StringBuilder sb = new StringBuilder();
        InputStream inputStream = null;
        BufferedReader reader = null;
        try {
            inputStream = req.getInputStream();
            reader = new BufferedReader(new InputStreamReader(inputStream, Charset.forName(CommonConstants.DEFAULT_CHARACTER_ENCODING)));
            String line = "";
            while ((line = reader.readLine()) != null) {
                sb.append(line);
            }

            if (0 == sb.length()) {
                Map<String, String> bodyMap = new HashMap<>();
                Map<String, String[]> parameterMap = req.getParameterMap();
                for (Map.Entry<String, String[]> entry : parameterMap.entrySet()) {
                    for (String value : entry.getValue()) {
                        bodyMap.put(entry.getKey(), value);
                    }
                }
                reqBody = bodyMap.toString();
            } else {
                reqBody = sb.toString();
            }
        } catch (IOException e) {
            log.error("解析post参数时发生错误：{}", e.getMessage(), e);
        } finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                    log.error(e.getMessage(), e);
                }
            }
            if (reader != null) {
                try {
                    reader.close();
                } catch (IOException e) {
                    log.error(e.getMessage(), e);
                }
            }
        }

        return reqBody;
    }
}
