package com.esv.datacenter.iot.module.omodel.service.impl;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.esv.datacenter.iot.common.component.BaseDataComponent;
import com.esv.datacenter.iot.common.em.DbDeletedEnum;
import com.esv.datacenter.iot.common.exception.EException;
import com.esv.datacenter.iot.common.response.ECode;
import com.esv.datacenter.iot.common.util.ReqUtils;
import com.esv.datacenter.iot.module.omodel.dao.ObjectModelPropertyDao;
import com.esv.datacenter.iot.module.omodel.entity.ObjectModelPropertyEntity;
import com.esv.datacenter.iot.module.omodel.form.ModelPropertyForm;
import com.esv.datacenter.iot.module.omodel.service.ObjectModelPropertyService;
import com.esv.datacenter.iot.module.omodel.vo.ModelPropertyVO;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;


@Service("objectModelPropertyService")
public class ObjectModelPropertyServiceImpl extends ServiceImpl<ObjectModelPropertyDao, ObjectModelPropertyEntity> implements ObjectModelPropertyService {

    @Autowired
    private BaseDataComponent baseDataComponent;

    @Override
    public Integer logicDeleteModelProperty(ObjectModelPropertyEntity entity) {
        return this.getBaseMapper().logicDeleteModelProperty(entity);
    }

    @Override
    public List<ModelPropertyVO> getModelPropertyList(Long modelId) {
        List<ObjectModelPropertyEntity> entityList = this.getBaseMapper()
                .selectList(new LambdaQueryWrapper<ObjectModelPropertyEntity>()
                        .eq(ObjectModelPropertyEntity::getModelId, modelId)
                        .eq(ObjectModelPropertyEntity::getDeleted, DbDeletedEnum.NO.getCode())
                        .orderByAsc(ObjectModelPropertyEntity::getSortNum));

        List<ModelPropertyVO> voList = new ArrayList<>();
        for (ObjectModelPropertyEntity entity : entityList) {
            ModelPropertyVO vo = new ModelPropertyVO();
            BeanUtils.copyProperties(entity, vo);
            vo.setCreateTime(entity.getCreateTime().getTime());
            vo.setUpdateTime(entity.getUpdateTime().getTime());
            voList.add(vo);
        }

        return voList;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveModelProperty(Long modelId, List<ModelPropertyForm> propertyList) {
        // 删除模型属性
        ObjectModelPropertyEntity propertyEntity = new ObjectModelPropertyEntity();
        propertyEntity.setDeleted(DbDeletedEnum.YES.getCode());
        propertyEntity.setModelId(modelId);
        this.logicDeleteModelProperty(propertyEntity);

        // 新增模型属性
        List<ObjectModelPropertyEntity> modelPropertyEntityList = new ArrayList<>();
        for (ModelPropertyForm propertyForm : propertyList) {
            ObjectModelPropertyEntity entity = new ObjectModelPropertyEntity();
            BeanUtils.copyProperties(propertyForm, entity);
            entity.setModelId(modelId);
            modelPropertyEntityList.add(entity);
        }
        this.saveBatch(modelPropertyEntityList);

        // 更新属性排序值
        for (ObjectModelPropertyEntity modelPropertyEntity : modelPropertyEntityList) {
            ObjectModelPropertyEntity entity = new ObjectModelPropertyEntity();
            entity.setId(modelPropertyEntity.getId());
            entity.setSortNum(modelPropertyEntity.getId());
            this.updateById(entity);
        }
    }

    @Override
    public void checkModelProperty(List<ModelPropertyForm> propertyList) {
        Map<Integer, String> dbTableFieldMap = this.baseDataComponent.getDbTableFieldMap();
        Map<String, Boolean> sqlReservedWordsMap = this.baseDataComponent.getSqlReservedWordsMap();
        String[] notBlandParams = new String[]{"propertyCode", "propertyName", "propertyType"};
        propertyList.forEach(property -> {
            // 必填校验
            ReqUtils.checkParamsNotBlank(JSON.parseObject(property.toString()), notBlandParams);

            // propertyCode是否为保留字校验
            if (sqlReservedWordsMap.containsKey(property.getPropertyCode())) {
                throw new EException(ECode.PARAM_ERROR.code(), "属性代码["+property.getPropertyCode()+"]为保留字段");
            }

            // propertyType校验
            if (!dbTableFieldMap.containsKey(property.getPropertyType())) {
                throw new EException(ECode.PARAM_ERROR.code(), "无效的属性类型["+property.getPropertyType()+"]");
            }

            // propertyDefaultValue校验
            String propertyType = dbTableFieldMap.get(property.getPropertyType());
            String propertyDefaultValue = StringUtils.trimToNull(property.getPropertyDefaultValue());
            if (Objects.nonNull(propertyDefaultValue)) {
                switch(propertyType) {
                    case "boolean":
                        if (!Boolean.TRUE.toString().equalsIgnoreCase(propertyDefaultValue)
                                && !Boolean.FALSE.toString().equalsIgnoreCase(propertyDefaultValue)) {
                            throw new EException(ECode.PARAM_ERROR.code(), "无效的默认值["+propertyDefaultValue+"]");
                        }
                        break;
                    case "number":
                        try {
                            new BigDecimal(propertyDefaultValue);
                        } catch (Exception e) {
                            throw new EException(ECode.PARAM_ERROR.code(), "无效的默认值["+propertyDefaultValue+"]");
                        }
                        break;
                    default:
                        break;
                }
            }
        });
    }

    @Override
    public void propertyCode2Lowercase(List<ModelPropertyForm> propertyList) {
        for (int i = 0; i < propertyList.size(); i++) {
            ModelPropertyForm form = propertyList.get(i);
            if (Objects.nonNull(form.getPropertyCode())) {
                form.setPropertyCode(form.getPropertyCode().toLowerCase());
                propertyList.set(i, form);
            }
        }
    }
}