package com.esv.superhive.iot.module.devicemodel.controller;

import com.alibaba.fastjson.JSONObject;
import com.esv.superhive.iot.common.exception.EException;
import com.esv.superhive.iot.common.response.ECode;
import com.esv.superhive.iot.common.response.EResponse;
import com.esv.superhive.iot.common.validator.groups.*;
import com.esv.superhive.iot.common.vo.PageResultVO;
import com.esv.superhive.iot.module.devicemodel.form.DeviceInstanceForm;
import com.esv.superhive.iot.module.devicemodel.service.DeviceInstanceService;
import com.esv.superhive.iot.module.devicemodel.service.DeviceTypeService;
import com.esv.superhive.iot.module.devicemodel.vo.DeviceDetailVO;
import com.esv.superhive.iot.module.devicemodel.vo.DeviceInstanceVO;
import com.esv.superhive.iot.module.devicemodel.vo.DeviceTypeVO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

/**
 * @description: 设备实例Controller
 * @author: huangchaobin@esvtek.com
 * @createTime: 2020/08/07 10:35
 * @version:1.0
 */
@Slf4j
@RestController
@RequestMapping("/deviceModel/instance")
@Validated
public class DeviceInstanceController {

    private DeviceTypeService deviceTypeService;
    private DeviceInstanceService deviceInstanceService;

    public DeviceInstanceController(DeviceTypeService deviceTypeService, DeviceInstanceService deviceInstanceService) {
        this.deviceTypeService = deviceTypeService;
        this.deviceInstanceService = deviceInstanceService;
    }

    /**
     * @description 新增设备实例
     * @param form:
     * @return com.esv.superhive.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/07 10:36
     **/
    @PostMapping("/add")
    public EResponse add(@RequestBody @Validated(ValidatorInsert.class) DeviceInstanceForm form) throws EException {
        if (!deviceTypeService.isValidTypeId(form.getDeviceTypeId())) {
            throw new EException(ECode.BIZ_PARAM_ERROR.code(), "无效的设备类型ID");
        }

        Long id = deviceInstanceService.insertDeviceInstance(form);
        JSONObject data = new JSONObject();
        data.put("id", id);

        return EResponse.ok(data);
    }

    /**
     * @description 删除设备实例
     * @param form:
     * @return com.esv.superhive.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/07 10:51
     **/
    @PostMapping("/delete")
    public EResponse delete(@RequestBody @Validated(ValidatorDelete.class) DeviceInstanceForm form) throws EException {
        if (!deviceInstanceService.isValidId(form.getId())) {
            throw new EException(ECode.BIZ_PARAM_ERROR.code(), "无效的设备ID");
        }

        deviceInstanceService.deleteInstance(form.getId());

        return EResponse.ok();
    }

    /**
     * @description 编辑设备实例信息
     * @param form:
     * @return com.esv.superhive.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/07 11:03
     **/
    @PostMapping("/edit")
    public EResponse edit(@RequestBody @Validated(ValidatorUpdate.class) DeviceInstanceForm form) throws EException {
        if (!deviceInstanceService.isValidId(form.getId())) {
            throw new EException(ECode.BIZ_PARAM_ERROR.code(), "无效的设备ID");
        }

        deviceInstanceService.updateInstance(form);

        return EResponse.ok();
    }

    /**
     * @description 分页查询设备实例
     * @param form:
     * @return com.esv.datacenter.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/07 14:08
     **/
    @PostMapping("/list")
    public EResponse<PageResultVO> list(@RequestBody @Validated(ValidatorList.class) DeviceInstanceForm form) {
        PageResultVO pageResultVO = deviceInstanceService.getInstance4Page(form);
        return EResponse.ok(pageResultVO);
    }

    @PostMapping("detail")
    public EResponse<DeviceDetailVO> detail(@RequestBody @Validated(ValidatorDetail.class) DeviceInstanceForm form) {
        DeviceInstanceVO deviceInstance = deviceInstanceService.deviceInstanceDetail(form.getId());
        if (deviceInstance == null) {
            return EResponse.error(ECode.DEVICE_NOT_FOUND);
        }
        Long deviceTypeId = deviceInstance.getDeviceTypeId();
        DeviceTypeVO deviceTypeVO = deviceTypeService.deviceTypeDetail(deviceTypeId);

        DeviceDetailVO deviceDetailVO = new DeviceDetailVO();
        deviceDetailVO.setDeviceInstance(deviceInstance);
        deviceDetailVO.setDeviceType(deviceTypeVO);
        return EResponse.ok(deviceDetailVO);
    }

    /**
     * @description 通过设备类型ID获取设备列表
     * @param form:
     * @return com.esv.superhive.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/10 10:24
     **/
    @PostMapping("/getInstanceByDeviceTypeId")
    public EResponse getInstanceByDeviceTypeId(@RequestBody @Validated(ValidatorListById.class) DeviceInstanceForm form) throws EException {
        if (!deviceTypeService.isValidTypeId(form.getDeviceTypeId())) {
            throw new EException(ECode.BIZ_PARAM_ERROR.code(), "无效的设备类型ID");
        }

        return EResponse.ok(deviceInstanceService.getInstanceListByTypeId(form.getDeviceTypeId()));
    }

    /**
     * @description 获取设备接入信息
     * @param form:
     * @return com.esv.superhive.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/14 9:43
     **/
    @PostMapping("/accessInfo/get")
    public EResponse getAccessInfo(@RequestBody @Validated(ValidatorDetail.class) DeviceInstanceForm form) throws EException {
        if (!deviceInstanceService.isValidId(form.getId())) {
            throw new EException(ECode.BIZ_PARAM_ERROR.code(), "无效的设备ID");
        }

        return EResponse.ok(deviceInstanceService.getAccessInfoByInstanceId(form.getId()));
    }
}
