package com.esv.superhive.iot.module.datamodel.controller;

import com.alibaba.fastjson.JSONObject;
import com.esv.superhive.iot.common.exception.EException;
import com.esv.superhive.iot.common.response.ECode;
import com.esv.superhive.iot.common.response.EResponse;
import com.esv.superhive.iot.common.validator.groups.*;
import com.esv.superhive.iot.module.datamodel.form.DataModelForm;
import com.esv.superhive.iot.module.datamodel.form.DataModelPropertyForm;
import com.esv.superhive.iot.module.datamodel.service.DataModelPropertyService;
import com.esv.superhive.iot.module.datamodel.service.DataModelService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.Objects;

/**
 * @description: 数据模型Controller
 * @author: huangchaobin@esvtek.com
 * @createTime: 2020/07/31 14:52
 * @version:1.0
 */
@Slf4j
@RestController
@RequestMapping("/dataModel")
@Validated
public class DataModelController {

    private DataModelService dataModelService;
    private DataModelPropertyService dataModelPropertyService;

    public DataModelController(DataModelService dataModelService, DataModelPropertyService dataModelPropertyService) {
        this.dataModelService = dataModelService;
        this.dataModelPropertyService = dataModelPropertyService;
    }

    /**
     * @description 创建模型
     * @param form:
     * @return com.esv.superhive.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/07/31 14:53
     **/
    @PostMapping("/add")
    public EResponse add(@RequestBody @Validated(ValidatorInsert.class) DataModelForm form) throws EException {
        List<DataModelPropertyForm> propertyList = form.getPropertyList();
        // 将模型属性代码转换为小写
        this.dataModelPropertyService.propertyCode2Lowercase(propertyList);
        // 模型属性校验
        this.dataModelPropertyService.checkModelProperty(propertyList);

        Long id = dataModelService.createObjectModel(form);
        JSONObject data = new JSONObject();
        data.put("id", id);

        return EResponse.ok(data);
    }

    /**
     * @description 分页查询模型列表
     * @param form:
     * @return com.esv.superhive.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/07/31 16:31
     **/
    @PostMapping("/list")
    public EResponse list(@RequestBody @Validated(ValidatorList.class) DataModelForm form) throws EException {
        String modelName = StringUtils.trimToNull(form.getName());
        if (Objects.nonNull(modelName)) {
            form.setName("%" + modelName + "%");
        } else {
            form.setName(null);
        }
        return EResponse.ok(dataModelService.getModel4Page(form));
    }

    /**
     * @description 获取模型详情
     * @param form:
     * @return com.esv.superhive.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/01 10:40
     **/
    @PostMapping("/detail")
    public EResponse detail(@RequestBody @Validated(ValidatorDetail.class) DataModelForm form) throws EException {
        // 校验模型ID是否有效
        this.checkIsValidModelId(form.getId());

        return EResponse.ok(dataModelService.getModelDetail(form));
    }

    /**
     * @description 编辑模型
     * @param form:
     * @return com.esv.superhive.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/01 10:40
     **/
    @PostMapping("/edit")
    public EResponse edit(@RequestBody @Validated(ValidatorUpdate.class) DataModelForm form) throws EException {
        // 校验模型ID是否有效
        this.checkIsValidModelId(form.getId());

        List<DataModelPropertyForm> propertyList = form.getPropertyList();
        // 将模型属性代码转换为小写
        this.dataModelPropertyService.propertyCode2Lowercase(propertyList);
        // 模型属性校验
        this.dataModelPropertyService.checkModelProperty(propertyList);

        // 更新模型
        this.dataModelService.updateModel(form);

        return EResponse.ok();
    }

    /**
     * @description 删除模型
     * @param form:
     * @return com.esv.superhive.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/01 11:35
     **/
    @PostMapping("/delete")
    public EResponse delete(@RequestBody @Validated(ValidatorDelete.class) DataModelForm form) throws EException {
        // 校验模型ID是否有效
        this.checkIsValidModelId(form.getId());

        dataModelService.deleteModel(form.getId());
        return EResponse.ok();
    }

    /**
     * @description 获取所有模型列表
     * @return com.esv.superhive.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/01 13:10
     **/
    @PostMapping("/all")
    public EResponse all() throws EException {
        return EResponse.ok(dataModelService.getAllModeList());
    }

    /**
     * @description 校验模型ID是否有效
     * @param modelId:
     * @return void
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/01 14:25
     **/
    private void checkIsValidModelId(Long modelId) {
        if (!dataModelService.isModelExits(modelId)) {
            throw new EException(ECode.BIZ_PARAM_ERROR.code(), "无效的模型ID");
        }
    }

}
