package com.esv.superhive.iot.module.dashboard.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.esv.superhive.iot.common.component.TimescaleComponent;
import com.esv.superhive.iot.common.constants.CommonConstants;
import com.esv.superhive.iot.common.util.SqlUtils;
import com.esv.superhive.iot.module.dashboard.dto.DataPreviewDTO;
import com.esv.superhive.iot.module.dashboard.req.DashboardReq;
import com.esv.superhive.iot.module.dashboard.req.HistoryDashboardReq;
import com.esv.superhive.iot.module.dashboard.req.TimeDomainReq;
import com.esv.superhive.iot.module.dashboard.service.DashboardService;
import com.esv.superhive.iot.module.dashboard.vo.ChartVO;
import com.esv.superhive.iot.module.dashboard.vo.FrequencyDomainVO;
import com.esv.superhive.iot.module.dashboard.vo.FunctionVO;
import com.esv.superhive.iot.module.dashboard.vo.ModelDataVO;
import com.zaxxer.hikari.HikariDataSource;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.sql.Timestamp;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * @description:
 * @project: superhive-iot-service
 * @name: com.esv.superhive.iot.module.dashboard.service.impl.DashboardServiceImpl
 * @author: chenfm
 * @email: chenfengman@esvtek.com
 * @createTime: 2020/8/3 20:24
 * @version: 1.0
 */
@Slf4j
@Service("dashboardService")
public class DashboardServiceImpl implements DashboardService {

    private JdbcTemplate jdbcTemplate;

    @Autowired
    public DashboardServiceImpl(TimescaleComponent timescaleComponent) {
        HikariDataSource dataSource = timescaleComponent.getHikariDataSource();
        jdbcTemplate = new JdbcTemplate(dataSource);
    }

    @Override
    public List<ModelDataVO> modelData(List<DashboardReq> dashboardReqList) {
        List<ModelDataVO> modelDataVOList = new ArrayList<>();
        for (DashboardReq dashboardReq : dashboardReqList) {
            Long modelId = dashboardReq.getModelId();
            String tableName = CommonConstants.MODEL_DATA_TABLE_PREFIX + modelId;
            String sql = "select time"
                    + SqlUtils.makeSqlParams(dashboardReq.getPropertyCodeList())
                    + " from "
                    + tableName
                    + " where device_id = ?"
                    + " order by time desc limit 100";
            log.info("select modelData sql: {}", sql);
            List<Map<String, Object>> dataList;
            try {
                dataList = jdbcTemplate.queryForList(sql, dashboardReq.getDeviceId());
            } catch (Exception e) {
                log.error(e.getMessage(), e);
                dataList = new ArrayList<>();
            }
            modelDataVOList.add(new ModelDataVO(modelId, dataList));
        }
        return modelDataVOList;
    }

    @Override
    public List<ModelDataVO> modelHistory(HistoryDashboardReq historyDashboardReq) {
        Timestamp beginTime = new Timestamp(historyDashboardReq.getBeginTime());
        Timestamp endTime = new Timestamp(historyDashboardReq.getEndTime());
        List<DashboardReq> dashboardReqList = historyDashboardReq.getDashboardReqList();
        List<ModelDataVO> modelDataVOList = new ArrayList<>();
        for (DashboardReq dashboardReq : dashboardReqList) {
            Long modelId = dashboardReq.getModelId();
            String tableName = CommonConstants.MODEL_DATA_TABLE_PREFIX + modelId;
            String sql = "select time"
                    + SqlUtils.makeSqlParams(dashboardReq.getPropertyCodeList())
                    + " from "
                    + tableName
                    + " where device_id = ?"
                    + " and (time > ? and time < ?)"
                    + " order by time";
            log.info("select modelData sql: {}", sql);
            List<Map<String, Object>> dataList =
                    jdbcTemplate.queryForList(sql, dashboardReq.getDeviceId(),
                            beginTime, endTime);
            ModelDataVO modelDataVO = new ModelDataVO(modelId, dataList);
            modelDataVOList.add(modelDataVO);
        }
        return modelDataVOList;
    }

    @Override
    public Map<String, Object> timeDomainData(TimeDomainReq timeDomainReq) {
        Timestamp searchTime = new Timestamp(timeDomainReq.getTime());
        Long modelId = timeDomainReq.getModelId();
        String tableName = CommonConstants.MODEL_DATA_TABLE_PREFIX + modelId;
        String sql = "select time"
                + SqlUtils.makeSqlParams(timeDomainReq.getPropertyCodeList())
                + " from "
                + tableName
                + " where device_id = ?"
                + " and time = ?"
                + " order by time desc"
                + " limit 1";
        log.info("select modelData sql: {}", sql);
        Map<String, Object> map = null;
        try {
            map = jdbcTemplate.queryForMap(sql, timeDomainReq.getDeviceId(), searchTime);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return map;
    }

    @Override
    public FrequencyDomainVO processFrequencyDomain(List<String> resultList) {
        String functionListStr = resultList.get(0);
        String xDataStr = resultList.get(1);
        String yDataStr = resultList.get(2);

        DecimalFormat df = new DecimalFormat("#0.00");
        JSONArray array = JSON.parseArray(functionListStr);
        List<FunctionVO> functionList = new ArrayList<>(array.size());
        for (Object o : array) {
            JSONArray dataArray = (JSONArray) o;
            FunctionVO functionVO = new FunctionVO();
            functionVO.setAmplitude(df.format(dataArray.getDouble(1)));
            functionVO.setFrequency(df.format(dataArray.getDouble(0)));
            functionVO.setPhase(df.format(dataArray.getDouble(2)));
            functionList.add(functionVO);
//            if (functionList.size() >= 5) {
//                break;
//            }
        }

        ChartVO chartVO = new ChartVO();
        List<Double> xList = new ArrayList<>();
        for (String xStr : xDataStr.split(",")) {
            xList.add(Double.parseDouble(xStr));
        }
        List<Double> yList = new ArrayList<>();
        for (String yStr : yDataStr.split(",")) {
            yList.add(Double.parseDouble(yStr));
        }
        chartVO.setXData(xList);
        chartVO.setYData(yList);

        Collections.sort(functionList);
        FrequencyDomainVO frequencyDomainVO = new FrequencyDomainVO();
        frequencyDomainVO.setChart(chartVO);
        frequencyDomainVO.setFunctionList(functionList);
        return frequencyDomainVO;
    }

    @Override
    public List<DataPreviewDTO> dataPreview(long modelId, long deviceId) {
        String tableName = CommonConstants.MODEL_DATA_TABLE_PREFIX + modelId;
        String sql = "select time, sampling_rate"
                + " from "
                + tableName
                + " where device_id = ?"
                + " order by time desc";
        log.info("select modelData sql: {}", sql);
        List<DataPreviewDTO> list = new ArrayList<>();
        try {
            List<Map<String, Object>> mapList = jdbcTemplate.queryForList(sql, deviceId);
            for (Map<String, Object> map : mapList) {
                DataPreviewDTO dataPreviewDTO = new DataPreviewDTO();
                Timestamp time = (Timestamp) map.get("time");
                BigDecimal samplingRate = (BigDecimal) map.get("sampling_rate");
                dataPreviewDTO.setTime(time.getTime());
                dataPreviewDTO.setSamplingRate(samplingRate.doubleValue());
                list.add(dataPreviewDTO);
            }
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }

        return list;
    }

}
