package com.esv.superhive.iot.module.datamodel.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.esv.superhive.iot.common.component.RedisComponent;
import com.esv.superhive.iot.common.util.ReqUtils;
import com.esv.superhive.iot.module.datamodel.dao.DataModelAlarmRuleDao;
import com.esv.superhive.iot.module.datamodel.dto.DataModelAlarmRuleDto;
import com.esv.superhive.iot.module.datamodel.entity.DataModelAlarmRuleEntity;
import com.esv.superhive.iot.module.datamodel.form.AlarmRuleItemForm;
import com.esv.superhive.iot.module.datamodel.form.DataModelAlarmRuleForm;
import com.esv.superhive.iot.module.datamodel.service.DataModelAlarmRuleService;
import com.esv.superhive.iot.module.datamodel.vo.AlarmRuleItemVO;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;


@Service("dataModelAlarmRuleService")
public class DataModelAlarmRuleServiceImpl extends ServiceImpl<DataModelAlarmRuleDao, DataModelAlarmRuleEntity> implements DataModelAlarmRuleService {

    @Value("${spring.application.name}")
    private String applicationName;

    @Autowired
    private RedisComponent redisComponent;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveAlarmRule(DataModelAlarmRuleForm form) {
        // 删除模型告警规则
        Long modelId = form.getModelId();
        this.getBaseMapper().delete(new LambdaQueryWrapper<DataModelAlarmRuleEntity>().eq(DataModelAlarmRuleEntity::getModelId, modelId));

        // 新增模型告警规则
        String[] notBlandParams = new String[]{"propertyId", "ruleExpression", "alarmLevel", "threshold"};
        List<AlarmRuleItemForm> alarmRuleItemList = form.getAlarmRuleItemList();
        for (AlarmRuleItemForm ruleItemForm : alarmRuleItemList) {
            // 告警规则必填校验
            ReqUtils.checkParamsNotBlank(JSON.parseObject(ruleItemForm.toString()), notBlandParams);

            DataModelAlarmRuleEntity entity = new DataModelAlarmRuleEntity();
            BeanUtils.copyProperties(ruleItemForm, entity);
            entity.setModelId(form.getModelId());
            entity.setTenantId(ReqUtils.getRequestTenantId());
            entity.setDepartmentId(ReqUtils.getRequestDepartmentId());
            this.getBaseMapper().insert(entity);

            // 更新排序
            DataModelAlarmRuleEntity sortEntity = new DataModelAlarmRuleEntity();
            sortEntity.setId(entity.getId());
            sortEntity.setSortNum(entity.getId());
            this.getBaseMapper().updateById(sortEntity);
        }
    }

    @Override
    public List<AlarmRuleItemVO> getAlarmRuleList(Long modelId) {
        List<DataModelAlarmRuleEntity> entityList = this.getBaseMapper().selectList(new LambdaQueryWrapper<DataModelAlarmRuleEntity>()
                .eq(DataModelAlarmRuleEntity::getModelId, modelId)
                .orderByAsc(DataModelAlarmRuleEntity::getSortNum));

        List<AlarmRuleItemVO> voList = new ArrayList<>();
        for (DataModelAlarmRuleEntity entity : entityList) {
            AlarmRuleItemVO vo = new AlarmRuleItemVO();
            BeanUtils.copyProperties(entity, vo);
            voList.add(vo);
        }

        return voList;
    }

    @Override
    public void cacheInitAlarmRule() {
        List<DataModelAlarmRuleDto> dataModelAlarmRuleDtoList = this.getBaseMapper().selectAll();
        if (Objects.isNull(dataModelAlarmRuleDtoList) || 0 == dataModelAlarmRuleDtoList.size()) {
            return;
        }

        Long modelId = -99L;
        List<DataModelAlarmRuleDto> alarmRuleDtoList = null;
        Map<Long, Object> alarmRuleMap = new HashMap<>(32);
        for (DataModelAlarmRuleDto entity : dataModelAlarmRuleDtoList) {
            if (!entity.getModelId().equals(modelId)) {
                if (null != alarmRuleDtoList && 0 < alarmRuleDtoList.size()) {
                    alarmRuleMap.put(modelId, alarmRuleDtoList);
                }
                alarmRuleDtoList = new ArrayList<>();
            }
            modelId = entity.getModelId();
            DataModelAlarmRuleDto dto = new DataModelAlarmRuleDto();
            BeanUtils.copyProperties(entity, dto);
            alarmRuleDtoList.add(dto);
        }
        if (!alarmRuleMap.containsKey(modelId)) {
            alarmRuleMap.put(modelId, alarmRuleDtoList);
        }

        // 初始化缓存数据模型告警规则
        for(Map.Entry<Long, Object> entry : alarmRuleMap.entrySet()){
            this.setCacheAlarmRule(entry.getKey(), (List<DataModelAlarmRuleDto>) entry.getValue());
        }
    }

    @Override
    public void setCacheAlarmRule(Long modelId, List<DataModelAlarmRuleDto> alarmRuleItemList) {
        String cacheKey = applicationName + "::data_model::alarm_rule::" + modelId;
        if (null == alarmRuleItemList || 0 == alarmRuleItemList.size()) {
            redisComponent.del(cacheKey);
        } else {
            JSONArray jsonArray = new JSONArray();
            alarmRuleItemList.forEach( dto -> {
                jsonArray.add(JSONObject.parse(dto.toString()));
            });
            redisComponent.set(cacheKey, jsonArray.toString());
        }
    }
}