package com.esv.datacenter.iot.module.datamodel.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.esv.datacenter.iot.common.component.TimescaleComponent;
import com.esv.datacenter.iot.common.em.DbDeletedEnum;
import com.esv.datacenter.iot.common.exception.EException;
import com.esv.datacenter.iot.common.response.ECode;
import com.esv.datacenter.iot.common.vo.PageResultVO;
import com.esv.datacenter.iot.module.datamodel.dao.DataModelDao;
import com.esv.datacenter.iot.module.datamodel.entity.DataModelEntity;
import com.esv.datacenter.iot.module.datamodel.entity.DataModelPropertyEntity;
import com.esv.datacenter.iot.module.datamodel.form.DataModelForm;
import com.esv.datacenter.iot.module.datamodel.form.DataModelPropertyForm;
import com.esv.datacenter.iot.module.datamodel.service.DataModelPropertyService;
import com.esv.datacenter.iot.module.datamodel.service.DataModelService;
import com.esv.datacenter.iot.module.datamodel.vo.DataDataModelDetailVO;
import com.esv.datacenter.iot.module.datamodel.vo.DataModelVO;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;


@Service("dataModelService")
public class DataModelServiceImpl extends ServiceImpl<DataModelDao, DataModelEntity> implements DataModelService {

    @Autowired
    private TimescaleComponent timescaleComponent;
    @Autowired
    private DataModelPropertyService dataModelPropertyService;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long createObjectModel(DataModelForm form) {
        // 0.模型名称校验：不能重复
        int modelCount = this.getBaseMapper().selectCount(new LambdaQueryWrapper<DataModelEntity>()
                .eq(DataModelEntity::getDeleted, DbDeletedEnum.NO.getCode())
                .eq(DataModelEntity::getName, form.getName()));
        if (0 < modelCount) {
            throw new EException(ECode.BIZ_PARAM_ERROR.code(), "模型[" + form.getName() + "]已存在");
        }

        // 1.创建模型
        DataModelEntity modelEntity = new DataModelEntity();
        modelEntity.setName(form.getName());
        modelEntity.setDescription(form.getDescription());
        this.getBaseMapper().insert(modelEntity);
        Long modelId = modelEntity.getId();

        // 2.保存模型属性
        this.dataModelPropertyService.saveModelProperty(modelId, form.getPropertyList());

        // 3.创建时序数据库表
        form.setId(modelId);
        this.createModelTimescaleDB(modelId, form.getPropertyList());

        return modelId;
    }

    @Override
    public PageResultVO getModel4Page(DataModelForm queryObj) {
        IPage<DataModelEntity> page = new Page<>(queryObj.getPageNum(), queryObj.getPageSize());
        this.baseMapper.select4Page(page, queryObj);

        List<DataModelEntity> modelEntityList = page.getRecords();
        List<DataModelVO> dataModelVOList = new ArrayList<>();
        for (DataModelEntity entity : modelEntityList) {
            DataModelVO vo = new DataModelVO();
            BeanUtils.copyProperties(entity, vo);
            vo.setCreateTime(entity.getCreateTime().getTime());
            vo.setUpdateTime(entity.getUpdateTime().getTime());
            dataModelVOList.add(vo);
        }

        return new PageResultVO(page, dataModelVOList);
    }

    @Override
    public DataDataModelDetailVO getModelDetail(DataModelForm form) {
        DataDataModelDetailVO dataModelDetailVO = new DataDataModelDetailVO();

        // 获取模型
        Long modelId = form.getId();
        DataModelEntity modelEntity = this.getById(modelId);
        BeanUtils.copyProperties(modelEntity, dataModelDetailVO);
        dataModelDetailVO.setCreateTime(modelEntity.getCreateTime().getTime());
        dataModelDetailVO.setUpdateTime(modelEntity.getUpdateTime().getTime());

        // 获取模型属性
        dataModelDetailVO.setPropertyList(dataModelPropertyService.getModelPropertyList(modelId));

        return dataModelDetailVO;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateModel(DataModelForm form) {
        Long modelId = form.getId();

        // 模型名称校验：不能重复
        int modelCount = this.getBaseMapper().selectCount(new LambdaQueryWrapper<DataModelEntity>()
                .eq(DataModelEntity::getName, form.getName())
                .eq(DataModelEntity::getDeleted, DbDeletedEnum.NO.getCode())
                .ne(DataModelEntity::getId, modelId));
        if (0 < modelCount) {
            throw new EException(ECode.BIZ_PARAM_ERROR.code(), "模型[" + form.getName() + "]已存在");
        }

        // 更新模型
        DataModelEntity modelEntity = new DataModelEntity();
        modelEntity.setId(modelId);
        modelEntity.setName(form.getName());
        modelEntity.setDescription(form.getDescription());
        this.getBaseMapper().updateById(modelEntity);

        // 判断是否修改模型属性并校验
        List<DataModelPropertyForm> propertyList = form.getPropertyList();
        if (null != propertyList && 0 < propertyList.size()) {
            // 保存模型属性
            this.dataModelPropertyService.saveModelProperty(modelId, propertyList);

            // 删除时序数据库表
            this.timescaleComponent.dropTable(modelId);

            // 创建时序数据库表
            this.createModelTimescaleDB(modelId, propertyList);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteModel(Long modelId) {
        // 逻辑删除模型
        DataModelEntity modelEntity = new DataModelEntity();
        modelEntity.setId(modelId);
        modelEntity.setDeleted(DbDeletedEnum.YES.getCode());
        this.baseMapper.logicDeleteModel(modelEntity);

        // 删除模型属性
        DataModelPropertyEntity propertyEntity = new DataModelPropertyEntity();
        propertyEntity.setDeleted(DbDeletedEnum.YES.getCode());
        propertyEntity.setModelId(modelId);
        this.dataModelPropertyService.logicDeleteModelProperty(propertyEntity);

        // 删除时序数据库表
        this.timescaleComponent.dropTable(modelId);
    }

    @Override
    public List<DataModelVO> getAllModeList() {
        List<DataModelEntity> modelEntityList = this.getBaseMapper().selectList(new LambdaQueryWrapper<DataModelEntity>()
                .orderByAsc(DataModelEntity::getName));
        List<DataModelVO> dataModelVOList = new ArrayList<>();
        for (DataModelEntity entity : modelEntityList) {
            DataModelVO vo = new DataModelVO();
            BeanUtils.copyProperties(entity, vo);
            vo.setCreateTime(entity.getCreateTime().getTime());
            vo.setUpdateTime(entity.getUpdateTime().getTime());
            dataModelVOList.add(vo);
        }

        return dataModelVOList;
    }

    @Override
    public Boolean isModelExits(Long modelId) {
        int count = this.getBaseMapper().selectCount(new LambdaQueryWrapper<DataModelEntity>()
                .eq(DataModelEntity::getId, modelId));
        if (0 == count) {
            return false;
        } else {
            return true;
        }
    }

    @Override
    public DataModelEntity getModelInfo(Long id) {
        return this.getBaseMapper().selectById(id);
    }

    private Boolean createModelTimescaleDB(Long modelId, List<DataModelPropertyForm> propertyList) {
        List<DataModelPropertyEntity> modelPropertyEntityList = new ArrayList<>();
        for (DataModelPropertyForm propertyForm : propertyList) {
            DataModelPropertyEntity entity = new DataModelPropertyEntity();
            BeanUtils.copyProperties(propertyForm, entity);
            entity.setModelId(modelId);
            modelPropertyEntityList.add(entity);
        }

        return timescaleComponent.createTable(modelId, modelPropertyEntityList);
    }

}