package com.esv.datacenter.iot.common.component;

import com.zaxxer.hikari.HikariDataSource;
import lombok.extern.slf4j.Slf4j;
import org.springframework.jdbc.datasource.DataSourceTransactionManager;
import org.springframework.stereotype.Component;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.support.TransactionTemplate;

/**
 * @description: 根据数据源配置信息，动态获取数据源
 * @author: huangchaobin@esvtek.com
 * @createTime: 2020/07/27 13:55
 * @version:1.0
 */
@Component
@Slf4j
public class DynamicDataSource {

    /**
     * @description 获取数据源（无事务）
     * 注意：一定要执行 dataSource.close();
     * @param dataSourceConfig:
     * @return com.zaxxer.hikari.HikariDataSource
     * @author huangChaobin@esvtek.com
     * @createTime 2020/07/27 14:01
     **/
    public HikariDataSource getDynamicDataSource(DataSourceConfig dataSourceConfig) {
        HikariDataSource dataSource = new HikariDataSource();
        dataSource.setJdbcUrl(dataSourceConfig.getJdbcUrl());
        dataSource.setDriverClassName(dataSourceConfig.getDriverClassName());
        dataSource.setUsername(dataSourceConfig.getUsername());
        dataSource.setPassword(dataSourceConfig.getPassword());

        dataSource.setConnectionTestQuery(dataSourceConfig.getValidationQuery());
        dataSource.setConnectionTimeout(dataSourceConfig.getConnectionTimeout());
        dataSource.setMinimumIdle(dataSourceConfig.getMinimumIdle());
        dataSource.setMaximumPoolSize(dataSourceConfig.getMaximumPoolSize());
        dataSource.setMaxLifetime(dataSourceConfig.getMaxLifetime());

        return dataSource;
    }

    /**
     * @description 获取数据源（有事务）
     * 注意：一定要执行 dataSource.close();
     * @param dataSourceConfig:
     * @return com.zaxxer.hikari.HikariDataSource
     * @author huangChaobin@esvtek.com
     * @createTime 2020/07/27 14:02
     **/
    public HikariDataSource getDynamicDataSource4Transaction(DataSourceConfig dataSourceConfig) {
        HikariDataSource dataSource = this.getDynamicDataSource(dataSourceConfig);

        // 事务的使用
        PlatformTransactionManager txManager = new DataSourceTransactionManager(dataSource);
        TransactionTemplate transactionTemplate = new TransactionTemplate(txManager);
        // 回滚的方法
        transactionTemplate.execute(transactionStatus -> {
            transactionStatus.setRollbackOnly();
            return null;
        });

        return dataSource;
    }

}
