package com.esv.datacenter.iot.common.component;

import com.esv.datacenter.iot.common.util.DateUtils;
import com.esv.datacenter.iot.module.datamodel.entity.DataModelPropertyEntity;
import com.zaxxer.hikari.HikariDataSource;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * @description:
 * @author: huangchaobin@esvtek.com
 * @createTime: 2020/08/03 14:19
 * @version:1.0
 */
@Component
@RefreshScope
@Slf4j
public class TimescaleComponent {

    @Value("${timescale.data-source.jdbc-url}")
    private String jdbcUrl;

    @Value("${timescale.data-source.driver-class-name}")
    private String driverClassName;

    @Value("${timescale.data-source.validation-query}")
    private String validationQuery;

    @Value("${timescale.data-source.username}")
    private String username;

    @Value("${timescale.data-source.password}")
    private String password;

    @Value("${timescale.data-source.connection-timeout}")
    private Long connectionTimeout;

    @Value("${timescale.data-source.minimum-idle}")
    private Integer minimumIdle;

    @Value("${timescale.data-source.maximum-pool-size}")
    private Integer maximumPoolSize;

    @Value("${timescale.data-source.max-lifetime}")
    private Long maxLifetime;

    @Value("${timescale.table-field.map}")
    private String tableFieldMap;

    @Value("${timescale.table-prefix}")
    private String tablePrefix;

    @Autowired
    private DynamicDataSource dynamicDataSource;
    @Autowired
    private BaseDataComponent baseDataComponent;

    /**
     * @description 创建表
     * @param modelId:
     * @param propertyEntityList:
     * @return java.lang.Boolean
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/03 14:22
     **/
    public Boolean createTable(Long modelId, List<DataModelPropertyEntity> propertyEntityList) {
        HikariDataSource dataSource = this.getHikariDataSource4Transaction();

        // 表名
        String table = tablePrefix + modelId;

        try {
            JdbcTemplate jdbcTemplate = new JdbcTemplate(dataSource);
            // 校验表名是否已存在
            if (checkTableExits(jdbcTemplate, table)) {
                log.warn("创建表失败，表[{}]已存在", table);
                return false;
            }

            // 获取建表SQL及超表SQL
            String tableSql = this.generateCreateTableSql(table, propertyEntityList);
            log.info("建表SQL：{}", tableSql);
            String indexSql = this.generateTableIndexSql(table);
            log.info("表索引SQL：{}", indexSql);
            String hyperSql = this.generateHyperTableSql(table);
            log.info("超表SQL：{}", hyperSql);

            // 使用jdbcTemplate来执行sql
            jdbcTemplate.execute(tableSql);
            jdbcTemplate.execute(indexSql);
            jdbcTemplate.execute(hyperSql);
            log.info("建表[{}]成功", table);
        } catch (Exception e) {
            log.error("建表[{}]失败", table);
            log.error(e.getMessage(), e);
            throw e;
        } finally {
            // 关闭数据源
            if (Objects.nonNull(dataSource) && !dataSource.isClosed()) {
                dataSource.close();
            }
        }

        return true;
    }

    /**
     * @description
     * @param modelId:
     * @return java.lang.Boolean
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/03 17:10
     **/
    public Boolean dropTable(Long modelId) {
        HikariDataSource dataSource = this.getHikariDataSource();
        String table = tablePrefix + modelId;

        try {
            // 校验表名是否已存在
            JdbcTemplate jdbcTemplate = new JdbcTemplate(dataSource);
            if (!checkTableExits(jdbcTemplate, table)) {
                log.warn("删除表失败，表[{}]不存在", table);
                return false;
            }

            // 获取删除表SQL
            String deleteTableSql = this.generateDeleteTableSql(table);
            log.info("删除表SQL：{}", deleteTableSql);

            // 使用jdbcTemplate来执行sql
            jdbcTemplate.execute(deleteTableSql);
            log.info("删除表[{}]成功", table);
        } catch (Exception e) {
            log.error("删除表[{}]失败", table);
            log.error(e.getMessage(), e);
            throw e;
        } finally {
            // 关闭数据源
            if (Objects.nonNull(dataSource) && !dataSource.isClosed()) {
                dataSource.close();
            }
        }

        return true;
    }

    private String generateCreateTableSql(String table, List<DataModelPropertyEntity> propertyEntityList) {
        // 获取字典表数据
        Map<Integer, String> dbTableFieldMap = this.baseDataComponent.getDbTableFieldMap();

        // 获取表字段映射
        Map<String, String> fieldMap = this.getTableFieldMap();

        // 生成SQL
        StringBuffer sb = new StringBuffer();
        sb.append("create table if not exists ").append(table).append("(")
                .append("time TIMESTAMPTZ NOT NULL,")
                .append("device_id int8 NULL");
        String propertyType;
        String defaultValue;
        for (DataModelPropertyEntity propertyEntity : propertyEntityList) {
            propertyType = dbTableFieldMap.get(propertyEntity.getPropertyType()).toLowerCase();
            defaultValue = StringUtils.trimToNull(propertyEntity.getPropertyDefaultValue());
            sb.append(",");
            sb.append(propertyEntity.getPropertyCode()).append(" ");
            sb.append(fieldMap.get(propertyType)).append(" NULL");
            if (Objects.nonNull(defaultValue) && !"array".equals(propertyType)) {
                sb.append(" DEFAULT ");
                switch (propertyType) {
                    case "boolean":
                        if ("true".equalsIgnoreCase(defaultValue)) {
                            sb.append("'1'::\"bit\"");
                        } else {
                            sb.append("'0'::\"bit\"");
                        }
                        break;
                    case "number":
                        sb.append(defaultValue);
                        break;
                    case "integer":
                        sb.append(defaultValue);
                        break;
                    case "date":
                        sb.append("'").append(defaultValue).append("'::date");
                        break;
                    case "time":
                        sb.append(DateUtils.getMilliSecondsOfDay(defaultValue));
                        break;
                    case "datetime":
                        sb.append("'").append(defaultValue).append("'");
                        break;
                    default:
                        sb.append("'").append(defaultValue).append("'::text");
                        break;
                }
            }
        }
        sb.append(");");

        return sb.toString();
    }

    private String generateTableIndexSql(String table) {
        StringBuffer sb = new StringBuffer();
        sb.append("CREATE INDEX ")
                .append(table)
                .append("_device_id_idx ON ")
                .append(table)
                .append(" (device_id);");
        return sb.toString();
    }

    private String generateHyperTableSql(String table) {
        StringBuffer sb = new StringBuffer();
        sb.append("SELECT create_hypertable('")
                .append(table).append("', 'time');");
        return sb.toString();
    }

    private Boolean checkTableExits(JdbcTemplate jdbcTemplate, String table) {
        StringBuffer sb = new StringBuffer();
        sb.append("select t1.tablename")
                .append(" from pg_tables t1, pg_class t2")
                .append(" where t1.tablename = '")
                .append(table)
                .append("' and t1.tablename = t2.relname")
                .append(" order by t1.tablename desc");
        String sql = sb.toString();

        // 使用jdbcTemplate来执行sql
        List<Map<String, Object>> resultMapList = jdbcTemplate.queryForList(sql);
        if (null != resultMapList && 0 < resultMapList.size()) {
            return true;
        }

        return false;
    }

    private String generateDeleteTableSql(String table) {
        StringBuffer sb = new StringBuffer();
        sb.append("drop table if exists ").append(table).append(";");
        return sb.toString();
    }

    public HikariDataSource getHikariDataSource() {
        HikariDataSource dataSource = this.dynamicDataSource.getDynamicDataSource(initDataSourceConfig());
        return dataSource;
    }

    private HikariDataSource getHikariDataSource4Transaction() {
        HikariDataSource dataSource = this.dynamicDataSource.getDynamicDataSource4Transaction(initDataSourceConfig());
        return dataSource;
    }

    private DataSourceConfig initDataSourceConfig() {
        DataSourceConfig dataSourceConfig = new DataSourceConfig(jdbcUrl, driverClassName, username, password,
                validationQuery, connectionTimeout, minimumIdle, maximumPoolSize, maxLifetime);
        return dataSourceConfig;
    }

    private Map<String, String> getTableFieldMap() {
        Map<String, String> fieldMap = new HashMap<>(32);
        String[] fields = tableFieldMap.split(",");
        for (String field : fields) {
            fieldMap.put(field.split("-")[0], field.split("-")[1]);
        }
        return fieldMap;
    }

}
