package com.esv.datacenter.iot.module.omodel.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.esv.datacenter.iot.common.component.TimescaleComponent;
import com.esv.datacenter.iot.common.em.DbDeletedEnum;
import com.esv.datacenter.iot.common.exception.EException;
import com.esv.datacenter.iot.common.response.ECode;
import com.esv.datacenter.iot.common.vo.PageResultVO;
import com.esv.datacenter.iot.module.omodel.dao.ObjectModelDao;
import com.esv.datacenter.iot.module.omodel.dto.ModelAndInstanceDto;
import com.esv.datacenter.iot.module.omodel.entity.ObjectModelEntity;
import com.esv.datacenter.iot.module.omodel.entity.ObjectModelInstanceEntity;
import com.esv.datacenter.iot.module.omodel.entity.ObjectModelPropertyEntity;
import com.esv.datacenter.iot.module.omodel.form.ModelPropertyForm;
import com.esv.datacenter.iot.module.omodel.form.ObjectModelForm;
import com.esv.datacenter.iot.module.omodel.service.ObjectModelInstanceService;
import com.esv.datacenter.iot.module.omodel.service.ObjectModelPropertyService;
import com.esv.datacenter.iot.module.omodel.service.ObjectModelService;
import com.esv.datacenter.iot.module.omodel.vo.ModelDetailVO;
import com.esv.datacenter.iot.module.omodel.vo.ModelVO;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;


@Service("objectModelService")
public class ObjectModelServiceImpl extends ServiceImpl<ObjectModelDao, ObjectModelEntity> implements ObjectModelService {

    @Autowired
    private TimescaleComponent timescaleComponent;
    @Autowired
    private ObjectModelPropertyService objectModelPropertyService;
    @Autowired
    private ObjectModelInstanceService objectModelInstanceService;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long createObjectModel(ObjectModelForm form) {
        // 0.模型名称校验：不能重复
        int modelCount = this.getBaseMapper().selectCount(new LambdaQueryWrapper<ObjectModelEntity>()
                .eq(ObjectModelEntity::getDeleted, DbDeletedEnum.NO.getCode())
                .eq(ObjectModelEntity::getName, form.getName()));
        if (0 < modelCount) {
            throw new EException(ECode.BIZ_PARAM_ERROR.code(), "模型[" + form.getName() + "]已存在");
        }

        // 1.创建模型
        ObjectModelEntity modelEntity = new ObjectModelEntity();
        modelEntity.setName(form.getName());
        modelEntity.setDescription(form.getDescription());
        this.getBaseMapper().insert(modelEntity);
        Long modelId = modelEntity.getId();

        // 2.保存模型属性
        this.objectModelPropertyService.saveModelProperty(modelId, form.getPropertyList());

        // 3.创建时序数据库表
        this.createModelTimescaleDB(modelEntity, form);

        return modelId;
    }

    @Override
    public PageResultVO getModel4Page(ObjectModelForm queryObj) {
        IPage<ObjectModelEntity> page = new Page<>(queryObj.getPageNum(), queryObj.getPageSize());
        this.baseMapper.select4Page(page, queryObj);

        List<ObjectModelEntity> modelEntityList = page.getRecords();
        List<ModelVO> modelVOList = new ArrayList<>();
        for (ObjectModelEntity entity : modelEntityList) {
            ModelVO vo = new ModelVO();
            BeanUtils.copyProperties(entity, vo);
            vo.setCreateTime(entity.getCreateTime().getTime());
            vo.setUpdateTime(entity.getUpdateTime().getTime());
            modelVOList.add(vo);
        }

        return new PageResultVO(page, modelVOList);
    }

    @Override
    public ModelDetailVO getModelDetail(ObjectModelForm form) {
        ModelDetailVO modelDetailVO = new ModelDetailVO();

        // 获取模型
        Long modelId = form.getId();
        ObjectModelEntity modelEntity = this.getById(modelId);
        BeanUtils.copyProperties(modelEntity, modelDetailVO);
        modelDetailVO.setCreateTime(modelEntity.getCreateTime().getTime());
        modelDetailVO.setUpdateTime(modelEntity.getUpdateTime().getTime());

        // 设置模型是否有实例
        int instanceCount = this.modelInstanceCount(modelId);
        modelDetailVO.setInstanceCount(instanceCount);
        if (0 == instanceCount) {
            modelDetailVO.setHasInstance(false);
        } else {
            modelDetailVO.setHasInstance(true);
        }

        // 获取模型属性
        modelDetailVO.setPropertyList(objectModelPropertyService.getModelPropertyList(modelId));

        return modelDetailVO;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateModel(ObjectModelForm form) {
        Long modelId = form.getId();

        // 模型名称校验：不能重复
        int modelCount = this.getBaseMapper().selectCount(new LambdaQueryWrapper<ObjectModelEntity>()
                .eq(ObjectModelEntity::getName, form.getName())
                .eq(ObjectModelEntity::getDeleted, DbDeletedEnum.NO.getCode())
                .ne(ObjectModelEntity::getId, modelId));
        if (0 < modelCount) {
            throw new EException(ECode.BIZ_PARAM_ERROR.code(), "模型[" + form.getName() + "]已存在");
        }

        // 更新模型
        ObjectModelEntity modelEntity = new ObjectModelEntity();
        modelEntity.setId(modelId);
        modelEntity.setName(form.getName());
        modelEntity.setDescription(form.getDescription());
        this.getBaseMapper().updateById(modelEntity);

        // 判断是否修改模型属性并校验
        List<ModelPropertyForm> propertyList = form.getPropertyList();
        if (null != propertyList && 0 < propertyList.size()) {
            if (0 < this.modelInstanceCount(modelId)) {
                throw new EException(ECode.BIZ_PARAM_ERROR.code(), "该模型已有实例，属性不能编辑");
            }

            // 保存模型属性
            this.objectModelPropertyService.saveModelProperty(modelId, propertyList);

            // 删除时序数据库表
            this.timescaleComponent.dropTable(modelId);

            // 创建时序数据库表
            this.createModelTimescaleDB(modelEntity, form);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteModel(Long modelId) {
        // 获取模型实例数
        int modelInstanceCount = this.modelInstanceCount(modelId);
        if (0 < modelInstanceCount) {
            throw new EException(ECode.BIZ_PARAM_ERROR.code(), "该模型已有" + modelInstanceCount + "个实例，不能删除");
        }

        // 逻辑删除模型
        ObjectModelEntity modelEntity = new ObjectModelEntity();
        modelEntity.setId(modelId);
        modelEntity.setDeleted(DbDeletedEnum.YES.getCode());
        this.baseMapper.logicDeleteModel(modelEntity);

        // 删除模型属性
        ObjectModelPropertyEntity propertyEntity = new ObjectModelPropertyEntity();
        propertyEntity.setDeleted(DbDeletedEnum.YES.getCode());
        propertyEntity.setModelId(modelId);
        this.objectModelPropertyService.logicDeleteModelProperty(propertyEntity);

        // 删除时序数据库表
        this.timescaleComponent.dropTable(modelId);
    }

    @Override
    public List<ModelVO> getAllModeList() {
        List<ObjectModelEntity> modelEntityList = this.getBaseMapper().selectList(new LambdaQueryWrapper<ObjectModelEntity>()
                .orderByAsc(ObjectModelEntity::getName));
        List<ModelVO> modelVOList = new ArrayList<>();
        for (ObjectModelEntity entity : modelEntityList) {
            ModelVO vo = new ModelVO();
            BeanUtils.copyProperties(entity, vo);
            vo.setCreateTime(entity.getCreateTime().getTime());
            vo.setUpdateTime(entity.getUpdateTime().getTime());
            modelVOList.add(vo);
        }

        return modelVOList;
    }

    @Override
    public Integer modelInstanceCount(Long modelId) {
        return this.objectModelInstanceService.getBaseMapper()
                .selectCount(new LambdaQueryWrapper<ObjectModelInstanceEntity>()
                        .eq(ObjectModelInstanceEntity::getDeleted, DbDeletedEnum.NO.getCode())
                        .eq(ObjectModelInstanceEntity::getModelId, modelId));
    }

    @Override
    public Boolean isModelExits(Long modelId) {
        int count = this.getBaseMapper().selectCount(new LambdaQueryWrapper<ObjectModelEntity>()
                .eq(ObjectModelEntity::getId, modelId));
        if (0 == count) {
            return false;
        } else {
            return true;
        }
    }

    @Override
    public ObjectModelEntity getModelInfo(Long id) {
        return this.getBaseMapper().selectById(id);
    }

    private Boolean createModelTimescaleDB(ObjectModelEntity modelEntity, ObjectModelForm form) {
        ModelAndInstanceDto modelAndInstanceDto = new ModelAndInstanceDto();
        modelAndInstanceDto.setObjectModelEntity(modelEntity);

        List<ObjectModelPropertyEntity> modelPropertyEntityList = new ArrayList<>();
        for (ModelPropertyForm propertyForm : form.getPropertyList()) {
            ObjectModelPropertyEntity entity = new ObjectModelPropertyEntity();
            BeanUtils.copyProperties(propertyForm, entity);
            entity.setModelId(modelEntity.getId());
            modelPropertyEntityList.add(entity);
        }
        modelAndInstanceDto.setObjectModelPropertyEntityList(modelPropertyEntityList);

        return timescaleComponent.createTable(modelAndInstanceDto);
    }

}