package com.esv.datacenter.iot.module.omodel.controller;

import com.alibaba.fastjson.JSONObject;
import com.esv.datacenter.iot.common.exception.EException;
import com.esv.datacenter.iot.common.response.ECode;
import com.esv.datacenter.iot.common.response.EResponse;
import com.esv.datacenter.iot.common.validator.groups.ValidatorDelete;
import com.esv.datacenter.iot.common.validator.groups.ValidatorDetail;
import com.esv.datacenter.iot.common.validator.groups.ValidatorInsert;
import com.esv.datacenter.iot.module.omodel.form.ModelInstanceForm;
import com.esv.datacenter.iot.module.omodel.service.ObjectModelInstanceService;
import com.esv.datacenter.iot.module.omodel.service.ObjectModelService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

/**
 * @description: 物模型实例Controller
 * @author: huangchaobin@esvtek.com
 * @createTime: 2020/08/03 10:14
 * @version:1.0
 */
@Slf4j
@RestController
@RequestMapping("/model/instance")
@Validated
public class ObjectModelInstanceController {

    private ObjectModelService objectModelService;
    private ObjectModelInstanceService objectModelInstanceService;

    public ObjectModelInstanceController(ObjectModelService objectModelService, ObjectModelInstanceService objectModelInstanceService) {
        this.objectModelService = objectModelService;
        this.objectModelInstanceService = objectModelInstanceService;
    }

    /**
     * @description 创建模型实例
     * @param form:
     * @return com.esv.datacenter.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/03 10:31
     **/
    @PostMapping("/add")
    public EResponse add(@RequestBody @Validated(ValidatorInsert.class) ModelInstanceForm form) throws EException {
        // 校验模型ID是否有效
        this.checkIsValidModelId(form.getModelId());

        Long id = objectModelInstanceService.createModelInstance(form);
        JSONObject data = new JSONObject();
        data.put("id", id);

        return EResponse.ok(data);
    }

    /**
     * @description 更新模型实例
     * @param form:
     * @return com.esv.datacenter.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/03 11:20
     **/
    @PostMapping("/edit")
    public EResponse edit(@RequestBody @Validated(ValidatorInsert.class) ModelInstanceForm form) throws EException {
        // 校验模型ID是否有效
        this.checkIsValidModelId(form.getModelId());
        // 校验模型实例ID是否有效
        this.checkIsValidModelInstanceId(form.getId());

        // 更新模型实例
        objectModelInstanceService.updateModelInstance(form);

        return EResponse.ok();
    }

    /**
     * @description 删除模型实例
     * @param form:
     * @return com.esv.datacenter.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/03 11:26
     **/
    @PostMapping("/delete")
    public EResponse delete(@RequestBody @Validated(ValidatorDelete.class) ModelInstanceForm form) throws EException {
        // 校验模型实例ID是否有效
        this.checkIsValidModelInstanceId(form.getId());

        // 更新模型实例
        objectModelInstanceService.deleteModelInstance(form.getId());

        return EResponse.ok();
    }

    /**
     * @description 获取模型实例详情
     * @param form:
     * @return com.esv.datacenter.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/03 13:45
     **/
    @PostMapping("/detail")
    public EResponse detail(@RequestBody @Validated(ValidatorDetail.class) ModelInstanceForm form) throws EException {
        objectModelInstanceService.getModelInstanceDetail(form.getId());

        return EResponse.ok();
    }

    /**
     * @description 校验模型ID是否有效
     * @param modelId:
     * @return void
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/01 14:25
     **/
    private void checkIsValidModelId(Long modelId) {
        if (!objectModelService.isModelExits(modelId)) {
            throw new EException(ECode.BIZ_PARAM_ERROR.code(), "无效的模型id值");
        }
    }

    /**
     * @description 校验是否有效模型实例ID
     * @param id:
     * @return void
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/03 11:18
     **/
    private void checkIsValidModelInstanceId(Long id) {
        if (!objectModelInstanceService.isModelInstanceExits(id)) {
            throw new EException(ECode.BIZ_PARAM_ERROR.code(), "无效的模型实例id值");
        }
    }
}
