package com.esv.datacenter.iot.module.devicemodel.controller;

import com.alibaba.fastjson.JSONObject;
import com.esv.datacenter.iot.common.exception.EException;
import com.esv.datacenter.iot.common.response.ECode;
import com.esv.datacenter.iot.common.response.EResponse;
import com.esv.datacenter.iot.common.validator.groups.*;
import com.esv.datacenter.iot.module.devicemodel.form.DeviceDataMapForm;
import com.esv.datacenter.iot.module.devicemodel.form.DeviceTypeForm;
import com.esv.datacenter.iot.module.devicemodel.service.DeviceDataMapService;
import com.esv.datacenter.iot.module.devicemodel.service.DeviceInstanceService;
import com.esv.datacenter.iot.module.devicemodel.service.DeviceTypeService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.Objects;

/**
 * @description: 设备类型Controller
 * @author: huangchaobin@esvtek.com
 * @createTime: 2020/08/06 20:14
 * @version:1.0
 */
@Slf4j
@RestController
@RequestMapping("/deviceModel")
@Validated
public class DeviceTypeController {

    private DeviceTypeService deviceTypeService;
    private DeviceInstanceService deviceInstanceService;
    private DeviceDataMapService deviceDataMapService;

    public DeviceTypeController(DeviceTypeService deviceTypeService, DeviceInstanceService deviceInstanceService,
                                DeviceDataMapService deviceDataMapService) {
        this.deviceTypeService = deviceTypeService;
        this.deviceInstanceService = deviceInstanceService;
        this.deviceDataMapService = deviceDataMapService;
    }

    /**
     * @description 新增设备类型
     * @param form:
     * @return com.esv.datacenter.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/06 20:16
     **/
    @PostMapping("/add")
    public EResponse add(@RequestBody @Validated(ValidatorInsert.class) DeviceTypeForm form) throws EException {
        Long id = deviceTypeService.insertDeviceType(form);
        JSONObject data = new JSONObject();
        data.put("id", id);

        return EResponse.ok(data);
    }

    /**
     * @description 更新设备类型信息
     * @param form:
     * @return com.esv.datacenter.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/06 20:20
     **/
    @PostMapping("/edit")
    public EResponse edit(@RequestBody @Validated(ValidatorUpdate.class) DeviceTypeForm form) throws EException {
        if (!deviceTypeService.isValidTypeId(form.getId())) {
            throw new EException(ECode.BIZ_PARAM_ERROR.code(), "无效的设备类型ID");
        }

        deviceTypeService.updateDeviceType(form);

        return EResponse.ok();
    }

    /**
     * @description 删除设备类型
     * @param form:
     * @return com.esv.datacenter.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/06 20:35
     **/
    @PostMapping("/delete")
    public EResponse delete(@RequestBody @Validated(ValidatorDelete.class) DeviceTypeForm form) throws EException {
        if (!deviceTypeService.isValidTypeId(form.getId())) {
            throw new EException(ECode.BIZ_PARAM_ERROR.code(), "无效的设备类型ID");
        }
        int deviceInstanceCount = deviceInstanceService.getInstanceCountByTypeId(form.getId());
        if (0 < deviceInstanceCount) {
            throw new EException(ECode.BIZ_PARAM_ERROR.code(), "该设备类型有设备实例，不能删除");
        }

        deviceTypeService.deleteDeviceType(form.getId());

        return EResponse.ok();
    }

    /**
     * @description 分页查询设备类型
     * @param form:
     * @return com.esv.datacenter.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/07 9:45
     **/
    @PostMapping("/list")
    public EResponse list(@RequestBody @Validated(ValidatorList.class) DeviceTypeForm form) throws EException {
        String name = StringUtils.trimToNull(form.getName());
        if (Objects.isNull(name)) {
            form.setName(null);
        } else {
            form.setName("%" + name + "%");
        }

        return EResponse.ok(deviceTypeService.getType4Page(form));
    }

    /**
     * @description 获取设备类型列表
     * @return com.esv.datacenter.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/07 9:57
     **/
    @PostMapping("/all")
    public EResponse all() throws EException {
        return EResponse.ok(deviceTypeService.getTypeList());
    }

    /**
     * @description 通过设备类型ID获取对应的数据模型列表
     * @param form:
     * @return com.esv.datacenter.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/07 14:39
     **/
    @PostMapping("/getDataModelByDeviceTypeId")
    public EResponse getDataModelByDeviceTypeId(@RequestBody @Validated(ValidatorDetail.class) DeviceTypeForm form) throws EException {
        if (!deviceTypeService.isValidTypeId(form.getId())) {
            throw new EException(ECode.BIZ_PARAM_ERROR.code(), "无效的设备类型ID");
        }

        return EResponse.ok(deviceTypeService.getDataModelByDeviceTypeId(form.getId()));
    }

    /**
     * @description 保存设备类型与数据模型的映射记录
     * @param form:
     * @return com.esv.datacenter.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/07 14:46
     **/
    @PostMapping("/saveDataModelByDeviceTypeId")
    public EResponse saveDataModelByDeviceTypeId(@RequestBody @Validated(ValidatorInsert.class) DeviceDataMapForm form) throws EException {
        if (!deviceTypeService.isValidTypeId(form.getDeviceTypeId())) {
            throw new EException(ECode.BIZ_PARAM_ERROR.code(), "无效的设备类型ID");
        }

        deviceDataMapService.saveDeviceDataMap(form.getDeviceTypeId(), form.getDataModelId());

        return EResponse.ok();
    }
}
