package com.esv.datacenter.iot.module.omodel.controller;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.esv.datacenter.iot.common.exception.EException;
import com.esv.datacenter.iot.common.response.ECode;
import com.esv.datacenter.iot.common.response.EResponse;
import com.esv.datacenter.iot.common.util.ReqUtils;
import com.esv.datacenter.iot.common.validator.groups.*;
import com.esv.datacenter.iot.module.omodel.form.ModelPropertyForm;
import com.esv.datacenter.iot.module.omodel.form.ObjectModelForm;
import com.esv.datacenter.iot.module.omodel.service.ObjectModelInstanceService;
import com.esv.datacenter.iot.module.omodel.service.ObjectModelService;
import com.esv.gateway.common.DataPermsType;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.Objects;

/**
 * @description: 物模型Controller
 * @author: huangchaobin@esvtek.com
 * @createTime: 2020/07/31 14:52
 * @version:1.0
 */
@Slf4j
@RestController
@RequestMapping("/model")
@Validated
public class ObjectModelController {

    private ObjectModelService objectModelService;
    private ObjectModelInstanceService objectModelInstanceService;

    public ObjectModelController(ObjectModelService objectModelService, ObjectModelInstanceService objectModelInstanceService) {
        this.objectModelService = objectModelService;
        this.objectModelInstanceService = objectModelInstanceService;
    }

    /**
     * @description 创建模型
     * @param form:
     * @return com.esv.datacenter.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/07/31 14:53
     **/
    @PostMapping("/add")
    public EResponse add(@RequestBody @Validated(ValidatorInsert.class) ObjectModelForm form) throws EException {
        // 参数校验：模型属性
        String[] notBlandParams = new String[]{"propertyCode", "propertyName", "propertyType"};
        List<ModelPropertyForm> propertyList = form.getPropertyList();
        propertyList.forEach(property -> {
            ReqUtils.checkParamsNotBlank(JSON.parseObject(property.toString()), notBlandParams);
        });

        Long id = objectModelService.createObjectModel(form);
        JSONObject data = new JSONObject();
        data.put("id", id);

        return EResponse.ok(data);
    }

    /**
     * @description 分页查询模型列表
     * @param form:
     * @return com.esv.datacenter.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/07/31 16:31
     **/
    @PostMapping("/list")
    public EResponse list(@RequestBody @Validated(ValidatorList.class) ObjectModelForm form) throws EException {
        if (!DataPermsType.DATA_PERM_ALL.equals(ReqUtils.getRequestDataPerm())) {
            form.setTenantId(ReqUtils.getRequestTenantId());
        }
        if (Objects.nonNull(StringUtils.trimToNull(form.getName()))) {
            form.setName("%" + form.getName() + "%");
        }
        return EResponse.ok(objectModelService.getModel4Page(form));
    }

    /**
     * @description 获取模型详情
     * @param form:
     * @return com.esv.datacenter.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/01 10:40
     **/
    @PostMapping("/detail")
    public EResponse detail(@RequestBody @Validated(ValidatorDetail.class) ObjectModelForm form) throws EException {
        // 校验模型ID是否有效
        this.checkIsValidModelId(form.getId());

        return EResponse.ok(objectModelService.getModelDetail(form));
    }

    /**
     * @description 编辑模型
     * @param form:
     * @return com.esv.datacenter.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/01 10:40
     **/
    @PostMapping("/edit")
    public EResponse edit(@RequestBody @Validated(ValidatorUpdate.class) ObjectModelForm form) throws EException {
        // 校验模型ID是否有效
        this.checkIsValidModelId(form.getId());

        return EResponse.ok(objectModelService.updateModel(form));
    }

    /**
     * @description 删除模型
     * @param form:
     * @return com.esv.datacenter.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/01 11:35
     **/
    @PostMapping("/delete")
    public EResponse delete(@RequestBody @Validated(ValidatorDelete.class) ObjectModelForm form) throws EException {
        // 校验模型ID是否有效
        this.checkIsValidModelId(form.getId());

        objectModelService.deleteModel(form.getId());
        return EResponse.ok();
    }

    /**
     * @description 获取所有模型列表
     * @return com.esv.datacenter.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/01 13:10
     **/
    @PostMapping("/all")
    public EResponse all() throws EException {
        return EResponse.ok(objectModelService.getAllModeList());
    }

    /**
     * @description 获取指定模型的实例列表
     * @param form:
     * @return com.esv.datacenter.iot.common.response.EResponse
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/01 13:43
     **/
    @PostMapping("/getModelInstanceList")
    public EResponse getModelInstanceList(@RequestBody @Validated(ValidatorDetail.class) ObjectModelForm form) throws EException {
        // 校验模型ID是否有效
        this.checkIsValidModelId(form.getId());

        return EResponse.ok(objectModelInstanceService.getModelInstanceList(form.getId()));
    }

    /**
     * @description 校验模型ID是否有效
     * @param modelId:
     * @return void
     * @author huangChaobin@esvtek.com
     * @createTime 2020/08/01 14:25
     **/
    private void checkIsValidModelId(Long modelId) {
        if (!objectModelService.isModelExits(modelId)) {
            throw new EException(ECode.BIZ_PARAM_ERROR.code(), "无效的模型id值");
        }
    }

}
