package com.esv.superhive.gateway.filter;

import com.esv.superhive.gateway.common.response.EResponse;
import com.esv.superhive.gateway.feign.AuthFeignClient;
import com.esv.superhive.gateway.feign.WebReqAuthReq;
import com.esv.superhive.gateway.feign.WebReqAuthRes;
import com.esv.superhive.gateway.util.LogUtils;
import com.esv.superhive.gateway.util.RequestUriUtils;
import com.esv.superhive.gateway.util.ResponseUtils;
import com.esv.gateway.common.GatewayHeaders;
import com.netflix.zuul.ZuulFilter;
import com.netflix.zuul.context.RequestContext;
import com.netflix.zuul.exception.ZuulException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Component;

import javax.servlet.http.HttpServletRequest;
import java.util.Objects;

/**
 * @description:
 * @author: 黄朝斌
 * @email: huangchaobin@esvtek.com
 * @createTime: 2020/06/30 9:59
 * @version:1.0
 */
@Slf4j
@RefreshScope
@Component
public class AuthRoutingFilter extends ZuulFilter {

    @Value("${zuul.prefix}")
    private String prefix;

    @Value("${management.endpoints.web.base-path}")
    private String actuatorBasePath;

    private static final String AUTH_HEADER_KEY = "Union-Authorization";

    private static final String MENU_CODE_HEADER_KEY = "Union-Menu-Code";

    @Autowired
    private AuthFeignClient authFeignClient;

    @Override
    public String filterType() {
        return "route";
    }

    @Override
    public int filterOrder() {
        return 0;
    }

    @Override
    public boolean shouldFilter() {
        return true;
    }

    @Override
    public Object run() throws ZuulException {
        RequestContext context = RequestContext.getCurrentContext();
        HttpServletRequest request = context.getRequest();

        String uri = request.getRequestURI();
        uri = uri.replaceFirst(prefix, "");
        // 端点监控uri直接放行
        if (uri.contains(actuatorBasePath)) {
            return null;
        }
        // 手机APP uri直接放行
        if (uri.startsWith("/app")) {
            return null;
        }

        String token = StringUtils.trimToEmpty(request.getHeader(AUTH_HEADER_KEY));
        String menuCode = StringUtils.trimToEmpty(request.getHeader(MENU_CODE_HEADER_KEY));

        // 访问鉴权
        WebReqAuthReq webReqAuthReq = new WebReqAuthReq();
        webReqAuthReq.setReqUrl(uri);
        webReqAuthReq.setToken(token);
        webReqAuthReq.setMenuCode(menuCode);
        EResponse<WebReqAuthRes> eResponse = authFeignClient.webReqAuth(LogUtils.getThreadTraceId(), webReqAuthReq);

        if (eResponse.success()) {
            WebReqAuthRes webReqAuthRes = eResponse.getData();
            Integer dataPerm = webReqAuthRes.getDataPerm();
            Long userId = webReqAuthRes.getUserId();
            String account = webReqAuthRes.getAccount();
            Long tenantId = webReqAuthRes.getTenantId();
            Long departmentId = webReqAuthRes.getDepartmentId();
            String departmentChildren = webReqAuthRes.getDepartmentChildren();

            // 添加header
            context.addZuulRequestHeader(GatewayHeaders.SYSTEM_CODE, RequestUriUtils.getSystemCode(uri));
            if (Objects.nonNull(dataPerm)) {
                context.addZuulRequestHeader(GatewayHeaders.DATA_PERM, String.valueOf(dataPerm));
            }
            if (Objects.nonNull(userId)) {
                context.addZuulRequestHeader(GatewayHeaders.USER_ID, String.valueOf(userId));
            }
            if (Objects.nonNull(account)) {
                context.addZuulRequestHeader(GatewayHeaders.USER_ACCOUNT, account);
            }
            if (Objects.nonNull(tenantId)) {
                context.addZuulRequestHeader(GatewayHeaders.TENANT_ID, String.valueOf(tenantId));
            }
            if (Objects.nonNull(departmentId)) {
                context.addZuulRequestHeader(GatewayHeaders.DEPARTMENT_ID, String.valueOf(departmentId));
            }
            if (Objects.nonNull(departmentChildren)) {
                context.addZuulRequestHeader(GatewayHeaders.DEPARTMENT_CHILDREN, departmentChildren);
            }
        } else {
            ResponseUtils.failureResponse(eResponse);
        }

        return null;
    }
}
