package com.esv.superhive.gateway.common.component;

import com.alibaba.fastjson.JSONObject;
import com.esv.superhive.gateway.util.LogUtils;
import com.netflix.client.ClientException;
import com.netflix.hystrix.exception.HystrixTimeoutException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.cloud.netflix.zuul.filters.route.FallbackProvider;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.client.ClientHttpResponse;
import org.springframework.stereotype.Component;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.SocketTimeoutException;

/**
 * @description: 服务熔断降级处理
 * @author: 黄朝斌
 * @email: huangchaobin@esvtek.com
 * @createTime: 2020/07/01 13:43
 * @version:1.0
 */
@Component
@Slf4j
public class ServiceFallbackProvider implements FallbackProvider {
    
    /**
     * getRoute方法的返回值就是要监听的挂掉的微服务的名字
     * 如果需要所有服务都走这个熔断回退，则return "*" 或 return null
     **/
    @Override
    public String getRoute() {
        return null;
    }

    /**
     * 当服务无法执行的时候，返回托底信息
     **/
    @Override
    public ClientHttpResponse fallbackResponse(String route, Throwable cause) {
        return new ClientHttpResponse() {

            /**
             * ClientHttpResponse的fallback的状态码
             */
            @Override
            public HttpStatus getStatusCode() throws IOException {
                return HttpStatus.OK;
            }

            /**
             * ClientHttpResponse的fallback的状态码
             */
            @Override
            public int getRawStatusCode() throws IOException {
                return this.getStatusCode().value();
            }

            /**
             * ClientHttpResponse的fallback的状态码
             */
            @Override
            public String getStatusText() throws IOException {
                return this.getStatusCode().getReasonPhrase();
            }

            /**
             * Close this response, freeing any resources created.
             */
            @Override
            public void close() {

            }

            /**
             * 设置响应体
             */
            @Override
            public InputStream getBody() throws IOException {
                Throwable causeLevel2 = cause.getCause();
                Throwable causeLevel3 = (null == causeLevel2) ? null : causeLevel2.getCause();

                // Log输出cause层级信息
                log.info(cause.toString());
                if (null != causeLevel2) {
                    log.info(causeLevel2.toString());
                }
                if (null != causeLevel3) {
                    log.info(causeLevel3.toString());
                }

                String message;
                if (cause instanceof ClientException) {
                    message = "网关未路由到服务[" + route + "]，请稍后再试";
                    if (null != causeLevel2 && causeLevel2 instanceof RuntimeException) {
                        if (causeLevel3 instanceof SocketTimeoutException || causeLevel3 instanceof HystrixTimeoutException) {
                            message = "服务[" + route + "]响应超时";
                        }
                    }
                } else if (cause instanceof HystrixTimeoutException) {
                    message = "服务[" + route + "]响应超时";
                } else {
                    message = "网关未定义错误[" + cause.toString() + "]";
                }

                JSONObject json = new JSONObject();
                json.put("code", 500);
                json.put("responseTime", System.currentTimeMillis());
                json.put("message", message);
                json.put("logId", LogUtils.getThreadTraceId());

                return new ByteArrayInputStream(json.toJSONString().getBytes());
            }

            /**
             * 设置响应头信息
             */
            @Override
            public HttpHeaders getHeaders() {
                HttpHeaders headers = new HttpHeaders();
                headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

                return headers;
            }
        };
    }

}
