package com.esv.flink.sink;

import com.alibaba.fastjson.JSONObject;
import com.esv.flink.bean.EmqData;
import lombok.extern.slf4j.Slf4j;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.streaming.api.functions.sink.RichSinkFunction;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.Timestamp;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Properties;

/**
 * @description:
 * @project: my-flink-project
 * @name: com.esv.flink.MyRichSinkFunction
 * @author: chenfm
 * @email: chenfengman@esvtek.com
 * @createTime: 2020/8/3 15:26
 * @version: 1.0
 */
@Slf4j
public class EmqDataRichSinkFunction extends RichSinkFunction<EmqData> {

    private String postgresUrl;
    private String postgresUser;
    private String postgresPwd;

    private transient Connection connection = null;

    static {
        //将postgresql驱动注册到DriverManager中去
        try {
            Class.forName("org.postgresql.Driver");
        } catch (ClassNotFoundException e) {
            log.error(e.getMessage(), e);
        }
    }

    public EmqDataRichSinkFunction(Properties properties) {
        this.postgresUrl = properties.getProperty("postgres.url", "jdbc:postgresql://192.168.31.248:54321/iot");
        this.postgresUser = properties.getProperty("postgres.user", "iot");
        this.postgresPwd = properties.getProperty("postgres.pwd", "123456");
    }

    private String makeInsertSql(String topic, JSONObject jsonObject) {
        String tableName = getTableName(topic);
        HashSet<String> keySet = new LinkedHashSet<>(jsonObject.keySet());
        HashSet<String> valueSet = new LinkedHashSet<>(keySet.size());
        keySet.forEach(s -> valueSet.add(String.valueOf(jsonObject.get(s))));

        return "insert into "
                + tableName
                + "(time, device_id"
                + getSqlParams(keySet, false)
                + ") values (?,?"
                + getSqlParams(valueSet, true)
                + ")";
    }

    private String getTableName(String topic) {
        String modelId = topic.split("/")[2];
        return "iot_data_model_" + modelId;
    }

    private String getSqlParams(Collection collection, boolean isString) {
        StringBuilder builder = new StringBuilder();
        for (Object key : collection) {
            builder.append(",");
            if (isString) {
                builder.append("'");
            }
            builder.append(key.toString());
            if (isString) {
                builder.append("'");
            }
        }
        return builder.toString();
    }

    @Override
    public void open(Configuration parameters) throws Exception {
        super.open(parameters);
        log.info("EmqDataRichSinkFunction open.");
        try {
            //数据库链接
            this.connection = DriverManager.getConnection(postgresUrl, postgresUser, postgresPwd);
            log.info("postgres connect success: {}", postgresUrl);
        } catch (Exception e) {
            log.error("postgres connect failed.");
            log.error(e.getMessage(), e);
            throw(e);
        }
    }

    @Override
    public void close() throws Exception {
        super.close();
        log.info("EmqDataRichSinkFunction close.");

        if (this.connection != null) {
            this.connection.close();
        }
    }

    @Override
    public void invoke(EmqData emqData, Context context) throws Exception {
        log.debug("EmqDataRichSinkFunction invoke. topic=" + emqData.getTopic());

        String deviceId = emqData.getTopic().split("/")[3];
        String sql = this.makeInsertSql(emqData.getTopic(), emqData.getJsonObject());
        log.debug("insert sql: {}", sql);
        PreparedStatement pstmt = null;
        try {
            pstmt = connection.prepareStatement(sql);
            pstmt.setTimestamp(1, new Timestamp(emqData.getReportTime()));
            pstmt.setInt(2, Integer.parseInt(deviceId));
            pstmt.executeUpdate();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        } finally {
            if (pstmt != null) {
                pstmt.close();
            }
        }
    }

}
