package com.esv.flink.sink;

import com.alibaba.fastjson.JSONObject;
import com.esv.flink.bean.EmqData;
import lombok.extern.slf4j.Slf4j;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.streaming.api.functions.sink.RichSinkFunction;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.Timestamp;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedHashSet;

/**
 * @description:
 * @project: my-flink-project
 * @name: com.esv.flink.MyRichSinkFunction
 * @author: chenfm
 * @email: chenfengman@esvtek.com
 * @createTime: 2020/8/3 15:26
 * @version: 1.0
 */
@Slf4j
public class EmqDataRichSinkFunction extends RichSinkFunction<EmqData> {

    private String postgresqlInfo;
    private Connection connection = null;

    static {
        //将postgresql驱动注册到DriverManager中去
        try {
            Class.forName("org.postgresql.Driver");
        } catch (ClassNotFoundException e) {
            log.error(e.getMessage(), e);
        }
    }

    public EmqDataRichSinkFunction(String postgresqlInfo) {
        this.postgresqlInfo = postgresqlInfo;
    }

    //获取数据库连接信息
    private void getConnection() {
        try {
            String[] infoArray = postgresqlInfo.split("\\$\\$");
            //数据库链接
            String url = infoArray[0];
            String user = infoArray[1];
            String password = infoArray[2];
            //数据库连接信息
            this.connection = DriverManager.getConnection(url, user, password);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
    }

    private String makeInsertSql(String topic, JSONObject jsonObject) {
        String tableName = getTableName(topic);
        HashSet<String> keySet = new LinkedHashSet<>(jsonObject.keySet());
        HashSet<String> valueSet = new LinkedHashSet<>(keySet.size());
        keySet.forEach(s -> valueSet.add(String.valueOf(jsonObject.get(s))));

        return "insert into "
                + tableName
                + "(time, device_id"
                + getSqlParams(keySet, false)
                + ") values (?,?"
                + getSqlParams(valueSet, true)
                + ")";
    }

    private String getTableName(String topic) {
        String modelId = topic.split("/")[2];
        return "iot_data_model_" + modelId;
    }

    private String getSqlParams(Collection collection, boolean isString) {
        StringBuilder builder = new StringBuilder();
        for (Object key : collection) {
            builder.append(",");
            if (isString) {
                builder.append("'");
            }
            builder.append(key.toString());
            if (isString) {
                builder.append("'");
            }
        }
        return builder.toString();
    }

    @Override
    public void open(Configuration parameters) throws Exception {
        super.open(parameters);
        log.info("EmqDataRichSinkFunction open.");
        this.getConnection();
    }

    @Override
    public void close() throws Exception {
        super.close();
        log.info("EmqDataRichSinkFunction close.");

        if (this.connection != null) {
            this.connection.close();
        }
    }

    @Override
    public void invoke(EmqData emqData, Context context) throws Exception {
        log.info("EmqDataRichSinkFunction invoke. topic=" + emqData.getTopic());

        String deviceId = emqData.getTopic().split("/")[3];
        String sql = this.makeInsertSql(emqData.getTopic(), emqData.getJsonObject());
        log.info("insert sql: {}", sql);
        PreparedStatement pstmt = null;
        try {
            pstmt = connection.prepareStatement(sql);
            pstmt.setTimestamp(1, new Timestamp(emqData.getReportTime()));
            pstmt.setInt(2, Integer.parseInt(deviceId));
            pstmt.executeUpdate();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        } finally {
            if (pstmt != null) {
                pstmt.close();
            }
        }
    }

}
