package com.esv.flink.sink;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.esv.flink.bean.DataModelAlarmRule;
import com.esv.flink.bean.DeviceDataAlarm;
import com.esv.flink.bean.EmqData;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.streaming.api.functions.sink.RichSinkFunction;
import redis.clients.jedis.Jedis;

import java.math.BigDecimal;
import java.sql.*;
import java.util.List;
import java.util.Properties;

/**
 * @description:
 * @project: my-flink-project
 * @name: com.esv.flink.sink.AlarmRedisSinkFunction
 * @author: chenfm
 * @email: chenfengman@esvtek.com
 * @createTime: 2020/8/6 17:14
 * @version: 1.0
 */
@Slf4j
public class AlarmRedisSinkFunction extends RichSinkFunction<EmqData> {

    private String redisUrl;
    private String mysqlUrl;
    private String mysqlUser;
    private String mysqlPwd;

    private transient Jedis jedis;
    private transient Connection connection;

    static {
        //将mysql驱动注册到DriverManager中去
        try {
            Class.forName("com.mysql.cj.jdbc.Driver");
        } catch (ClassNotFoundException e) {
            log.error(e.getMessage(), e);
        }
    }

    public AlarmRedisSinkFunction(Properties properties) {
        this.redisUrl = properties.getProperty("redis.url", "192.168.31.248");
        this.mysqlUrl = properties.getProperty("mysql.url", "jdbc:mysql://192.168.31.248:3306/data_center?useUnicode=true&characterEncoding=utf8&autoReconnect=true&useSSL=false");
        this.mysqlUser = properties.getProperty("mysql.user", "data-center");
        this.mysqlPwd = properties.getProperty("mysql.pwd", "123456");
    }

    @Override
    public void open(Configuration parameters) throws Exception {
        super.open(parameters);
        try {
            jedis = new Jedis(redisUrl);
            log.info("redis connect success: {}", redisUrl);
        } catch (Exception e) {
            log.error("redis connect failed.");
            log.error(e.getMessage(), e);
            throw(e);
        }
        try {
            //数据库链接
            this.connection = DriverManager.getConnection(mysqlUrl, mysqlUser, mysqlPwd);
            log.info("mysql connect success: {}", mysqlUrl);
        } catch (Exception e) {
            log.error("mysql connect failed.");
            log.error(e.getMessage(), e);
            throw(e);
        }
    }

    @Override
    public void close() throws Exception {
        super.close();
        jedis.close();
        connection.close();
    }

    @Override
    public void invoke(EmqData emqData, Context context) throws Exception {
        long modelId = emqData.getModelId();
        String value = jedis.get("datacenter-iot-service::data_model::alarm_rule::" + modelId);
        log.debug("redis data model alarm rule value: {}", value);
        if (StringUtils.isBlank(value)) {
            return;
        }
        JSONObject jsonObject = emqData.getJsonObject();
        if (value.startsWith("\"") && value.endsWith("\"")) {
            value = value.substring(1, value.length() - 1);
            value = StringEscapeUtils.unescapeJava(value);
        }
        List<DataModelAlarmRule> ruleList = JSON.parseArray(value, DataModelAlarmRule.class);
        for (DataModelAlarmRule dataModelAlarmRule : ruleList) {
            Long ruleId = this.checkRule(dataModelAlarmRule, jsonObject);
            if (ruleId != null) {
                // 触发告警, 保存到mysql
                DeviceDataAlarm deviceDataAlarm =
                        new DeviceDataAlarm(emqData.getDeviceId(), ruleId, new Timestamp(emqData.getReportTime()));
                deviceDataAlarm.setReportValue(jsonObject.getString(dataModelAlarmRule.getPropertyCode()));
                this.saveAlarmToMysql(deviceDataAlarm);
            }
        }
    }

    /**
     * description 保存告警信息到mysql
     * param [deviceDataAlarm]
     * return void
     * author chenfm
     * createTime 2020/8/7 11:23
     **/
    private void saveAlarmToMysql(DeviceDataAlarm deviceDataAlarm) {
        String sql = "insert into device_data_alarm(device_id, alarm_rule_id, report_value, report_time, alarm_time)" +
                " values(?, ?, ?, ?, ?)";
        log.info("保存告警信息sql: {}", sql);
        try (PreparedStatement preparedStatement = connection.prepareStatement(sql)) {
            preparedStatement.setLong(1, deviceDataAlarm.getDeviceId());
            preparedStatement.setLong(2, deviceDataAlarm.getAlarmRuleId());
            preparedStatement.setString(3, deviceDataAlarm.getReportValue());
            preparedStatement.setTimestamp(4, deviceDataAlarm.getReportTime());
            preparedStatement.setTimestamp(5, deviceDataAlarm.getAlarmTime());
            preparedStatement.executeUpdate();
        } catch (SQLException e) {
            log.error(e.getMessage(), e);
        }
    }

    /**
     * description 判断是否满足告警条件
     * param [dataModelAlarmRule, jsonObject]
     * return java.lang.Long
     * author chenfm
     * createTime 2020/8/7 11:22
     **/
    private Long checkRule(DataModelAlarmRule dataModelAlarmRule, JSONObject jsonObject) {
        int propertyType = dataModelAlarmRule.getPropertyType();
        if (propertyType > 4) {
            // 不是数值类型, 无法比较大小
            return null;
        }
        String value = String.valueOf(jsonObject.get(dataModelAlarmRule.getPropertyCode()));
        String threshold = dataModelAlarmRule.getThreshold();
        switch (dataModelAlarmRule.getRuleExpression()) {
            case 1: // 大于(>)
                if ((propertyType != 1) && (stringValueCompare(value, threshold) > 0)) {
                    return dataModelAlarmRule.getId();
                }
                break;
            case 2: // 小于(<)
                if ((propertyType != 1) && (stringValueCompare(value, threshold) < 0)) {
                    return dataModelAlarmRule.getId();
                }
                break;
            case 3: // 等于(=)
                if ((propertyType == 1 && StringUtils.equals(threshold, value))
                        || (propertyType != 1 && stringValueCompare(value, threshold) == 0)) {
                    return dataModelAlarmRule.getId();
                }
                break;
            case 4: // 不等于(!=)
                if ((propertyType == 1 && !StringUtils.equals(threshold, value))
                        || (propertyType != 1 && stringValueCompare(value, threshold) != 0)) {
                    return dataModelAlarmRule.getId();
                }
                break;
            default:
                return null;
        }
        return null;
    }

    private int stringValueCompare(String value, String threshold) {
        return new BigDecimal(value).compareTo(new BigDecimal(threshold));
    }

}
