/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.esv.flink;

import com.esv.flink.bean.EmqData;
import com.esv.flink.sink.AlarmRedisSinkFunction;
import com.esv.flink.sink.EmqDataRichSinkFunction;
import lombok.extern.slf4j.Slf4j;
import org.apache.flink.api.common.functions.FlatMapFunction;
import org.apache.flink.api.common.functions.MapFunction;
import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.api.java.utils.ParameterTool;
import org.apache.flink.streaming.api.datastream.DataStream;
import org.apache.flink.streaming.api.datastream.SingleOutputStreamOperator;
import org.apache.flink.streaming.api.environment.StreamExecutionEnvironment;
import org.apache.flink.streaming.api.functions.sink.RichSinkFunction;
import org.apache.flink.streaming.api.functions.source.ParallelSourceFunction;
import org.apache.flink.util.Collector;
import org.fusesource.mqtt.client.Future;
import org.fusesource.mqtt.client.FutureConnection;
import org.fusesource.mqtt.client.Message;

import java.nio.charset.StandardCharsets;

/**
 * Skeleton for a Flink Streaming Job.
 *
 * <p>For a tutorial how to write a Flink streaming application, check the
 * tutorials and examples on the <a href="https://flink.apache.org/docs/stable/">Flink Website</a>.
 *
 * <p>To package your application into a JAR file for execution, run
 * 'mvn clean package' on the command line.
 *
 * <p>If you change the name of the main class (with the public static void main(String[] args))
 * method, change the respective entry in the POM.xml file (simply search for 'mainClass').
 */
@Slf4j
public class StreamingJob {

	public static void main(String[] args) throws Exception {
		log.info("start flink.");
		// set up the streaming execution environment
		final StreamExecutionEnvironment env = StreamExecutionEnvironment.getExecutionEnvironment();
		ParameterTool params = ParameterTool.fromArgs(new String[0]);
		env.getConfig().setGlobalJobParameters(params);

        String brokerUrl = params.get("brokerUrl", "tcp://192.168.31.248:1883");
        String postgresqlInfo = params.get("postgresqlInfo", "jdbc:postgresql://192.168.31.248:5432/iot$$iot$$123456");
        String redisInfo = params.get("redisInfo", "192.168.31.248");
        String mysqlInfo = params.get("mysqlInfo", "192.168.31.248:3306$$data_center$$123456");
        log.info("load param brokerUrl: {}", brokerUrl);
        log.info("load param postgresqlInfo: {}", postgresqlInfo);
        log.info("load param redisInfo: {}", redisInfo);
        log.info("load param mysqlInfo: {}", mysqlInfo);

		env.setParallelism(1);
        EmqSource emqSource = new EmqSource(brokerUrl);
		DataStream<String> inStream = env.addSource(emqSource);
		inStream.print();

		SingleOutputStreamOperator<EmqData> emqDataStream = inStream.flatMap(new FlatMapFunction<String, EmqData>() {
			@Override
			public void flatMap(String s, Collector<EmqData> collector) throws Exception {
				String[] tokens = s.toLowerCase().split("@@");
				if (tokens.length > 1) {
					String topic = tokens[0];
					String content = tokens[1];
					collector.collect(new EmqData(topic, content, 1));
				}
			}
		});
		RichSinkFunction<EmqData> emqDataRichSinkFunction = new EmqDataRichSinkFunction(postgresqlInfo);
		emqDataStream.addSink(emqDataRichSinkFunction);
		RichSinkFunction<EmqData> alarmRedisSinkFunction = new AlarmRedisSinkFunction(redisInfo, mysqlInfo);
		emqDataStream.addSink(alarmRedisSinkFunction);

		SingleOutputStreamOperator<Tuple2<String, Integer>> res = emqDataStream.map(new MapFunction<EmqData, Tuple2<String, Integer>>() {
			@Override
			public Tuple2<String, Integer> map(EmqData emqData) throws Exception {
				return new Tuple2<>(emqData.getTopic(), emqData.getCount());
			}
		}).keyBy(0).sum(1);

		if (params.has("output")) {
			res.writeAsText(params.get("output"));
		} else {
			log.info("Printing result to stdout. Use --output to specify output path.");
			res.print();
		}
//		res.writeAsText("D://flink_result.txt");

		// execute program
		env.execute("Streaming WordCount");
	}

	public static class EmqSource implements ParallelSourceFunction<String> {
		private static final long serialVersionUID = 1L;
		private volatile boolean isRunning = true;
		private String brokerUrl;

		public EmqSource(String brokerUrl) {
		    this.brokerUrl = brokerUrl;
        }

		public void run(SourceContext<String> ctx) throws Exception {
			EmqClient emqClient = new EmqClient(brokerUrl);
			FutureConnection connection = emqClient.run();
			int num = 0;
			while (isRunning) {
				Future<Message> future = connection.receive();
				Message message = future.await();
				num ++;
				String topic = message.getTopic();
				String context = new String(message.getPayload(), StandardCharsets.UTF_8);
				ctx.collect(topic + "@@" + context);
				log.info("接收数据条数num:" + num);
				System.out.println("接收数据条数num:" + num);
			}
			connection.disconnect();
		}

		public void cancel() {
			isRunning = false;
		}
	}

}
