package com.esv.freight.file.module.upload.service.impl;

import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.esv.freight.file.common.exception.EException;
import com.esv.freight.file.common.util.AESSecretUtils;
import com.esv.freight.file.module.upload.dao.FileMetaDao;
import com.esv.freight.file.module.upload.entity.FileMetaEntity;
import com.esv.freight.file.module.upload.form.FileForm;
import com.esv.freight.file.module.upload.service.FileMetaService;
import com.esv.freight.file.module.upload.vo.FileVO;
import com.mongodb.client.gridfs.model.GridFSFile;
import lombok.extern.slf4j.Slf4j;
import org.bson.types.ObjectId;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.gridfs.GridFsTemplate;
import org.springframework.stereotype.Service;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.Base64;


@Service("fileMetaService")
@Slf4j
@RefreshScope
public class FileMetaServiceImpl extends ServiceImpl<FileMetaDao, FileMetaEntity> implements FileMetaService {

    @Value("${aes.sha1prng.key:3.1415926535}")
    private String AES_KEY;

    @Value("${file.download.base-url:undefined}")
    private String FILE_DOWNLOAD_BASE_URL;

    private GridFsTemplate gridFsTemplate;

    @Autowired
    public FileMetaServiceImpl(GridFsTemplate gridFsTemplate) {
        this.gridFsTemplate = gridFsTemplate;
    }

    @Override
    public FileVO uploadFile(FileForm fileForm) throws EException {
        String fileType = fileForm.getFileType();
        String fileName = fileForm.getFileName();
        String fileData = fileForm.getFileData();
        byte[] bytes = Base64.getDecoder().decode(fileData);
        InputStream inputStream = new ByteArrayInputStream(bytes);

        // MongoDB存储文件
        ObjectId objectId;
        String fileId;
        try {
            objectId = gridFsTemplate.store(inputStream, fileName, getFileContentType(fileType));
            fileId = objectId.toString();
        } catch (Exception e) {
            log.error("存储文件数据时发生错误");
            log.error(e.getMessage(), e);
            throw new EException("存储文件数据时发生错误");
        }

        // MySQL存储文件元数据
        FileMetaEntity metaEntity = new FileMetaEntity();
        metaEntity.setFileId(fileId);
        metaEntity.setFileSize((long) bytes.length);
        metaEntity.setFileName(fileName);
        metaEntity.setFileType(fileType);
        try {
            this.baseMapper.insert(metaEntity);
        } catch (Exception e) {
            log.error("存储文件元信息时发生错误");
            log.error(e.getMessage(), e);

            // 删除MongoDB存储文件
            try {
                deleteMongoFile(fileId);
            } catch (Exception ex) {
                log.error("删除文件数据时发生错误，文件id={}", fileId);
                log.error(e.getMessage(), e);
            }

            throw new EException("存储文件元信息发生错误");
        }

        // 返回数据
        String aesId = AESSecretUtils.encryptToStr(fileId + "," + System.currentTimeMillis(), AES_KEY);
        String downloadUrl = FILE_DOWNLOAD_BASE_URL + aesId;
        FileVO fileVO = new FileVO(fileId, downloadUrl);

        return fileVO;
    }

    /**
     * description 删除MongoDB文件
     * param [id]
     * return void
     * author Administrator
     * createTime 2020/04/14 15:33
     **/
    private void deleteMongoFile(String id) throws Exception {
        Query query = Query.query(Criteria.where("_id").is(id));
        GridFSFile gridFSFile = gridFsTemplate.findOne(query);
        if (null != gridFSFile) {
            gridFsTemplate.delete(query);
        }
    }

    /**
     * description 获取文件的ContentType
     * param [fileType]
     * return java.lang.String
     * author Administrator
     * createTime 2020/04/14 14:47
     **/
    private String getFileContentType(String fileType) {
        String contentType;
        switch (fileType) {
            case "image":
                contentType = "application/x-jpg";
                break;
            case "pdf":
                contentType = "application/pdf";
                break;
            case "excel":
                contentType = "application/vnd.ms-excel";
                break;
            case "apk":
                contentType = "application/vnd.android.package-archive";
                break;
            default:
                contentType = "undefined";
                break;
        }
        return contentType;
    }

}