package com.esv.freight.customer.schedule;

import com.esv.freight.customer.common.unionpay.AcpService;
import com.esv.freight.customer.common.unionpay.DemoBase;
import com.esv.freight.customer.common.unionpay.SDKConfig;
import com.esv.freight.customer.common.util.DateUtils;
import com.esv.freight.customer.module.pay.entity.UnionpayPublicCertUpdateEntity;
import com.esv.freight.customer.module.pay.entity.UnionpayStatementEntity;
import com.esv.freight.customer.module.pay.service.UnionpayPublicCertUpdateService;
import com.esv.freight.customer.module.pay.service.UnionpayStatementService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.util.*;

/**
 * @description: 银联定时任务
 * @project: freight-customer-service
 * @name: com.esv.freight.customer.schedule.UnionpayTask
 * @author: 黄朝斌
 * @email: huangchaobin@esvtek.com
 * @createTime: 2020/06/12 15:44
 * @version:1.0
 */
@Component
@Slf4j
public class UnionpayTask {

    @Value("${unionpay.mer-id}")
    private String unionPayMerId;

    @Autowired
    private UnionpayStatementService unionpayStatementService;
    @Autowired
    private UnionpayPublicCertUpdateService unionpayPublicCertUpdateService;

    /**
     * 定时任务触发器：获取前1/2天对账单
     * 定时策略：每天1-2点，每20分钟执行一次
     **/
    @Scheduled(cron = "0 0/20 1-2 * * ?")
    public void getStatementTaskTrigger() {
        this.initLogTraceId();
        log.info("------------------- [Unionpay_Task_001]获取银联对账单开始 -------------------");
        this.getStatementWork();
        log.info("------------------- [Unionpay_Task_001]获取银联对账单结束 -------------------");
    }

    /**
     * 定时任务触发器：银联加密公钥更新
     * 定时策略：每天2点30执行一次
     **/
    @Scheduled(cron = "0 30 2 * * ?")
    public void updateCertTaskTrigger() {
        this.initLogTraceId();
        log.info("------------------- [Unionpay_Task_002]银联加密公钥更新开始 -------------------");
        this.updateCertTaskWork();
        log.info("------------------- [Unionpay_Task_002]银联加密公钥更新结束 -------------------");
    }

    /**
     * 获取对账单
     **/
    private void getStatementWork() {
        // 生成要获取的对账单日期
        List<String> dateList = new ArrayList<>(2);
        dateList.add(this.getBeforeDate(1));
        dateList.add(this.getBeforeDate(2));

        for (int i = 0; i < dateList.size(); i++) {
            String settleDate = dateList.get(i);
            UnionpayStatementEntity unionpayStatementEntity = this.unionpayStatementService.getRecordByDate(settleDate);
            if (null == unionpayStatementEntity) {
                Map<String, String> resMap = this.reqUnionpay(settleDate);
                // TODO、对账文件存储处理（当前没有有效返回数据）
            } else {
                continue;
            }
        }
    }

    /**
     * description 请求银联获取对账单
     * param [settleDate]
     * return java.util.Map<java.lang.String,java.lang.String>
     * author HuangChaobin
     * createTime 2020/06/12 16:59
     **/
    private Map<String, String> reqUnionpay(String settleDate) {
        Map<String, String> resMap = null;
        Map<String, String> contentData = new HashMap<>(128);

        /***银联全渠道系统，产品参数，除了encoding自行选择外其他不需修改***/
        //版本号 全渠道默认值
        contentData.put("version", DemoBase.version);
        //字符集编码 可以使用UTF-8,GBK两种方式
        contentData.put("encoding", DemoBase.encoding);
        //签名方法
        contentData.put("signMethod", SDKConfig.getConfig().getSignMethod());
        //交易类型 76-对账文件下载
        contentData.put("txnType", "76");
        //交易子类 01-对账文件下载
        contentData.put("txnSubType", "01");
        //业务类型，固定
        contentData.put("bizType", "000000");

        /***商户接入参数***/
        //接入类型，商户接入填0 ，不需修改（0：直连商户， 1： 收单机构 2：平台商户）
        contentData.put("accessType", "0");
        //商户代码，请替换正式商户号测试，如使用的是自助化平台注册的777开头的商户号，该商户号没有权限测文件下载接口的，
        // 请使用测试参数里写的文件下载的商户号和日期测。如需777商户号的真实交易的对账文件，请使用自助化平台下载文件
        contentData.put("merId", unionPayMerId);
        //清算日期，如果使用正式商户号测试则要修改成自己想要获取对账文件的日期， 测试环境如果使用700000000000001商户号则固定填写0119
        contentData.put("settleDate", settleDate);
        //订单发送时间，取系统时间，格式为yyyyMMddHHmmss，必须取当前时间，否则会报txnTime无效
        contentData.put("txnTime", DateUtils.format(DateUtils.getSysdate(), DateUtils.DATE_FORMAT4));
        //文件类型，一般商户填写00即可
        contentData.put("fileType", "00");

        /**请求参数设置完毕，以下对请求参数进行签名并发送http post请求，接收同步应答报文------------->**/
        //报文中certId,signature的值是在signData方法中获取并自动赋值的，只要证书配置正确即可
        Map<String, String> reqData = AcpService.sign(contentData, DemoBase.encoding);
        //获取请求银联的前台地址：对应属性文件acp_sdk.properties文件中的acpsdk.fileTransUrl
        String url = SDKConfig.getConfig().getFileTransUrl();
        Map<String, String> rspData = AcpService.post(reqData, url, DemoBase.encoding);

        if (!rspData.isEmpty()) {
            if (AcpService.validate(rspData, DemoBase.encoding)) {
                String respCode = rspData.get("respCode");
                if ("00".equals(respCode)) {
                    resMap = rspData;
                } else {
                    //其他应答码为失败请排查原因
                    log.error("[银联开放平台]获取对账单失败");
                }
            } else {
                log.error("[银联开放平台]验证签名失败");
            }
        } else {
            //未返回正确的http状态
            log.error("[银联开放平台]未获取到返回报文或返回http状态码非200");
        }
        return resMap;
    }

    /**
     * description 当前日期减天数
     * param [days]
     * return java.lang.String
     * author HuangChaobin
     * createTime 2020/06/12 16:07
     **/
    private String getBeforeDate(int days) {
        Date cur = DateUtils.getSysdate();
        Date beforeDate = DateUtils.minusDays(cur, days);
        return DateUtils.format(beforeDate, DateUtils.DATE_FORMAT7);
    }

    /**
     * 银联加密公钥更新
     **/
    private void updateCertTaskWork() {
        Map<String, String> contentData = new HashMap<>(128);

        /***银联全渠道系统，产品参数，除了encoding自行选择外其他不需修改***/
        //版本号 全渠道默认值
        contentData.put("version", DemoBase.version);
        //字符集编码 可以使用UTF-8,GBK两种方式
        contentData.put("encoding", DemoBase.encoding);
        //签名方法
        contentData.put("signMethod", SDKConfig.getConfig().getSignMethod());
        //交易类型 95-银联加密公钥更新查询
        contentData.put("txnType", "95");
        //交易子类型  默认00
        contentData.put("txnSubType", "00");
        //业务类型，固定
        contentData.put("bizType", "000000");
        //渠道类型
        contentData.put("channelType", "07");
        //01：敏感信息加密公钥(只有01可用)
        contentData.put("certType", "01");

        /***商户接入参数***/
        //商户号码（商户号码777290058110097仅做为测试调通交易使用，该商户号配置了需要对敏感信息加密）测试时请改成自己申请的商户号，【自己注册的测试777开头的商户号不支持代收产品】
        contentData.put("merId", unionPayMerId);
        //接入类型，商户接入填0 ，不需修改（0：直连商户， 1： 收单机构 2：平台商户）
        contentData.put("accessType", "0");
        String txnTime = DateUtils.format(DateUtils.getSysdate(), DateUtils.DATE_FORMAT4);
        //商户订单号，8-40位数字字母，不能含“-”或“_”，可以自行定制规则
        contentData.put("orderId", "ZSGX" + txnTime);
        //订单发送时间，取系统时间，格式为yyyyMMddHHmmss，必须取当前时间，否则会报txnTime无效
        contentData.put("txnTime", txnTime);

        /**请求参数设置完毕，以下对请求参数进行签名并发送http post请求，接收同步应答报文------------->**/
        //报文中certId,signature的值是在signData方法中获取并自动赋值的，只要证书配置正确即可
        Map<String, String> reqData = AcpService.sign(contentData, DemoBase.encoding);
        //交易请求url从配置文件读取对应属性文件acp_sdk.properties中的 acpsdk.backTransUrl
        String requestBackUrl = SDKConfig.getConfig().getBackRequestUrl();
        //发送请求报文并接受同步应答（默认连接超时时间30秒，读取返回结果超时时间30秒）;这里调用signData之后，调用submitUrl之前不能对submitFromData中的键值对做任何修改，如果修改会导致验签不通过
        Map<String, String> rspData = AcpService.post(reqData, requestBackUrl, DemoBase.encoding);

        UnionpayPublicCertUpdateEntity entity = new UnionpayPublicCertUpdateEntity();
        if (!rspData.isEmpty()) {
            if (AcpService.validate(rspData, DemoBase.encoding)) {
                String respCode = rspData.get("respCode");
                if ("00".equals(respCode)) {
                    int resultCode = AcpService.updateEncryptCert(rspData, "UTF-8");
                    if (resultCode == 1) {
                        entity.setUpdateResult(0);
                        log.info("[银联开放平台]加密公钥更新成功");
                    } else if (resultCode == 0) {
                        entity.setUpdateResult(1);
                        log.info("[银联开放平台]加密公钥无更新");
                    } else {
                        entity.setUpdateResult(2);
                        log.error("[银联开放平台]加密公钥更新失败");
                    }
                } else {
                    //其他应答码为失败请排查原因
                    entity.setUpdateResult(3);
                    log.error("[银联开放平台]加密公钥更新查询失败");
                }
            } else {
                entity.setUpdateResult(3);
                log.error("[银联开放平台]验证签名失败");
            }
        } else {
            //未返回正确的http状态
            entity.setUpdateResult(3);
            log.error("[银联开放平台]未获取到返回报文或返回http状态码非200");
        }

        // 保存更新记录
        entity.setUpdateDate(DateUtils.format(DateUtils.getSysdate(), DateUtils.DATE_FORMAT2));
        this.unionpayPublicCertUpdateService.saveRecord(entity);
    }

    /**
     * description 设置日志traceId
     * param []
     * return void
     * author HuangChaobin
     * createTime 2020/06/15 14:00
     **/
    private void initLogTraceId() {
        String traceId = MDC.get("traceId");
        if (StringUtils.isBlank(traceId)) {
            traceId = UUID.randomUUID().toString().replace("-", "");
            MDC.put("traceId", traceId);
        }
    }

}
