package com.esv.freight.customer.module.driver.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.esv.freight.customer.common.component.ErrorMessageComponent;
import com.esv.freight.customer.common.exception.EException;
import com.esv.freight.customer.common.util.ReqUtils;
import com.esv.freight.customer.common.vo.PageResultVO;
import com.esv.freight.customer.module.carrier.CarrierConstants;
import com.esv.freight.customer.module.driver.DriverConstants;
import com.esv.freight.customer.module.driver.dto.DriverDetailDto;
import com.esv.freight.customer.module.driver.dto.DriverVehicleDto;
import com.esv.freight.customer.module.driver.entity.DriverAccountEntity;
import com.esv.freight.customer.module.driver.form.DriverQueryForm;
import com.esv.freight.customer.module.driver.form.DriverVehicleForm;
import com.esv.freight.customer.module.driver.form.DriverVehicleQueryForm;
import com.esv.freight.customer.module.driver.service.DriverAccountService;
import com.esv.freight.customer.module.driver.service.DriverVehicleService;
import com.esv.freight.customer.module.driver.vo.*;
import com.esv.freight.customer.module.vehicle.VehicleConstants;
import com.esv.freight.customer.module.vehicle.dto.VehicleDetailDto;
import com.esv.freight.customer.module.vehicle.entity.VehicleDriverEntity;
import com.esv.freight.customer.module.vehicle.entity.VehicleEntity;
import com.esv.freight.customer.module.vehicle.service.VehicleDriverService;
import com.esv.freight.customer.module.vehicle.service.VehicleService;
import com.esv.freight.customer.module.vehicle.vo.VehicleDetailVO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @description:
 * @project: freight-customer-service
 * @name: com.esv.freight.customer.module.driver.service.impl.DriverVehicleServiceImpl
 * @author: 黄朝斌
 * @email: huangchaobin@esvtek.com
 * @createTime: 2020/04/29 16:55
 * @version:1.0
 */
@Service("driverVehicleService")
@Slf4j
public class DriverVehicleServiceImpl implements DriverVehicleService {

    private ErrorMessageComponent errorMessageComponent;

    private DriverAccountService driverAccountService;

    private VehicleService vehicleService;

    private VehicleDriverService vehicleDriverService;

    public DriverVehicleServiceImpl(ErrorMessageComponent errorMessageComponent, DriverAccountService driverAccountService,
                                    VehicleService vehicleService, VehicleDriverService vehicleDriverService) {
        this.errorMessageComponent = errorMessageComponent;
        this.driverAccountService = driverAccountService;
        this.vehicleService = vehicleService;
        this.vehicleDriverService = vehicleDriverService;
    }

    @Override
    public List<DriverVehicleListVO> getDriverBindVehicleList(Long id) {
        // 1:校验是否有效司机ID
        int count = this.driverAccountService.getBaseMapper().selectCount(new QueryWrapper<DriverAccountEntity>().lambda()
                .eq(DriverAccountEntity::getId, id));
        if (0 == count) {
            throw new EException(1001, errorMessageComponent.getCarrierDriverVehicleList1001());
        }

        // 2:查询司机绑定的车辆列表
        List<DriverVehicleListVO> voList = this.vehicleDriverService.getDriverBindVehicleList(id);
        if (0 == voList.size()) {
            throw new EException(1002, errorMessageComponent.getCarrierDriverVehicleList1002());
        }

        return voList;
    }

    @Override
    public List<VehicleDriverListVO> getVehicleBindDriverList(Long id) {
        // 1:校验是否有效车辆ID
        int count = this.vehicleService.getBaseMapper().selectCount(new QueryWrapper<VehicleEntity>().lambda()
                .eq(VehicleEntity::getId, id));
        if (0 == count) {
            throw new EException(1001, errorMessageComponent.getCarrierDriverDriverList1001());
        }

        // 2:查询司机绑定的车辆列表
        List<VehicleDriverListVO> voList = this.vehicleDriverService.getVehicleBindDriverList(id);
        if (0 == voList.size()) {
            throw new EException(1002, errorMessageComponent.getCarrierDriverDriverList1002());
        }

        return voList;
    }

    @Override
    public Integer driverBindVehicle(DriverVehicleForm form) {
        // 1:校验
        // 1.1:校验司机ID是否有效
        DriverAccountEntity driverAccountEntity = this.driverAccountService.getById(form.getDriverId());
        if (null == driverAccountEntity) {
            throw new EException(1001, errorMessageComponent.getCarrierDriverVehicleBind1001());
        }
        // 1.2:校验司机帐号是否停用
        if (DriverConstants.ACCOUNT_STATUS_BLOCK.equals(driverAccountEntity.getAccountStatus())) {
            throw new EException(1002, errorMessageComponent.getCarrierDriverVehicleBind1002());
        }
        // 1.3:校验司机帐号是否未审核通过
        if (!DriverConstants.ACCOUNT_AUDIT_STATUS_SUCCESS.equals(driverAccountEntity.getAuditStatus())) {
            throw new EException(1003, errorMessageComponent.getCarrierDriverVehicleBind1003());
        }
        // 1.4:校验车辆ID是否有效
        VehicleEntity vehicleEntity = this.vehicleService.getById(form.getVehicleId());
        if (null == vehicleEntity) {
            throw new EException(1010, errorMessageComponent.getCarrierDriverVehicleBind1010());
        }
        // 1.5:校验车辆是否已停用
        if (VehicleConstants.VEHICLE_ACCOUNT_STATUS_BLOCK.equals(vehicleEntity.getVehicleStatus())) {
            throw new EException(1011, errorMessageComponent.getCarrierDriverVehicleBind1011());
        }
        // 1.6:校验车辆是否未审核通过
        if (!VehicleConstants.VEHICLE_AUDIT_STATUS_SUCCESS.equals(vehicleEntity.getAuditStatus())) {
            throw new EException(1012, errorMessageComponent.getCarrierDriverVehicleBind1012());
        }
        // 1.7:校验司机与车辆所属承运商是否不一致
        if (!driverAccountEntity.getCarrierId().equals(vehicleEntity.getCarrierId())) {
            throw new EException(1020, errorMessageComponent.getCarrierDriverVehicleBind1020());
        }
        // 1.8:校验是否重复绑定
        int count = this.vehicleDriverService.getBaseMapper().selectCount(new QueryWrapper<VehicleDriverEntity>().lambda()
                .eq(VehicleDriverEntity::getVehicleId, form.getVehicleId())
                .eq(VehicleDriverEntity::getDriverId, form.getDriverId()));
        if (0 < count) {
            throw new EException(1021, errorMessageComponent.getCarrierDriverVehicleBind1021());
        }

        // 2:绑定司机与车辆
        VehicleDriverEntity vehicleDriverEntity = new VehicleDriverEntity();
        vehicleDriverEntity.setCarrierId(driverAccountEntity.getCarrierId());
        vehicleDriverEntity.setVehicleId(form.getVehicleId());
        vehicleDriverEntity.setDriverId(form.getDriverId());
        vehicleDriverEntity.setSelected(false);
        vehicleDriverEntity.setOperateUser(ReqUtils.getRequestUserAccount());
        return this.vehicleDriverService.getBaseMapper().insert(vehicleDriverEntity);
    }

    @Override
    public Integer driverUnbindVehicle(DriverVehicleForm form) {
        // 1:校验
        // 1.1:校验司机ID是否有效
        DriverAccountEntity driverAccountEntity = this.driverAccountService.getById(form.getDriverId());
        if (null == driverAccountEntity) {
            throw new EException(1001, errorMessageComponent.getCarrierDriverVehicleUnbind1001());
        }
        String driverAccount = driverAccountEntity.getAccount();
        // 1.2:校验车辆ID是否有效
        VehicleEntity vehicleEntity = this.vehicleService.getById(form.getVehicleId());
        if (null == vehicleEntity) {
            throw new EException(1010, errorMessageComponent.getCarrierDriverVehicleUnbind1010());
        }
        // 1.3:校验车辆与司机是否绑定
        VehicleDriverEntity vehicleDriverEntity = this.vehicleDriverService.getBaseMapper().selectOne(new QueryWrapper<VehicleDriverEntity>().lambda()
                        .eq(VehicleDriverEntity::getVehicleId, form.getVehicleId())
                        .eq(VehicleDriverEntity::getDriverId, form.getDriverId()));
        if (null == vehicleDriverEntity) {
            throw new EException(1020, errorMessageComponent.getCarrierDriverVehicleUnbind1020());
        }
        // 1.4:校验司机自主绑定的车辆，平台不可解绑
        if (VehicleConstants.VEHICLE_SOURCE_TYPE_REGISTER.equals(vehicleEntity.getSourceType())
                && driverAccount.equals(vehicleEntity.getCreateUser())) {
            throw new EException(1021, errorMessageComponent.getCarrierDriverVehicleUnbind1021());
        }

        // 1:解绑司机与车辆
        return this.vehicleDriverService.getBaseMapper().delete(new QueryWrapper<VehicleDriverEntity>().lambda()
                        .eq(VehicleDriverEntity::getVehicleId, form.getVehicleId())
                        .eq(VehicleDriverEntity::getDriverId, form.getDriverId()));
    }

    @Override
    public VehicleDriverEntity getRecordBy2Id(DriverVehicleForm form) {
        return this.vehicleDriverService.getBaseMapper().selectOne(new QueryWrapper<VehicleDriverEntity>().lambda()
                .eq(VehicleDriverEntity::getVehicleId, form.getVehicleId())
                .eq(VehicleDriverEntity::getDriverId, form.getDriverId()));
    }

    @Override
    public JSONObject checkBind(DriverVehicleForm form) {
        JSONObject data = new JSONObject();

        // 1:校验司机与车辆是否绑定
        VehicleDriverEntity vehicleDriverEntity = this.getRecordBy2Id(form);
        if (null == vehicleDriverEntity) {
            throw new EException(1001, errorMessageComponent.getCarrierDriverVehicleCheckBind1001());
        }

        // 2:获取司机信息
        DriverDetailDto driverDetailDto = this.driverAccountService.getDriverDetailById(form.getDriverId());
        if (null == driverDetailDto) {
            throw new EException(1002, errorMessageComponent.getCarrierDriverVehicleCheckBind1002());
        } else {
            DriverDetailVO driverDetailVO = new DriverDetailVO();
            BeanUtils.copyProperties(driverDetailDto, driverDetailVO);
            driverDetailVO.setCreateTime(driverDetailDto.getCreateTime().getTime());
            if (CarrierConstants.CARRIER_TYPE_COMPANY.equals(driverDetailDto.getCarrierType())) {
                driverDetailVO.setCarrierName(driverDetailDto.getCarrierFullName());
            } else {
                driverDetailVO.setCarrierName(driverDetailDto.getCarrierContactor());
            }
            data.put("driverInfo", driverDetailVO);
        }

        // 3:获取车辆信息
        VehicleDetailDto vehicleDetailDto = this.vehicleService.getVehicleDetail(form.getVehicleId());
        if (null == vehicleDetailDto) {
            throw new EException(1003, errorMessageComponent.getCarrierDriverVehicleCheckBind1003());
        } else {
            VehicleDetailVO vehicleDetailVO = new VehicleDetailVO();
            BeanUtils.copyProperties(vehicleDetailDto, vehicleDetailVO);
            vehicleDetailVO.setCreateTime(vehicleDetailDto.getCreateTime().getTime());
            if (CarrierConstants.CARRIER_TYPE_COMPANY.equals(vehicleDetailDto.getCarrierType())) {
                vehicleDetailVO.setCarrierName(vehicleDetailDto.getCarrierFullName());
            } else {
                vehicleDetailVO.setCarrierName(vehicleDetailDto.getCarrierContactor());
            }
            data.put("vehicleInfo", vehicleDetailVO);
        }

        return data;
    }

    @Override
    public List<CarrierDriverVehicleVO> getDriverVehicle4Carrier(Long carrierId) {
        List<DriverVehicleDto> dtoList = this.vehicleDriverService.getDriverVehicle4Carrier(carrierId);
        List<CarrierDriverVehicleVO> voList = new ArrayList<>();

        List<CarrierDriverBindVehicleVO> driverVehicleList;
        CarrierDriverVehicleVO vo;
        for (int i = 0; i < dtoList.size(); i++) {
            DriverVehicleDto dto = dtoList.get(i);
            CarrierDriverBindVehicleVO bindVO = new CarrierDriverBindVehicleVO();
            BeanUtils.copyProperties(dto, bindVO);

            // 判断是否同一个承运商
            if (0 != voList.size() && voList.get(voList.size() - 1).getCarrierId().equals(dto.getCarrierId())) {
                vo = voList.get(voList.size() - 1);
                driverVehicleList = vo.getDriverVehicle();
                driverVehicleList.add(bindVO);
                vo.setDriverVehicle(driverVehicleList);
            } else {
                driverVehicleList = new ArrayList<>();
                vo = new CarrierDriverVehicleVO();
                vo.setCarrierId(dto.getCarrierId());
                if (CarrierConstants.CARRIER_TYPE_COMPANY.equals(dto.getCarrierType())) {
                    vo.setCarrierName(dto.getCarrierFullName());
                } else {
                    vo.setCarrierName(dto.getCarrierContactor());
                }
                driverVehicleList.add(bindVO);
                vo.setDriverVehicle(driverVehicleList);
                voList.add(vo);
            }
        }

        return voList;
    }

    @Override
    public PageResultVO selectVehicleCanBindDriverList(DriverQueryForm form) {
        // 1:校验是否有效车辆ID
        VehicleEntity vehicleEntity = this.vehicleService.getById(form.getId());
        if (null == vehicleEntity) {
            throw new EException(1001, errorMessageComponent.getCarrierDriverVehicleCanBind1001());
        }

        // 2:构建查询条件
        Map<String, Object> queryObj = new HashMap<>(2);
        queryObj.put("vehicleId", form.getId());
        queryObj.put("keywords", StringUtils.trimToNull(form.getKeywords()));

        // 3:查询
        IPage<DriverVehicleDto> page = new Page<>(form.getPageNum(), form.getPageSize());
        this.vehicleDriverService.getVehicleCanBindDriverList(page, queryObj);

        // 4:数据转换
        List<DriverVehicleDto> dtoList = page.getRecords();
        List<VehicleCanBindDriverVO> targetRecordList = new ArrayList<>();
        for (DriverVehicleDto dto : dtoList) {
            VehicleCanBindDriverVO vo = new VehicleCanBindDriverVO();
            vo.setId(dto.getDriverId());
            vo.setAccount(dto.getDriverAccount());
            vo.setName(dto.getDriverName());

            targetRecordList.add(vo);
        }

        return new PageResultVO(page, targetRecordList);
    }

    @Override
    public PageResultVO getDriverVehicle4Params(DriverVehicleQueryForm form) {
        // 3:查询
        IPage<DriverVehicleDto> page = new Page<>(form.getPageNum(), form.getPageSize());
        this.vehicleDriverService.selectDriverVehicle4Params(page, form);

        // 4:数据转换
        List<DriverVehicleDto> dtoList = page.getRecords();
        List<CarrierDriverBindVehicleDetailVO> targetRecordList = new ArrayList<>();
        for (DriverVehicleDto dto : dtoList) {
            CarrierDriverBindVehicleDetailVO vo = new CarrierDriverBindVehicleDetailVO();
            BeanUtils.copyProperties(dto, vo);
            if (CarrierConstants.CARRIER_TYPE_COMPANY.equals(dto.getCarrierType())) {
                vo.setCarrierName(dto.getCarrierFullName());
            } else {
                vo.setCarrierName(dto.getCarrierContactor());
            }

            targetRecordList.add(vo);
        }

        return new PageResultVO(page, targetRecordList);
    }

    @Override
    public void driverDefaultVehicle(DriverVehicleForm form) {
        // 1:校验司机与车辆是否绑定
        VehicleDriverEntity vehicleDriverEntity = this.vehicleDriverService.getBaseMapper().selectOne(new LambdaQueryWrapper<VehicleDriverEntity>()
                .eq(VehicleDriverEntity::getDriverId, form.getDriverId())
                .eq(VehicleDriverEntity::getVehicleId, form.getVehicleId()));
        if (null == vehicleDriverEntity) {



            throw new EException(1001, errorMessageComponent.getCarrierDriverVehicleSetDefault1001());
        }

        // 2:判断是否已默认：0-不是、1-是
        if (vehicleDriverEntity.getSelected()) {
            return;
        }

        // 3:设置司机所有绑定车辆为非默认
        VehicleDriverEntity updateEntity = new VehicleDriverEntity();
        updateEntity.setSelected(false);
        this.vehicleDriverService.getBaseMapper().update(updateEntity,
                new LambdaQueryWrapper<VehicleDriverEntity>()
                        .eq(VehicleDriverEntity::getDriverId, form.getDriverId()));

        // 4:设置司机绑定车辆为默认
        updateEntity = new VehicleDriverEntity();
        updateEntity.setSelected(true);
        this.vehicleDriverService.getBaseMapper().update(updateEntity,
                new LambdaQueryWrapper<VehicleDriverEntity>()
                        .eq(VehicleDriverEntity::getDriverId, form.getDriverId())
                        .eq(VehicleDriverEntity::getVehicleId, form.getVehicleId()));
    }
}
