package com.esv.freight.customer.common.unionpay;

import com.esv.freight.customer.common.exception.EException;
import com.esv.freight.customer.common.pojo.AppUnionPayOrderReq;
import com.esv.freight.customer.common.pojo.AppUnionPayOrderRes;
import com.esv.freight.customer.common.response.ECode;
import com.esv.freight.customer.common.util.DateUtils;
import com.esv.freight.customer.common.util.MapUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.Map;

/**
 * @description: 银联开放平台组件
 * @project: freight-customer-service
 * @name: com.esv.freight.customer.common.unionpay.UnionPayComponent
 * @author: 黄朝斌
 * @email: huangchaobin@esvtek.com
 * @createTime: 2020/05/29 9:29
 * @version:1.0
 */
@Slf4j
@Component
public class UnionPayComponent {

    /**
     * description 创建手机APP支付订单
     * param [appUnionPayOrder]
     * return AppUnionPayOrderRes
     * author HuangChaobin
     * createTime 2020/05/30 13:51
     **/
    public AppUnionPayOrderRes createAppUnionPayOrder(AppUnionPayOrderReq appUnionPayOrderReq) throws EException {
        Map<String, String> contentData = new HashMap<>(128);

        /***银联全渠道系统，产品参数，除了encoding自行选择外其他不需修改***/
        //版本号 全渠道默认值
        contentData.put("version", DemoBase.version);
        //字符集编码 可以使用UTF-8,GBK两种方式
        contentData.put("encoding", DemoBase.encoding);
        //签名方法
        contentData.put("signMethod", SDKConfig.getConfig().getSignMethod());
        //交易类型 01:消费
        contentData.put("txnType", "01");
        //交易子类 01：消费
        contentData.put("txnSubType", "01");
        //填写000201
        contentData.put("bizType", "000201");
        //渠道类型 08手机
        contentData.put("channelType", "08");

        /***商户接入参数***/
        //商户号码，请改成自己申请的商户号或者open上注册得来的777商户号测试
        contentData.put("merId", appUnionPayOrderReq.getMerId());
        //接入类型，商户接入填0 ，不需修改（0：直连商户， 1： 收单机构 2：平台商户）
        contentData.put("accessType", "0");
        //商户订单号，8-40位数字字母，不能含“-”或“_”，可以自行定制规则
        contentData.put("orderId", appUnionPayOrderReq.getBillId());
        //订单描述
        contentData.put("orderDesc", appUnionPayOrderReq.getBillDesc());
        //订单发送时间，取系统时间，格式为yyyyMMddHHmmss，必须取当前时间，否则会报txnTime无效
        contentData.put("txnTime", appUnionPayOrderReq.getTxnTime());
        //账号类型 01：银行卡02：存折03：IC卡帐号类型(卡介质)
        contentData.put("accType", "01");
        //交易金额 单位为分，不能带小数点
        contentData.put("txnAmt", String.valueOf(appUnionPayOrderReq.getTxnAmt()));
        //境内商户固定 156 人民币
        contentData.put("currencyCode", "156");

        // 请求方保留域，
        // 透传字段，查询、通知、对账文件中均会原样出现，如有需要请启用并修改自己希望透传的数据。
        // 出现部分特殊字符时可能影响解析，请按下面建议的方式填写：
        // 1. 如果能确定内容不会出现&={}[]"'等符号时，可以直接填写数据，建议的方法如下。
//		contentData.put("reqReserved", "透传信息1|透传信息2|透传信息3");
        // 2. 内容可能出现&={}[]"'符号时：
        // 1) 如果需要对账文件里能显示，可将字符替换成全角＆＝｛｝【】“‘字符（自己写代码，此处不演示）；
        // 2) 如果对账文件没有显示要求，可做一下base64（如下）。
        //    注意控制数据长度，实际传输的数据长度不能超过1024位。
        //    查询、通知等接口解析时使用new String(Base64.decodeBase64(reqReserved), DemoBase.encoding);解base64后再对数据做后续解析。
//		contentData.put("reqReserved", Base64.encodeBase64String("任意格式的信息都可以".toString().getBytes(DemoBase.encoding)));

        //后台通知地址（需设置为外网能访问 http https均可），支付成功后银联会自动将异步通知报文post到商户上送的该地址，【支付失败的交易银联不会发送后台通知】
        //后台通知参数详见open.unionpay.com帮助中心 下载  产品接口规范  网关支付产品接口规范 消费交易 商户通知
        //注意:1.需设置为外网能访问，否则收不到通知
        // 2.http https均可
        // 3.收单后台通知后需要10秒内返回http200或302状态码
        // 4.如果银联通知服务器发送通知后10秒内未收到返回状态码或者应答码非http200或302，那么银联会间隔一段时间再次发送。总共发送5次，银联后续间隔1、2、4、5 分钟后会再次通知。
        // 5.后台通知地址如果上送了带有？的参数，例如：http://abc/web?a=b&c=d 在后台通知处理程序验证签名之前需要编写逻辑将这些字段去掉再验签，否则将会验签失败
        contentData.put("backUrl", DemoBase.backUrl);

        // 前台返回商户结果时使用，前台类交易需上送
        contentData.put("frontUrl", DemoBase.frontUrl);

        /**对请求参数进行签名并发送http post请求，接收同步应答报文**/
        //报文中certId,signature的值是在signData方法中获取并自动赋值的，只要证书配置正确即可。
        Map<String, String> reqData = AcpService.sign(contentData, DemoBase.encoding);
        String reqDataJsonStr = MapUtils.map2Json(reqData).toJSONString();
        log.info("[银联开放平台]请求银联参数：{}", reqDataJsonStr);

        //交易请求url从配置文件读取对应属性文件acp_sdk.properties中的 acpsdk.backTransUrl
        String requestAppUrl = SDKConfig.getConfig().getAppRequestUrl();
        //发送请求报文并接受同步应答（默认连接超时时间30秒，读取返回结果超时时间30秒）;这里调用signData之后，调用submitUrl之前不能对submitFromData中的键值对做任何修改，如果修改会导致验签不通过
        Map<String, String> rspData = AcpService.post(reqData, requestAppUrl, DemoBase.encoding);
        String resDataJsonStr = MapUtils.map2Json(rspData).toJSONString();
        log.info("[银联开放平台]银联响应参数：{}", resDataJsonStr);

        /**对应答码的处理，请根据您的业务逻辑来编写程序,以下应答码处理逻辑仅供参考------------->**/
        //应答码规范参考open.unionpay.com帮助中心 下载  产品接口规范  《平台接入接口规范-第5部分-附录》
        String tn;
        if (!rspData.isEmpty()) {
            if (AcpService.validate(rspData, DemoBase.encoding)) {
                log.info("[银联开放平台]验证签名成功");
                String respCode = rspData.get("respCode");
                if (("00").equals(respCode)) {
                    //成功,获取tn号
                    tn = rspData.get("tn");
                } else {
                    //其他应答码为失败请排查原因或做失败处理
                    throw new EException(ECode.THIRD_PARTY_ERROR.code(), "未定义异常");
                }
            } else {
                log.error("验证签名失败");
                throw new EException(ECode.THIRD_PARTY_ERROR.code(), "创建银联受理订单号失败");
            }
        } else {
            //未返回正确的http状态
            log.error("[银联开放平台]未获取到返回报文或返回http状态码非200");
            throw new EException(ECode.THIRD_PARTY_ERROR.code(), "[银联开放平台]创建银联受理订单号失败");
        }

        AppUnionPayOrderRes appUnionPayOrderRes = new AppUnionPayOrderRes();
        appUnionPayOrderRes.setReqMessage(reqDataJsonStr);
        appUnionPayOrderRes.setRspMessage(resDataJsonStr);
        appUnionPayOrderRes.setTn(tn);
        return appUnionPayOrderRes;
    }

    /**
     * description 查询交易
     * param [appUnionPayOrderReq]
     * return java.util.Map<java.lang.String,java.lang.String>
     * author HuangChaobin
     * createTime 2020/06/03 10:44
     **/
    public Map<String, String> queryUnionPayOrder(AppUnionPayOrderReq appUnionPayOrderReq) throws EException {
        Map<String, String> data = new HashMap<>(32);

        /***银联全渠道系统，产品参数，除了encoding自行选择外其他不需修改***/
        //版本号 全渠道默认值
        data.put("version", DemoBase.version);
        //字符集编码 可以使用UTF-8,GBK两种方式
        data.put("encoding", DemoBase.encoding);
        //签名方法
        data.put("signMethod", SDKConfig.getConfig().getSignMethod());
        //交易类型 00-默认
        data.put("txnType", "00");
        //交易子类型  默认00
        data.put("txnSubType", "00");
        //业务类型
        data.put("bizType", "000201");

        /***商户接入参数***/
        //商户号码，请改成自己申请的商户号或者open上注册得来的777商户号测试
        data.put("merId", appUnionPayOrderReq.getMerId());
        //接入类型，商户接入固定填0，不需修改
        data.put("accessType", "0");

        /***要调通交易以下字段必须修改***/
        //****商户订单号，每次发交易测试需修改为被查询的交易的订单号
        data.put("orderId", appUnionPayOrderReq.getBillId());
        //****订单发送时间，每次发交易测试需修改为被查询的交易的订单发送时间
        data.put("txnTime", appUnionPayOrderReq.getTxnTime());

        /**请求参数设置完毕，以下对请求参数进行签名并发送http post请求，接收同步应答报文------------->**/
        //报文中certId,signature的值是在signData方法中获取并自动赋值的，只要证书配置正确即可。
        Map<String, String> reqData = AcpService.sign(data, DemoBase.encoding);
        //交易请求url从配置文件读取对应属性文件acp_sdk.properties中的 acpsdk.singleQueryUrl
        String url = SDKConfig.getConfig().getSingleQueryUrl();
        //发送请求报文并接受同步应答（默认连接超时时间30秒，读取返回结果超时时间30秒）;这里调用signData之后，调用submitUrl之前不能对submitFromData中的键值对做任何修改，如果修改会导致验签不通过
        Map<String, String> rspData = AcpService.post(reqData, url, DemoBase.encoding);

        /**对应答码的处理，请根据您的业务逻辑来编写程序,以下应答码处理逻辑仅供参考------------->**/
        //应答码规范参考open.unionpay.com帮助中心 下载  产品接口规范  《平台接入接口规范-第5部分-附录》
        if (!rspData.isEmpty()) {
            if (AcpService.validate(rspData, DemoBase.encoding)) {
                log.info("[银联开放平台]验证签名成功");
            } else {
                log.error("[银联开放平台]验证签名失败");
                throw new EException(ECode.THIRD_PARTY_ERROR.code(), "[银联开放平台]验证签名失败");
            }
        } else {
            //未返回正确的http状态
            log.error("[银联开放平台]未获取到返回报文或返回http状态码非200");
            throw new EException(ECode.THIRD_PARTY_ERROR.code(), "[银联开放平台]银联响应异常");
        }

        return rspData;
    }

}
