package com.esv.freight.customer.common.filter;

import com.alibaba.fastjson.JSONObject;
import com.esv.freight.customer.common.constants.CommonConstants;
import com.esv.freight.customer.common.util.ReqUtils;
import com.esv.freight.customer.common.wrapper.RestRequestWrapper;
import com.esv.freight.customer.common.wrapper.RestResponseWrapper;
import lombok.extern.slf4j.Slf4j;

import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.Enumeration;

/**
 * @description: Rest请求日志Filter
 * @project: SpringCloudTemplate
 * @name: com.esv.freight.customer.filter.RestLogFilter
 * @author: 黄朝斌
 * @email: huangchaobin@esvtek.com
 * @createTime: 2020/03/09 17:44
 * @version:1.0
 */
@Slf4j
public class RestLogFilter implements Filter {

    @Override
    public void init(FilterConfig filterConfig) {

    }

    @Override
    public void destroy() {

    }

    @Override
    public void doFilter(ServletRequest servletRequest, ServletResponse servletResponse, FilterChain filterChain) throws IOException, ServletException {
        RestRequestWrapper requestWrapper = new RestRequestWrapper((HttpServletRequest)servletRequest);
        RestResponseWrapper responseWrapper = new RestResponseWrapper((HttpServletResponse) servletResponse);

        // 日志输出请求体
        this.logReq(requestWrapper);

        // 日志输出请求头
        this.logReqHeader(requestWrapper);

        filterChain.doFilter(requestWrapper, responseWrapper);

        // 日志输出返回体
        this.logRes(requestWrapper, responseWrapper);
    }

    /**
     * 日志输出请求体
     **/
    private void logReq(RestRequestWrapper requestWrapper) {
        String url = requestWrapper.getRequestURI();
        String method = requestWrapper.getMethod();
        String reqBody = "";

        // 获取get、post请求体
        if (CommonConstants.HTTP_REQUEST_METHOD_GET.equalsIgnoreCase(method)) {
            Enumeration em = requestWrapper.getParameterNames();
            while (em.hasMoreElements()) {
                String k = em.nextElement().toString();
                String v = requestWrapper.getParameter(k);
                reqBody += "&" + k + "=" + v;
            }
            reqBody = reqBody.replaceFirst("&", "");
        } else if (CommonConstants.HTTP_REQUEST_METHOD_POST.equalsIgnoreCase(method)) {
            reqBody = ReqUtils.getPostBody(requestWrapper);
        }

        // 请求体日志截取
        if (CommonConstants.LOG_MAX_LENGTH < reqBody.length()) {
            reqBody = reqBody.substring(0, CommonConstants.LOG_CUT_LENGTH) + "……";
        }

        // 日志输出请求体
        if (url.startsWith(CommonConstants.SPRING_BOOT_ADMIN_PREFIX_URL)) {
            log.debug("[IP={}]收到{}请求，url：{}，params：{}", ReqUtils.getHttpClientIp(requestWrapper), method, url, reqBody);
        } else {
            log.info("[IP={}]收到{}请求，url：{}，body：{}", ReqUtils.getHttpClientIp(requestWrapper), method, url, reqBody);
        }
    }

    /**
     * 日志输出请求头
     **/
    private void logReqHeader(RestRequestWrapper request) {
        JSONObject headerJson = new JSONObject();
        Enumeration headerNames = request.getHeaderNames();
        while (headerNames.hasMoreElements()) {
            String key = (String) headerNames.nextElement();
            headerJson.put(key, request.getHeader(key));
        }
        log.info("请求头：{}", headerJson.toJSONString());
    }

    /**
     * 日志输出返回体
     **/
    private void logRes(RestRequestWrapper requestWrapper, RestResponseWrapper responseWrapper) {
        byte[] bytes = responseWrapper.getBody();
        String resBody = null;
        try {
            resBody = new String(bytes, CommonConstants.DEFAULT_CHARACTER_ENCODING);
        } catch (UnsupportedEncodingException e) {
            log.error(e.getMessage(), e);
        }
        String url = requestWrapper.getRequestURI();
        if (CommonConstants.LOG_MAX_LENGTH < resBody.length()) {
            resBody = resBody.substring(0, CommonConstants.LOG_CUT_LENGTH) + "……";
        }
        if (url.startsWith(CommonConstants.SPRING_BOOT_ADMIN_PREFIX_URL)) {
            log.debug("请求响应：{}", resBody);
        } else {
            log.info("请求响应：{}", resBody);
        }
    }
}