package com.esv.freight.customer.module.driver.controller;

import com.alibaba.fastjson.JSONObject;
import com.esv.freight.customer.common.component.ErrorMessageComponent;
import com.esv.freight.customer.common.component.PasswordComponent;
import com.esv.freight.customer.common.exception.EException;
import com.esv.freight.customer.common.response.ECode;
import com.esv.freight.customer.common.response.EResponse;
import com.esv.freight.customer.common.util.ReqUtils;
import com.esv.freight.customer.common.util.VerifyUtils;
import com.esv.freight.customer.common.validator.groups.ValidatorDetail;
import com.esv.freight.customer.common.validator.groups.ValidatorInsert;
import com.esv.freight.customer.common.validator.groups.ValidatorList;
import com.esv.freight.customer.common.validator.groups.ValidatorUpdate;
import com.esv.freight.customer.module.carrier.CarrierConstants;
import com.esv.freight.customer.module.driver.DriverConstants;
import com.esv.freight.customer.module.driver.dto.DriverDetailDto;
import com.esv.freight.customer.module.driver.entity.DriverAuditHistoryEntity;
import com.esv.freight.customer.module.driver.form.AccountScoreForm;
import com.esv.freight.customer.module.driver.form.DriverAuditForm;
import com.esv.freight.customer.module.driver.form.DriverInfoForm;
import com.esv.freight.customer.module.driver.form.DriverQueryForm;
import com.esv.freight.customer.module.driver.service.DriverAccountService;
import com.esv.freight.customer.module.driver.service.DriverInfoService;
import com.esv.freight.customer.module.driver.validator.groups.ValidatorDetailAccount;
import com.esv.freight.customer.module.driver.validator.groups.ValidatorLogin;
import com.esv.freight.customer.module.driver.validator.groups.ValidatorPasswordReset;
import com.esv.freight.customer.module.driver.validator.groups.ValidatorRegister;
import com.esv.freight.customer.module.driver.vo.DriverAuditHistoryVO;
import com.esv.freight.customer.module.driver.vo.DriverDetailVO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.ArrayList;
import java.util.List;

/**
 * @description:
 * @project: freight-customer-service
 * @name: com.esv.freight.customer.module.driver.controller.DriverController
 * @author: 黄朝斌
 * @email: huangchaobin@esvtek.com
 * @createTime: 2020/04/28 17:18
 * @version:1.0
 */
@Slf4j
@RestController
@RequestMapping("/carrier/driver")
@Validated
public class DriverController {

    private ErrorMessageComponent errorMessageComponent;
    private PasswordComponent passwordComponent;
    private DriverAccountService driverAccountService;
    private DriverInfoService driverInfoService;

    @Autowired
    public DriverController(ErrorMessageComponent errorMessageComponent, PasswordComponent passwordComponent,
                            DriverAccountService driverAccountService, DriverInfoService driverInfoService) {
        this.errorMessageComponent = errorMessageComponent;
        this.passwordComponent = passwordComponent;
        this.driverAccountService = driverAccountService;
        this.driverInfoService = driverInfoService;
    }

    /**
     * description 新增司机
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/28 17:24
     **/
    @PostMapping("/add")
    public EResponse add(@RequestBody @Validated(ValidatorInsert.class) DriverInfoForm form) throws EException {
        /****************************** 参数校验 ******************************/
        // 所驾驶车辆4.5吨以上必填道路运输从业资格证信息
        if (DriverConstants.ACCOUNT_DRIVING_VEHICLE_TYPE_2.equals(form.getCertificateVehicle())) {
            String[] notBlankParams = new String[] {"certificateEndDate", "certificateNumber", "certificateUrl"};
            ReqUtils.checkParamsNotBlank(JSONObject.parseObject(form.toString()), notBlankParams);
        }
        // 校验身份证号码
        if (StringUtils.isNotBlank(form.getIdCard()) && !VerifyUtils.isValidIdCard18(form.getIdCard())) {
            throw new EException(ECode.PARAM_ERROR.code(), "无效的身份证号码");
        }
        /****************************** 参数校验 ******************************/

        Long id = driverAccountService.insertDriver(form);
        JSONObject data = new JSONObject();
        data.put("id", id);
        return EResponse.ok(data);
    }

    /**
     * description 编辑司机信息
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/28 19:26
     **/
    @PostMapping("/edit")
    public EResponse edit(@RequestBody @Validated(ValidatorUpdate.class) DriverInfoForm form) throws EException {
        /****************************** 参数校验 ******************************/
        // 所驾驶车辆4.5吨以上必填道路运输从业资格证信息
        if (DriverConstants.ACCOUNT_DRIVING_VEHICLE_TYPE_2.equals(form.getCertificateVehicle())) {
            String[] notBlankParams = new String[] {"certificateEndDate", "certificateNumber", "certificateUrl"};
            ReqUtils.checkParamsNotBlank(JSONObject.parseObject(form.toString()), notBlankParams);
        }
        // 校验身份证号码
        if (StringUtils.isNotBlank(form.getIdCard()) && !VerifyUtils.isValidIdCard18(form.getIdCard())) {
            throw new EException(ECode.PARAM_ERROR.code(), "无效的身份证号码");
        }
        /****************************** 参数校验 ******************************/

        driverAccountService.updateDriver(form);
        return EResponse.ok();
    }

    /**
     * description 查询司机详情-通过ID
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/28 20:26
     **/
    @PostMapping("/getDetailById")
    public EResponse getDetailById(@RequestBody @Validated(ValidatorDetail.class) DriverQueryForm form) throws EException {
        DriverDetailDto dto = driverAccountService.getDriverDetailById(form.getId());
        if (null == dto) {
            throw new EException(1001, errorMessageComponent.getCarrierDriverDetail1001());
        }

        // 数据转换
        DriverDetailVO vo = new DriverDetailVO();
        BeanUtils.copyProperties(dto, vo);
        vo.setCreateTime(dto.getCreateTime().getTime());
        if (CarrierConstants.CARRIER_TYPE_COMPANY.equals(dto.getCarrierType())) {
            vo.setCarrierName(dto.getCarrierFullName());
        } else {
            vo.setCarrierName(dto.getCarrierContactor());
        }

        return EResponse.ok(vo);
    }

    /**
     * description 查询司机详情-通过帐号
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/28 20:26
     **/
    @PostMapping("/getDetailByAccount")
    public EResponse getDetailByAccount(@RequestBody @Validated(ValidatorDetailAccount.class) DriverQueryForm form) throws EException {
        DriverDetailDto dto = driverAccountService.getDriverDetailByAccount(form.getAccount());
        if (null == dto) {
            throw new EException(1001, errorMessageComponent.getCarrierDriverDetailByAccount1001());
        }

        // 数据转换
        DriverDetailVO vo = new DriverDetailVO();
        BeanUtils.copyProperties(dto, vo);
        vo.setCreateTime(dto.getCreateTime().getTime());
        if (CarrierConstants.CARRIER_TYPE_COMPANY.equals(dto.getCarrierType())) {
            vo.setCarrierName(dto.getCarrierFullName());
        } else {
            vo.setCarrierName(dto.getCarrierContactor());
        }

        return EResponse.ok(vo);
    }

    /**
     * description 审核司机信息
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/29 9:38
     **/
    @PostMapping("/audit")
    public EResponse audit(@RequestBody @Validated DriverAuditForm form) throws EException {
        if (DriverConstants.ACCOUNT_AUDIT_STATUS_FAILURE.equals(form.getAuditStatus())) {
            String[] notBlankParams = new String[] {"remark"};
            ReqUtils.checkParamsNotBlank(JSONObject.parseObject(form.toString()), notBlankParams);
        }
        driverAccountService.auditDriver(form);

        return EResponse.ok();
    }

    /**
     * description 停用帐号
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/29 10:10
     **/
    @PostMapping("/block")
    public EResponse block(@RequestBody @Validated(ValidatorDetail.class) DriverQueryForm form) throws EException {
        driverAccountService.blockDriver(form.getId());

        return EResponse.ok();
    }

    /**
     * description 停用帐号
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/29 10:10
     **/
    @PostMapping("/unblock")
    public EResponse unblock(@RequestBody @Validated(ValidatorDetail.class) DriverQueryForm form) throws EException {
        driverAccountService.unblockDriver(form.getId());

        return EResponse.ok();
    }

    /**
     * description 注册司机
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/29 11:03
     **/
    @PostMapping("/register")
    public EResponse register(@RequestBody @Validated(ValidatorRegister.class) DriverInfoForm form) throws EException {
        Long id = driverAccountService.registerDriver(form);
        JSONObject data = new JSONObject();
        data.put("id", id);
        
        return EResponse.ok(data);
    }

    /**
     * description 查看帐号的审核历史
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/29 11:30
     **/
    @PostMapping("/auditHistory")
    public EResponse auditHistory(@RequestBody @Validated(ValidatorDetail.class) DriverQueryForm form) throws EException {
        List<DriverAuditHistoryEntity> entityList = driverAccountService.getAuditHistory(form.getId());
        
        // 数据转换
        List<DriverAuditHistoryVO> voList = new ArrayList<>();
        entityList.forEach(entity -> {
            DriverAuditHistoryVO vo = new DriverAuditHistoryVO();
            BeanUtils.copyProperties(entity, vo);
            vo.setOperateTime(entity.getOperateTime().getTime());
            voList.add(vo);
        });
        
        return EResponse.ok(voList);
    }

    /**
     * description 分页查询司机列表
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/29 14:13
     **/
    @PostMapping("/list")
    public EResponse list(@RequestBody @Validated(ValidatorList.class) DriverQueryForm form) throws EException {
           return EResponse.ok(driverAccountService.getDriver4Page(form));
    }

    /**
     * description 帐号密码校验
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/30 11:14
     **/
    @PostMapping("/account/check")
    public EResponse accountCheck(@RequestBody @Validated(ValidatorLogin.class) DriverInfoForm form) throws EException {
        // 查询司机详情-通过帐号
        DriverDetailDto dto = driverAccountService.getDriverDetailByAccount(form.getAccount());

        // 校验
        if (null == dto) {
            throw new EException(1001, errorMessageComponent.getCarrierDriverAccountCheck1001());
        }
        String password = passwordComponent.generatePwd4Salt(form.getPassword(), dto.getSalt());
        if (!password.equals(dto.getPassword())) {
            throw new EException(1002, errorMessageComponent.getCarrierDriverAccountCheck1002());
        }

        // 数据转换
        DriverDetailVO vo = new DriverDetailVO();
        BeanUtils.copyProperties(dto, vo);
        vo.setCreateTime(dto.getCreateTime().getTime());

        return EResponse.ok(vo);
    }

    /**
     * description 重置司机帐号密码
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/05/16 16:51
     **/
    @PostMapping("/account/password/reset")
    public EResponse resetDriverPwd(@RequestBody @Validated(ValidatorPasswordReset.class) DriverInfoForm form) throws EException {
        driverAccountService.resetDriverPwd(form);
        return EResponse.ok();
    }

    /**
     * description 更新司机整体评分
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author HuangChaobin
     * createTime 2020/06/01 13:32
     **/
    @PostMapping("/score/overall/update")
    public EResponse updateOverallScore(@RequestBody @Validated(ValidatorUpdate.class) AccountScoreForm form) throws EException {
        driverInfoService.updateOverallScore(form);
        return EResponse.ok();
    }
}
