package com.esv.freight.customer.module.carrier.controller;

import com.alibaba.fastjson.JSONObject;
import com.esv.freight.customer.common.exception.EException;
import com.esv.freight.customer.common.response.EResponse;
import com.esv.freight.customer.common.util.ReqUtils;
import com.esv.freight.customer.common.validator.groups.ValidatorDetail;
import com.esv.freight.customer.common.validator.groups.ValidatorInsert;
import com.esv.freight.customer.common.validator.groups.ValidatorList;
import com.esv.freight.customer.common.validator.groups.ValidatorUpdate;
import com.esv.freight.customer.module.carrier.CarrierConstants;
import com.esv.freight.customer.module.carrier.dto.CarrierInfoDto;
import com.esv.freight.customer.module.carrier.form.CarrierInfoForm;
import com.esv.freight.customer.module.carrier.form.CarrierQueryForm;
import com.esv.freight.customer.module.carrier.service.CarrierAccountService;
import com.esv.freight.customer.module.carrier.vo.CarrierInfoBriefVO;
import com.esv.freight.customer.module.carrier.vo.CarrierInfoDetailVO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.ArrayList;
import java.util.List;

/**
 * @description: 承运商帐号Controller
 * @project: freight-customer-service
 * @name: com.esv.freight.customer.module.carrier.controller.AccountController
 * @author: 黄朝斌
 * @email: huangchaobin@esvtek.com
 * @createTime: 2020/04/23 20:21
 * @version:1.0
 */
@Slf4j
@RestController
@RequestMapping("/carrier/account")
@Validated
public class CarrierAccountController {

    private CarrierAccountService carrierAccountService;

    public CarrierAccountController(CarrierAccountService carrierAccountService) {
        this.carrierAccountService = carrierAccountService;
    }

    /**
     * description 新增承运商帐号
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/24 14:49
     **/
    @PostMapping("/add")
    public EResponse createAccountByManage(@RequestBody @Validated(ValidatorInsert.class) CarrierInfoForm form) throws EException {
        /****************************** 参数校验 ******************************/
        // 承运商为企业
        if (CarrierConstants.CARRIER_TYPE_COMPANY.equals(form.getCarrierType())) {
            String[] notBlankParams = new String[] {"uniCreditCode", "carrierFullName", "provinceCode", "cityCode", "districtCode", "detailAddress"
                , "legalPerson", "businessLicenseUrl"};
            ReqUtils.checkParamsNotBlank(JSONObject.parseObject(form.toString()), notBlankParams);
        }
        /****************************** 参数校验 ******************************/

        Long id = carrierAccountService.insertCarrier(form);
        JSONObject data = new JSONObject();
        data.put("id", id);
        return EResponse.ok(data);
    }

    /**
     * description 编辑承运商信息
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/24 14:50
     **/
    @PostMapping("/edit")
    public EResponse editAccountByManage(@RequestBody @Validated(ValidatorUpdate.class) CarrierInfoForm form) throws EException {
        /****************************** 参数校验 ******************************/
        // 承运商为企业
        if (CarrierConstants.CARRIER_TYPE_COMPANY.equals(form.getCarrierType())) {
            String[] notBlankParams = new String[] {"uniCreditCode", "carrierFullName", "provinceCode", "cityCode", "districtCode", "detailAddress"
                    , "legalPerson", "businessLicenseUrl"};
            ReqUtils.checkParamsNotBlank(JSONObject.parseObject(form.toString()), notBlankParams);
        }
        /****************************** 参数校验 ******************************/

        return EResponse.ok(carrierAccountService.updateCarrier(form));
    }

    /**
     * description 查询承运商详情
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/24 17:21
     **/
    @PostMapping("/detail")
    public EResponse detail(@RequestBody @Validated(ValidatorDetail.class) CarrierQueryForm form) throws EException {
        // 查询
        CarrierInfoDto dto = carrierAccountService.getCarrierDetail(form);
        
        // 数据转换
        CarrierInfoDetailVO vo = new CarrierInfoDetailVO();
        BeanUtils.copyProperties(dto, vo);
        vo.setCreateTime(dto.getCreateTime().getTime());

        return EResponse.ok(vo);
    }

    /**
     * description 查询所有承运商列表
     * param []
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/24 17:44
     **/
    @PostMapping("/all")
    public EResponse all() throws EException {
        // 查询
        List<CarrierInfoDto> dtoList = carrierAccountService.getAllCarrierBrief();

        // 数据转换
        List<CarrierInfoBriefVO> voList = new ArrayList<>();
        dtoList.forEach(dto -> {
            CarrierInfoBriefVO vo = new CarrierInfoBriefVO();
            BeanUtils.copyProperties(dto, vo);
            if (CarrierConstants.CARRIER_TYPE_COMPANY.equals(dto.getCarrierType())) {
                vo.setCarrierName(dto.getCarrierFullName());
            } else {
                vo.setCarrierName(dto.getContactor());
            }
            voList.add(vo);
        });

        return EResponse.ok(voList);
    }

    /**
     * description 分页查询帐号列表
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/26 9:34
     **/
    @PostMapping("/list")
    public EResponse list(@RequestBody @Validated(ValidatorList.class) CarrierQueryForm form) throws EException {
        return EResponse.ok(carrierAccountService.getCarrier4Page(form));
    }
}
