package com.esv.freight.customer.module.driver.controller;

import com.alibaba.fastjson.JSONObject;
import com.esv.freight.customer.common.component.ErrorMessageComponent;
import com.esv.freight.customer.common.exception.EException;
import com.esv.freight.customer.common.response.ECode;
import com.esv.freight.customer.common.response.EResponse;
import com.esv.freight.customer.common.util.ReqUtils;
import com.esv.freight.customer.common.util.VerifyUtils;
import com.esv.freight.customer.common.validator.groups.ValidatorDetail;
import com.esv.freight.customer.common.validator.groups.ValidatorInsert;
import com.esv.freight.customer.common.validator.groups.ValidatorUpdate;
import com.esv.freight.customer.module.driver.DriverConstants;
import com.esv.freight.customer.module.driver.dto.DriverDetailDto;
import com.esv.freight.customer.module.driver.form.DriverAuditForm;
import com.esv.freight.customer.module.driver.form.DriverInfoForm;
import com.esv.freight.customer.module.driver.form.DriverQueryForm;
import com.esv.freight.customer.module.driver.service.DriverAccountService;
import com.esv.freight.customer.module.driver.vo.DriverDetailVO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

/**
 * @description:
 * @project: freight-customer-service
 * @name: com.esv.freight.customer.module.driver.controller.DriverController
 * @author: 黄朝斌
 * @email: huangchaobin@esvtek.com
 * @createTime: 2020/04/28 17:18
 * @version:1.0
 */
@Slf4j
@RestController
@RequestMapping("/carrier/driver")
@Validated
public class DriverController {

    private ErrorMessageComponent errorMessageComponent;

    private DriverAccountService driverAccountService;

    @Autowired
    public DriverController(ErrorMessageComponent errorMessageComponent, DriverAccountService driverAccountService) {
        this.errorMessageComponent = errorMessageComponent;
        this.driverAccountService = driverAccountService;
    }

    /**
     * description 新增司机
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/28 17:24
     **/
    @PostMapping("/add")
    public EResponse add(@RequestBody @Validated(ValidatorInsert.class) DriverInfoForm form) throws EException {
        /****************************** 参数校验 ******************************/
        // 所驾驶车辆4.5吨以上必填道路运输从业资格证信息
        if (DriverConstants.ACCOUNT_DRIVING_VEHICLE_TYPE_2.equals(form.getCertificateVehicle())) {
            String[] notBlankParams = new String[] {"certificateEndDate", "certificateNumber", "certificateUrl"};
            ReqUtils.checkParamsNotBlank(JSONObject.parseObject(form.toString()), notBlankParams);
        }
        // 校验身份证号码
        if (StringUtils.isNotBlank(form.getIdCard()) && !VerifyUtils.isValidIdCard18(form.getIdCard())) {
            throw new EException(ECode.PARAM_ERROR.code(), "无效的身份证号码");
        }
        /****************************** 参数校验 ******************************/

        Long id = driverAccountService.insertDriver(form);
        JSONObject data = new JSONObject();
        data.put("id", id);
        return EResponse.ok(data);
    }

    /**
     * description 编辑司机信息
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/28 19:26
     **/
    @PostMapping("/edit")
    public EResponse edit(@RequestBody @Validated(ValidatorUpdate.class) DriverInfoForm form) throws EException {
        /****************************** 参数校验 ******************************/
        // 所驾驶车辆4.5吨以上必填道路运输从业资格证信息
        if (DriverConstants.ACCOUNT_DRIVING_VEHICLE_TYPE_2.equals(form.getCertificateVehicle())) {
            String[] notBlankParams = new String[] {"certificateEndDate", "certificateNumber", "certificateUrl"};
            ReqUtils.checkParamsNotBlank(JSONObject.parseObject(form.toString()), notBlankParams);
        }
        // 校验身份证号码
        if (StringUtils.isNotBlank(form.getIdCard()) && !VerifyUtils.isValidIdCard18(form.getIdCard())) {
            throw new EException(ECode.PARAM_ERROR.code(), "无效的身份证号码");
        }
        /****************************** 参数校验 ******************************/

        driverAccountService.updateDriver(form);
        return EResponse.ok();
    }

    /**
     * description 查询司机详情
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/28 20:26
     **/
    @PostMapping("/detail")
    public EResponse detail(@RequestBody @Validated(ValidatorDetail.class) DriverQueryForm form) throws EException {
        DriverDetailDto dto = driverAccountService.getDriverDetail(form.getId());
        if (null == dto) {
            throw new EException(1001, errorMessageComponent.getCarrierDriverDetail1001());
        }

        // 数据转换
        DriverDetailVO vo = new DriverDetailVO();
        BeanUtils.copyProperties(dto, vo);
        vo.setCreateTime(dto.getCreateTime().getTime());

        return EResponse.ok(vo);
    }

    /**
     * description 审核司机信息
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/29 9:38
     **/
    @PostMapping("/audit")
    public EResponse audit(@RequestBody @Validated DriverAuditForm form) throws EException {
        if (DriverConstants.ACCOUNT_AUDIT_STATUS_FAILURE.equals(form.getAuditStatus())) {
            String[] notBlankParams = new String[] {"remark"};
            ReqUtils.checkParamsNotBlank(JSONObject.parseObject(form.toString()), notBlankParams);
        }
        driverAccountService.auditDriver(form);

        return EResponse.ok();
    }

    /**
     * description 停用帐号
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/29 10:10
     **/
    @PostMapping("/block")
    public EResponse block(@RequestBody @Validated(ValidatorDetail.class) DriverQueryForm form) throws EException {
        driverAccountService.blockDriver(form.getId());

        return EResponse.ok();
    }

    /**
     * description 停用帐号
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/29 10:10
     **/
    @PostMapping("/unblock")
    public EResponse unblock(@RequestBody @Validated(ValidatorDetail.class) DriverQueryForm form) throws EException {
        driverAccountService.unblockDriver(form.getId());

        return EResponse.ok();
    }
}
