package com.esv.freight.customer.module.goodsowner.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.esv.freight.customer.common.component.ErrorMessageComponent;
import com.esv.freight.customer.common.exception.EException;
import com.esv.freight.customer.common.util.FeignUtils;
import com.esv.freight.customer.feign.FeignBaseService;
import com.esv.freight.customer.module.goodsowner.GoodsOwnerConstants;
import com.esv.freight.customer.module.goodsowner.dao.RegularlyRouteDao;
import com.esv.freight.customer.module.goodsowner.dto.RouteDetailDto;
import com.esv.freight.customer.module.goodsowner.entity.*;
import com.esv.freight.customer.module.goodsowner.form.DeleteAddressForm;
import com.esv.freight.customer.module.goodsowner.form.RegularlyRouteForm;
import com.esv.freight.customer.module.goodsowner.service.AccountService;
import com.esv.freight.customer.module.goodsowner.service.DeliveryAddressService;
import com.esv.freight.customer.module.goodsowner.service.ReceiveAddressService;
import com.esv.freight.customer.module.goodsowner.service.RegularlyRouteService;
import com.esv.freight.customer.module.goodsowner.vo.RouteDetailVO;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Arrays;
import java.util.List;


@Service("regularlyRouteService")
public class RegularlyRouteServiceImpl extends ServiceImpl<RegularlyRouteDao, RegularlyRouteEntity> implements RegularlyRouteService {

    private ErrorMessageComponent errorMessageComponent;

    private FeignBaseService feignBaseService;

    private AccountService accountService;

    private DeliveryAddressService deliveryAddressService;

    private ReceiveAddressService receiveAddressService;

    @Autowired
    public RegularlyRouteServiceImpl(ErrorMessageComponent errorMessageComponent, FeignBaseService feignBaseService,
                                     AccountService accountService, DeliveryAddressService deliveryAddressService, ReceiveAddressService receiveAddressService) {
        this.errorMessageComponent = errorMessageComponent;
        this.feignBaseService = feignBaseService;
        this.accountService = accountService;
        this.deliveryAddressService = deliveryAddressService;
        this.receiveAddressService = receiveAddressService;
    }

    @Override
    public Long add(RegularlyRouteForm form) {
        // 1.校验线路名称是否重复
        List<RegularlyRouteEntity> routeEntityList = this.getOwnerRouteByName(form);
        if (0 < routeEntityList.size()) {
            throw new EException(1001, errorMessageComponent.getGoodsOwnerRegularlyRouteAdd1001());
        }

        // 2.校验获取ID是否有效
        AccountEntity accountEntity = this.accountService.getById(form.getOwnerId());
        if (null == accountEntity) {
            throw new EException(1002, errorMessageComponent.getGoodsOwnerRegularlyRouteAdd1002());
        }

        // 3.校验发货地址ID是否有效
        DeliveryAddressEntity deliveryAddressEntity = this.deliveryAddressService.getById(form.getDeliveryId());
        if (null == deliveryAddressEntity) {
            throw new EException(1003, errorMessageComponent.getGoodsOwnerRegularlyRouteAdd1003());
        }

        // 4.校验收货地址ID是否有效
        ReceiveAddressEntity receiveAddressEntity = this.receiveAddressService.getById(form.getReceiveId());
        if (null == receiveAddressEntity) {
            throw new EException(1004, errorMessageComponent.getGoodsOwnerRegularlyRouteAdd1004());
        }

        // 5.获取路线编码
        JSONObject batchIdReqJson = new JSONObject();
        batchIdReqJson.put("prefix", "LX");
        batchIdReqJson.put("formatter", "yyyyMMdd");
        batchIdReqJson.put("length", 13);
        JSONObject batchIdResJson;
        try {
            batchIdResJson = FeignUtils.getFeignResultData(feignBaseService.getBatchId(batchIdReqJson));
        } catch (Exception e) {
            log.error("调用[基础服务]生成路线编号失败");
            throw new EException("生成路线编号时发生错误");
        }
        String routeNumber = batchIdResJson.getString("batchId");

        // 6.新增
        RegularlyRouteEntity routeEntity = new RegularlyRouteEntity();
        BeanUtils.copyProperties(form, routeEntity);
        routeEntity.setId(null);
        routeEntity.setRouteNumber(routeNumber);
        this.baseMapper.insert(routeEntity);

        return routeEntity.getId();
    }

    @Override
    public Integer edit(RegularlyRouteForm form) {
        // 1.校验线路ID是否有效
        RegularlyRouteEntity routeEntity = this.baseMapper.selectById(form.getId());
        if (null == routeEntity) {
            throw new EException(1001, errorMessageComponent.getGoodsOwnerRegularlyRouteEdit1001());
        }

        // 2.校验线路名称是否重复
        QueryWrapper<RegularlyRouteEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.ne("id", form.getId());
        queryWrapper.eq("owner_id", form.getOwnerId());
        queryWrapper.eq("route_name", form.getRouteName());
        List<RegularlyRouteEntity> routeEntityList = this.baseMapper.selectList(queryWrapper);
        if (0 < routeEntityList.size()) {
            throw new EException(1002, errorMessageComponent.getGoodsOwnerRegularlyRouteEdit1002());
        }

        // 3.校验获取ID是否有效
        AccountEntity accountEntity = this.accountService.getById(form.getOwnerId());
        if (null == accountEntity) {
            throw new EException(1003, errorMessageComponent.getGoodsOwnerRegularlyRouteEdit1003());
        }

        // 4.校验发货地址ID是否有效
        DeliveryAddressEntity deliveryAddressEntity = this.deliveryAddressService.getById(form.getDeliveryId());
        if (null == deliveryAddressEntity) {
            throw new EException(1004, errorMessageComponent.getGoodsOwnerRegularlyRouteEdit1004());
        }

        // 5.校验收货地址ID是否有效
        ReceiveAddressEntity receiveAddressEntity = this.receiveAddressService.getById(form.getReceiveId());
        if (null == receiveAddressEntity) {
            throw new EException(1005, errorMessageComponent.getGoodsOwnerRegularlyRouteEdit1005());
        }

        // 6.更新
        routeEntity = new RegularlyRouteEntity();
        BeanUtils.copyProperties(form, routeEntity);
        int count = this.baseMapper.updateById(routeEntity);

        return count;
    }

    @Override
    public Integer delete(DeleteAddressForm form) {
        String[] ids = form.getId().split(",");
        int count = this.baseMapper.deleteBatchIds(Arrays.asList(ids));
        return count;
    }

    @Override
    public RouteDetailVO getRouteDetail(RegularlyRouteForm form) {
        RouteDetailDto dto = this.baseMapper.selectRouteDetail(form.getId());
        if (null == dto) {
            throw new EException(1001, errorMessageComponent.getGoodsOwnerRegularlyRouteDetail1001());
        }
        RouteDetailVO vo = new RouteDetailVO();
        BeanUtils.copyProperties(dto, vo);
        if (GoodsOwnerConstants.OWNER_TYPE_PERSONAL.equals(dto.getOwnerType())) {
            vo.setOwnerName(dto.getContactor());
        } else {
            vo.setOwnerName(dto.getOwnerFullName());
        }

        return vo;
    }

    @Override
    public List<RegularlyRouteEntity> getOwnerRouteByName(RegularlyRouteForm form) {
        QueryWrapper<RegularlyRouteEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("owner_id", form.getOwnerId());
        queryWrapper.eq("route_name", form.getRouteName());
        return this.baseMapper.selectList(queryWrapper);
    }
}