package com.esv.freight.customer.module.vehicle.controller;

import com.alibaba.fastjson.JSONObject;
import com.esv.freight.customer.common.component.ErrorMessageComponent;
import com.esv.freight.customer.common.exception.EException;
import com.esv.freight.customer.common.response.ECode;
import com.esv.freight.customer.common.response.EResponse;
import com.esv.freight.customer.common.validator.groups.ValidatorDetail;
import com.esv.freight.customer.common.validator.groups.ValidatorInsert;
import com.esv.freight.customer.common.validator.groups.ValidatorUpdate;
import com.esv.freight.customer.module.carrier.service.CarrierInfoService;
import com.esv.freight.customer.module.vehicle.VehicleConstants;
import com.esv.freight.customer.module.vehicle.dto.VehicleDetailDto;
import com.esv.freight.customer.module.vehicle.entity.VehicleAuditHistoryEntity;
import com.esv.freight.customer.module.vehicle.form.VehicleAuditForm;
import com.esv.freight.customer.module.vehicle.form.VehicleInfoForm;
import com.esv.freight.customer.module.vehicle.form.VehicleQueryForm;
import com.esv.freight.customer.module.vehicle.service.VehicleAuditHistoryService;
import com.esv.freight.customer.module.vehicle.service.VehicleService;
import com.esv.freight.customer.module.vehicle.validator.groups.ValidatorAuditHistory;
import com.esv.freight.customer.module.vehicle.vo.VehicleAuditVO;
import com.esv.freight.customer.module.vehicle.vo.VehicleDetailVO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

/**
 * @description: 承运商车辆Controller
 * @project: freight-customer-service
 * @name: com.esv.freight.customer.module.vehicle.controller.VehicleController
 * @author: 黄朝斌
 * @email: huangchaobin@esvtek.com
 * @createTime: 2020/04/26 15:11
 * @version:1.0
 */
@Slf4j
@RestController
@RequestMapping("/carrier/vehicle")
@Validated
public class VehicleController {

    private ErrorMessageComponent errorMessageComponent;

    private VehicleService vehicleService;

    private VehicleAuditHistoryService vehicleAuditHistoryService;

    private CarrierInfoService carrierInfoService;

    @Autowired
    public VehicleController(ErrorMessageComponent errorMessageComponent, VehicleService vehicleService,
                             VehicleAuditHistoryService vehicleAuditHistoryService , CarrierInfoService carrierInfoService) {
        this.errorMessageComponent = errorMessageComponent;
        this.vehicleService = vehicleService;
        this.vehicleAuditHistoryService = vehicleAuditHistoryService;
        this.carrierInfoService = carrierInfoService;
    }

    /**
     * description 新增车辆
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/26 15:17
     **/
    @PostMapping("/add")
    public EResponse add(@RequestBody @Validated(ValidatorInsert.class) VehicleInfoForm form) throws EException {
        /****************************** 参数校验 ******************************/
        // 总质量/整备质量不能同时为空
        if (null == form.getTotalMass() && null == form.getUnladenMass()) {
            throw new EException(ECode.PARAM_ERROR.code(), "参数totalMass|unladenMass不能同时为空");
        }
        // 道路运输证号，4.5吨及以下普货车辆可以为空
        BigDecimal bigDecimal4_5 = new BigDecimal("4.5");
        if (-1 == bigDecimal4_5.compareTo(form.getLoadCapacity()) && StringUtils.isBlank(form.getRoadCertificateNumber())) {
            throw new EException(ECode.PARAM_ERROR.code(), "4.5顿以上载重的车辆，道路运输证号不能为空");
        }
        /****************************** 参数校验 ******************************/

        Long id = vehicleService.insertVehicle(form);
        JSONObject data = new JSONObject();
        data.put("id", id);
        return EResponse.ok(data);
    }

    /**
     * description 更新车辆信息
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/27 11:05
     **/
    @PostMapping("/edit")
    public EResponse edit(@RequestBody @Validated(ValidatorUpdate.class) VehicleInfoForm form) throws EException {
        vehicleService.updateVehicle(form);
        return EResponse.ok();
    }

    /**
     * description 获取车辆详情
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/27 14:13
     **/
    @PostMapping("/detail")
    public EResponse detail(@RequestBody @Validated(ValidatorDetail.class) VehicleQueryForm form) throws EException {
        VehicleDetailDto dto = vehicleService.getVehicleDetail(form.getId());
        if (null == dto) {
            throw new EException(1001, errorMessageComponent.getCarrierVehicleDetail1001());
        }
        VehicleDetailVO vo = new VehicleDetailVO();
        BeanUtils.copyProperties(dto, vo);
        vo.setCarrierName(carrierInfoService.getCarrierNameById(dto.getCarrierId()));
        vo.setCreateTime(dto.getCreateTime().getTime());

        return EResponse.ok(vo);
    }

    /**
     * description 审核车辆
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/27 15:48
     **/
    @PostMapping("/audit")
    public EResponse audit(@RequestBody VehicleAuditForm form) throws EException {
        if (VehicleConstants.VEHICLE_AUDIT_STATUS_FAILURE.equals(form.getAuditState()) && StringUtils.isBlank(form.getRemark())) {
            throw new EException(ECode.PARAM_ERROR.code(), "审核不通过时remark不能为空");
        }
        vehicleService.auditVehicle(form);

        return EResponse.ok();
    }

    /**
     * description 查看审核历史
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/27 16:36
     **/
    @PostMapping("/auditList")
    public EResponse auditList(@RequestBody @Validated(ValidatorAuditHistory.class) VehicleQueryForm form) throws EException {
        // 判断车辆是否存在
        if (!vehicleService.vehicleExits(form.getId())) {
            throw new EException(1001, errorMessageComponent.getCarrierVehicleAuditList1001());
        }

        // 查询审核历史
        List<VehicleAuditHistoryEntity> entityList = vehicleAuditHistoryService.getVehicleAuditList(form.getId());

        // 数据转换
        List<VehicleAuditVO> voList = new ArrayList<>();
        entityList.forEach(entity -> {
            VehicleAuditVO vo = new VehicleAuditVO();
            BeanUtils.copyProperties(entity, vo);
            vo.setOperateTime(entity.getOperateTime().getTime());
            voList.add(vo);
        });

        return EResponse.ok(voList);
    }
}
