package com.esv.freight.customer.module.contract.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.aspose.words.Document;
import com.aspose.words.FontSettings;
import com.aspose.words.SaveFormat;
import com.esv.freight.customer.common.constants.CommonConstants;
import com.esv.freight.customer.common.util.FeignUtils;
import com.esv.freight.customer.common.util.InputStreamUtils;
import com.esv.freight.customer.feign.FeignFileService;
import com.esv.freight.customer.module.contract.entity.ContractOnlineRecordEntity;
import com.esv.freight.customer.module.contract.service.ContractOnlineRecordService;
import com.esv.freight.customer.module.contract.service.PdfService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Lazy;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.util.Base64;
import java.util.UUID;

/**
 * @description:
 * @project: freight-customer-service
 * @name: com.esv.freight.customer.module.contract.service.impl.PdfServiceImpl
 * @author: 黄朝斌
 * @email: huangchaobin@esvtek.com
 * @createTime: 2020/05/28 20:17
 * @version:1.0
 */
@Service("pdfService")
@Slf4j
public class PdfServiceImpl implements PdfService {

    @Autowired
    private FeignFileService feignFileService;

    @Autowired
    @Lazy
    private ContractOnlineRecordService contractOnlineRecordService;

    @Value("${font.windows.dir}")
    private String fontWindowsDir;

    @Override
    @Async
    public void asyncSaveTargetFileInfo(String traceId, ContractOnlineRecordEntity entity, byte[] fileBytes) {
        String contractNumber = entity.getContractNumber();
        if (StringUtils.isEmpty(traceId) || CommonConstants.NULL_STRING.equalsIgnoreCase(traceId)) {
            traceId = UUID.randomUUID().toString().replace("-", "");
        }
        MDC.put("traceId", traceId);
        log.info("------异步保存合同[{}]目标文件及信息开始------", contractNumber);

        // 转换源文件为PDF
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        InputStream inputStream = InputStreamUtils.byte2InputStream(fileBytes);
        byte[] targetFileBytes;
        try {
            String os = System.getProperty("os.name").toLowerCase();
            if (!os.contains("windows")) {
                FontSettings fontSettings = new FontSettings();
                fontSettings.setFontsFolder(fontWindowsDir, false);
            }
            Document document = new Document(inputStream);
            document.save(outputStream, SaveFormat.PDF);
            targetFileBytes = outputStream.toByteArray();
        } catch (Exception e) {
            log.error("合同[{}]转换为PDF时失败：{}", contractNumber, e.getMessage());
            log.info("------异步保存合同[{}]目标文件及信息结束------", contractNumber);
            return;
        } finally {
            IOUtils.closeQuietly(inputStream);
            IOUtils.closeQuietly(outputStream);
        }

        // 保存PDF文件
        JSONObject feignReqJson = new JSONObject();
        feignReqJson.put("fileType", "pdf");
        feignReqJson.put("fileName", "电子合同-" + contractNumber + ".pdf");
        feignReqJson.put("fileData", Base64.getEncoder().encodeToString(targetFileBytes));
        JSONObject feignResJson;
        try {
            feignResJson = FeignUtils.getFeignResultData(feignFileService.uploadSingleFile(feignReqJson));
        } catch (Exception e) {
            log.error("调用[文件服务]上传目标合同[{}]文件数据失败：{}", contractNumber, e.getMessage(), e);
            log.info("------异步保存合同[{}]目标文件及信息结束------", contractNumber);
            return;
        }

        // 更新合同记录信息
        ContractOnlineRecordEntity onlineRecordEntity = new ContractOnlineRecordEntity();
        onlineRecordEntity.setId(entity.getId());
        onlineRecordEntity.setTargetFileId(feignResJson.getString("id"));
        onlineRecordEntity.setTargetFileUrl(feignResJson.getString("url"));
        contractOnlineRecordService.getBaseMapper().updateById(onlineRecordEntity);

        log.info("------异步保存合同[{}]目标文件及信息结束------", contractNumber);
    }
}
