package com.esv.freight.customer.module.goodsowner.controller;

import com.alibaba.fastjson.JSONObject;
import com.esv.freight.customer.common.component.ErrorMessageComponent;
import com.esv.freight.customer.common.component.PasswordComponent;
import com.esv.freight.customer.common.exception.EException;
import com.esv.freight.customer.common.response.ECode;
import com.esv.freight.customer.common.response.EResponse;
import com.esv.freight.customer.common.util.ReqUtils;
import com.esv.freight.customer.common.util.VerifyUtils;
import com.esv.freight.customer.common.validator.groups.ValidatorDetail;
import com.esv.freight.customer.common.validator.groups.ValidatorInsert;
import com.esv.freight.customer.common.validator.groups.ValidatorList;
import com.esv.freight.customer.common.validator.groups.ValidatorUpdate;
import com.esv.freight.customer.common.vo.PageResultVO;
import com.esv.freight.customer.module.goodsowner.GoodsOwnerConstants;
import com.esv.freight.customer.module.goodsowner.dto.AccountInfoDto;
import com.esv.freight.customer.module.goodsowner.entity.AuditHistoryEntity;
import com.esv.freight.customer.module.goodsowner.form.AccountForm;
import com.esv.freight.customer.module.goodsowner.form.AccountInfoForm;
import com.esv.freight.customer.module.goodsowner.form.AccountQueryForm;
import com.esv.freight.customer.module.goodsowner.form.AccountScoreForm;
import com.esv.freight.customer.module.goodsowner.service.GoodsOwnerAccountService;
import com.esv.freight.customer.module.goodsowner.service.GoodsOwnerInfoService;
import com.esv.freight.customer.module.goodsowner.validator.groups.*;
import com.esv.freight.customer.module.goodsowner.vo.AccountInfoVO;
import com.esv.freight.customer.module.goodsowner.vo.AuditHistoryVO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.ArrayList;
import java.util.List;

/**
 * @description: 货主帐号Controller
 * @project: freight-customer-service
 * @name: com.esv.freight.customer.module.goodsowner.controller.GoodsOwnerController
 * @author: 黄朝斌
 * @email: huangchaobin@esvtek.com
 * @createTime: 2020/04/17 14:33
 * @version:1.0
 */
@Slf4j
@RestController
@RequestMapping("/goodsowner/account")
@Validated
public class GoodsOwnerController {

    private ErrorMessageComponent errorMessageComponent;
    private PasswordComponent passwordComponent;
    private GoodsOwnerAccountService goodsOwnerAccountService;
    private GoodsOwnerInfoService goodsOwnerInfoService;

    @Autowired
    public GoodsOwnerController(GoodsOwnerAccountService goodsOwnerAccountService, ErrorMessageComponent errorMessageComponent, 
                                PasswordComponent passwordComponent, GoodsOwnerInfoService goodsOwnerInfoService) {
        this.goodsOwnerAccountService = goodsOwnerAccountService;
        this.errorMessageComponent = errorMessageComponent;
        this.passwordComponent = passwordComponent;
        this.goodsOwnerInfoService = goodsOwnerInfoService;
    }

    /**
     * description 平台用户创建帐号
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/17 17:39
     **/
    @PostMapping("/add")
    public EResponse createAccountByManage(@RequestBody @Validated(ValidatorInsert.class) AccountInfoForm form) throws EException {
        /****************************** 参数校验 ******************************/
        // 货主为企业
        if (GoodsOwnerConstants.OWNER_TYPE_COMPANY.equals(form.getOwnerType())) {
            String[] notBlankParams = new String[] {"ownerFullName", "uniCreditCode", "creditCopyUrl", "legalPerson"};
            ReqUtils.checkParamsNotBlank(JSONObject.parseObject(form.toString()), notBlankParams);
        }
        // 校验身份证号码
        if (!VerifyUtils.isValidIdCard18(form.getIdCard())) {
            throw new EException(ECode.PARAM_ERROR.code(), "无效的身份证号码");
        }
        /****************************** 参数校验 ******************************/

        return EResponse.ok(goodsOwnerAccountService.createAccountByManager(form));
    }

    /**
     * description 更新帐号信息
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/18 14:06
     **/
    @PostMapping("/edit")
    public EResponse updateAccountInfo(@RequestBody @Validated(ValidatorUpdate.class) AccountInfoForm form) throws EException {
        /****************************** 参数校验 ******************************/
        // 校验身份证号码
        if (StringUtils.isNotBlank(form.getIdCard()) && !VerifyUtils.isValidIdCard18(form.getIdCard())) {
            throw new EException(ECode.PARAM_ERROR.code(), "无效的身份证号码");
        }
        /****************************** 参数校验 ******************************/

        goodsOwnerAccountService.updateAccountInfo(form);
        return EResponse.ok();
    }

    /**
     * description 停用帐号
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/18 15:11
     **/
    @PostMapping("/block")
    public EResponse blockAccount(@RequestBody @Validated(ValidatorBlock.class) AccountForm form) throws EException {
        goodsOwnerAccountService.blockAccount(form.getId());
        return EResponse.ok();
    }

    /**
     * description 启用帐号
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/18 15:11
     **/
    @PostMapping("/unblock")
    public EResponse unblockAccount(@RequestBody @Validated(ValidatorUnblock.class) AccountForm form) throws EException {
        goodsOwnerAccountService.unblockAccount(form.getId());
        return EResponse.ok();
    }

    /**
     * description 审核帐号
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/18 16:08
     **/
    @PostMapping("/audit")
    public EResponse auditAccount(@RequestBody @Validated(ValidatorAudit.class) AccountForm form) throws EException {
        /****************************** 参数校验 ******************************/
        // 校验备注
        if (GoodsOwnerConstants.OWNER_AUDIT_STATUS_FAILURE.equals(form.getAuditStatus()) && StringUtils.isBlank(form.getRemark())) {
            throw new EException(ECode.PARAM_ERROR.code(), "参数remark不能为空");
        }
        /****************************** 参数校验 ******************************/

        goodsOwnerAccountService.auditAccount(form);
        return EResponse.ok();
    }

    /**
     * description 查看帐号的审核历史
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/18 16:50
     **/
    @PostMapping("/audit/history")
    public EResponse getAccountAuditHistory(@RequestBody @Validated(ValidatorAuditHistory.class) AccountForm form) throws EException {
        // 查询审核历史
        List<AuditHistoryEntity> auditHistoryEntityList = goodsOwnerAccountService.getAuditHistory(form.getId());

        // 数据转换
        List<AuditHistoryVO> auditHistoryVOList = new ArrayList<>();
        auditHistoryEntityList.forEach(auditHistoryEntity -> {
            AuditHistoryVO auditHistoryVO = new AuditHistoryVO();
            BeanUtils.copyProperties(auditHistoryEntity, auditHistoryVO);
            auditHistoryVO.setOperateTime(auditHistoryEntity.getOperateTime().getTime());
            auditHistoryVOList.add(auditHistoryVO);
        });

        return EResponse.ok(auditHistoryVOList);
    }

    /**
     * description 获取帐号详细信息
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/20 10:12
     **/
    @PostMapping("/detail")
    public EResponse getAccountDetail(@RequestBody @Validated(ValidatorDetail.class) AccountForm form) throws EException {
        /****************************** 参数校验 ******************************/
        if (null == form.getId() && null == form.getAccount()) {
            throw new EException(ECode.PARAM_ERROR.code(), "参数id、account不能同时为空");
        }
        /****************************** 参数校验 ******************************/

        AccountInfoDto accountInfoDto = goodsOwnerAccountService.getAccountInfo(form);
        // 查询结果处理
        if (null != form.getId()) {
            if (null == accountInfoDto) {
                throw new EException(1001, errorMessageComponent.getGoodsOwnerAccountDetail1001());
            }
        } else if (null != form.getAccount()) {
            if (null == accountInfoDto) {
                throw new EException(1002, errorMessageComponent.getGoodsOwnerAccountDetail1002());
            }
        } else {

        }

        AccountInfoVO accountInfoVO = new AccountInfoVO();
        BeanUtils.copyProperties(accountInfoDto, accountInfoVO);
        return EResponse.ok(accountInfoVO);
    }

    /**
     * description 帐号密码校验
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/20 11:22
     **/
    @PostMapping("/password/check")
    public EResponse checkAccountPwd(@RequestBody @Validated(ValidatorPasswordCheck.class) AccountForm form) throws EException {
        AccountInfoDto accountInfoDto = goodsOwnerAccountService.getAccountInfo(form);
        // 校验帐号是否存在
        if (null == accountInfoDto) {
            throw new EException(1001, errorMessageComponent.getGoodsOwnerAccountPasswordCheck1001());
        }
        // 校验密码是否正确
        if (!passwordComponent.checkPwd4Salt(form.getPassword(), accountInfoDto.getSalt(), accountInfoDto.getPassword())) {
            throw new EException(1002, errorMessageComponent.getGoodsOwnerAccountPasswordCheck1002());
        }

        AccountInfoVO accountInfoVO = new AccountInfoVO();
        BeanUtils.copyProperties(accountInfoDto, accountInfoVO);
        return EResponse.ok(accountInfoVO);
    }

    /**
     * description 重置帐号密码
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/05/16 16:29
     **/
    @PostMapping("/password/reset")
    public EResponse resetAccountPwd(@RequestBody @Validated(ValidatorPasswordReset.class) AccountForm form) throws EException {
        goodsOwnerAccountService.resetAccountPwd(form);
        return EResponse.ok();
    }

    /**
     * description 帐号注册
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/20 14:56
     **/
    @PostMapping("/register")
    public EResponse accountRegister(@RequestBody @Validated(ValidatorRegister.class) AccountForm form) throws EException {
        Long id = goodsOwnerAccountService.registerAccount(form.getAccount());
        JSONObject data = new JSONObject();
        data.put("id", id);
        return EResponse.ok(data);
    }

    /**
     * description 分页查询帐号列表
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/20 20:03
     **/
    @PostMapping("/list")
    public EResponse getAccountList(@RequestBody @Validated(ValidatorList.class) AccountQueryForm form) throws EException {
        PageResultVO pageResult = goodsOwnerAccountService.selectAccountInfoList(form);
        return EResponse.ok(pageResult);
    }

    /**
     * description 获取所有可用货主帐号列表
     * param []
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/21 14:11
     **/
    @PostMapping("/getAllAvailable")
    public EResponse getAllAvailable() throws EException {
        return EResponse.ok(goodsOwnerAccountService.selectAllAvailableAccount());
    }

    /**
     * description 更新货主整体评分
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author HuangChaobin
     * createTime 2020/06/01 11:01
     **/
    @PostMapping("/score/overall/update")
    public EResponse updateOverallScore(@RequestBody @Validated(ValidatorUpdate.class) AccountScoreForm form) throws EException {
        goodsOwnerInfoService.updateOverallScore(form);
        return EResponse.ok();
    }
}
