package com.esv.freight.customer.module.pay.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.esv.freight.customer.common.component.ErrorMessageComponent;
import com.esv.freight.customer.common.exception.EException;
import com.esv.freight.customer.common.pojo.AppUnionPayOrderReq;
import com.esv.freight.customer.common.pojo.AppUnionPayOrderRes;
import com.esv.freight.customer.common.unionpay.UnionPayComponent;
import com.esv.freight.customer.common.util.DateUtils;
import com.esv.freight.customer.feign.FreightBillService;
import com.esv.freight.customer.module.pay.PayConstants;
import com.esv.freight.customer.module.pay.dao.CustomerUnionpayOrderDao;
import com.esv.freight.customer.module.pay.entity.CustomerUnionpayOrderEntity;
import com.esv.freight.customer.module.pay.form.CustomerPayOrderForm;
import com.esv.freight.customer.module.pay.service.UnionpayOrderService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.util.Map;


@Service("unionpayOrderService")
@Slf4j
public class UnionpayOrderServiceImpl extends ServiceImpl<CustomerUnionpayOrderDao, CustomerUnionpayOrderEntity> implements UnionpayOrderService {

    @Value("${unionpay.mer-id}")
    private String unionPayMerId;

    private ErrorMessageComponent errorMessageComponent;
    private UnionPayComponent unionPayComponent;
    private FreightBillService freightBillService;

    @Autowired
    public UnionpayOrderServiceImpl(ErrorMessageComponent errorMessageComponent, UnionPayComponent unionPayComponent,
                                    FreightBillService freightBillService) {
        this.errorMessageComponent = errorMessageComponent;
        this.unionPayComponent = unionPayComponent;
        this.freightBillService = freightBillService;
    }

    @Override
    public CustomerUnionpayOrderEntity getRecordByBillId(String billId) {
        return this.baseMapper.selectOne(new LambdaQueryWrapper<CustomerUnionpayOrderEntity>().eq(CustomerUnionpayOrderEntity::getBillId, billId));
    }

    @Override
    public CustomerUnionpayOrderEntity getRecordByTn(String tn) {
        return this.baseMapper.selectOne(new LambdaQueryWrapper<CustomerUnionpayOrderEntity>().eq(CustomerUnionpayOrderEntity::getTn, tn));
    }

    @Override
    public String createCustomerPayOrder(CustomerPayOrderForm form) {
        // 判断订单是否存在
        CustomerUnionpayOrderEntity unionpayOrderEntity = this.getRecordByBillId(form.getBillId());
        if (null != unionpayOrderEntity) {
            if (PayConstants.ORDER_STATUS_SUCCESS.equals(unionpayOrderEntity.getOrderStatus())) {
                throw new EException(1001, errorMessageComponent.getPayCustomerOrderCreate1001());
            } else {
                // TODO 请求银联查询订单支付状态
            }
        }

        // 请求支付平台获取支付订单号
        AppUnionPayOrderReq appUnionPayOrderReq = new AppUnionPayOrderReq();
        appUnionPayOrderReq.setMerId(unionPayMerId);
        appUnionPayOrderReq.setBillId(form.getBillId());
        appUnionPayOrderReq.setBillDesc(form.getBillDesc());
        appUnionPayOrderReq.setTxnAmt(form.getTxnAmt());
        appUnionPayOrderReq.setTxnTime(DateUtils.format(DateUtils.getSysdate(), DateUtils.DATE_FORMAT4));
        AppUnionPayOrderRes appUnionPayOrderRes = unionPayComponent.createAppUnionPayOrder(appUnionPayOrderReq);
        String tn = appUnionPayOrderRes.getTn();

        // 新增或更新订单
        CustomerUnionpayOrderEntity customerUnionPayOrderEntity = new CustomerUnionpayOrderEntity();
        if (null == unionpayOrderEntity) {
            // 新增订单信息
            customerUnionPayOrderEntity.setCustomerId(form.getCustomerId());
            customerUnionPayOrderEntity.setMerId(unionPayMerId);
            customerUnionPayOrderEntity.setBillId(form.getBillId());
            customerUnionPayOrderEntity.setBillDesc(form.getBillDesc());
            customerUnionPayOrderEntity.setTxnTime(appUnionPayOrderReq.getTxnTime());
            customerUnionPayOrderEntity.setTxnAmt(form.getTxnAmt());
            customerUnionPayOrderEntity.setTn(tn);
            customerUnionPayOrderEntity.setOrderStatus(PayConstants.ORDER_STATUS_PAYING);
            this.getBaseMapper().insert(customerUnionPayOrderEntity);
        } else {
            // 更新订单信息
            customerUnionPayOrderEntity.setId(unionpayOrderEntity.getId());
            customerUnionPayOrderEntity.setTn(tn);
            customerUnionPayOrderEntity.setTxnTime(appUnionPayOrderReq.getTxnTime());
            customerUnionPayOrderEntity.setBillDesc(form.getBillDesc());
            customerUnionPayOrderEntity.setTxnAmt(form.getTxnAmt());
            this.baseMapper.updateById(customerUnionPayOrderEntity);
        }

        return tn;
    }

    @Override
    public Integer getOrderStatusByTn(String tn) {
        CustomerUnionpayOrderEntity customerUnionPayOrderEntity = this.getRecordByTn(tn);
        if (null == customerUnionPayOrderEntity) {
            throw new EException(1001, "该支付订单流水号[" + tn + "]不存在");
        }

        return this.queryOrderStatus2Unionpay(customerUnionPayOrderEntity);
    }

    @Override
    public Integer getOrderStatusByBillId(String billId) {
        CustomerUnionpayOrderEntity customerUnionPayOrderEntity = this.getRecordByBillId(billId);
        if (null == customerUnionPayOrderEntity) {
            throw new EException(1001, "该账单号[" + billId + "]不存在");
        }

        return this.queryOrderStatus2Unionpay(customerUnionPayOrderEntity);
    }

    /**
     * description 请求银联查询订单支付状态
     * param [customerUnionPayOrderEntity]
     * return java.lang.Integer
     * author HuangChaobin
     * createTime 2020/06/03 11:08
     **/
    @Override
    public Integer queryOrderStatus2Unionpay(CustomerUnionpayOrderEntity customerUnionPayOrderEntity) {
        Integer orderStatus;

        // 判断订单是否支付完成
        orderStatus = customerUnionPayOrderEntity.getOrderStatus();
        if (PayConstants.ORDER_STATUS_SUCCESS.equals(orderStatus)) {
            return orderStatus;
        } else {
            // 判断订单是否超过30分钟，暂时取消这个判断逻辑
//            Long current = System.currentTimeMillis();
//            Long createTime = customerUnionPayOrderEntity.getCreateTime().getTime();
//            if (PayConstants.UNIONPAY_QUERY_MAX_TIME < (current - createTime)) {
//                return orderStatus;
//            }
        }

        // 请求银联查询订单支付状态
        AppUnionPayOrderReq appUnionPayOrderReq = new AppUnionPayOrderReq();
        appUnionPayOrderReq.setMerId(unionPayMerId);
        appUnionPayOrderReq.setBillId(customerUnionPayOrderEntity.getBillId());
        appUnionPayOrderReq.setTxnTime(customerUnionPayOrderEntity.getTxnTime());
        Map<String, String> queryResultMap = unionPayComponent.queryUnionPayOrder(appUnionPayOrderReq);
        if ("00".equals(queryResultMap.get("respCode"))) {
            String origRespCode = queryResultMap.get("origRespCode");
            if (("00").equals(origRespCode)) {
                //交易成功，更新商户订单状态
                orderStatus = PayConstants.ORDER_STATUS_SUCCESS;
            } else if (("03").equals(origRespCode) ||
                    ("04").equals(origRespCode) ||
                    ("05").equals(origRespCode)) {
                //订单处理中或交易状态未明，需稍后发起交易状态查询交易 【如果最终尚未确定交易是否成功请以对账文件为准】
                orderStatus = PayConstants.ORDER_STATUS_CONFIRM;
            } else {
                //其他应答码为交易失败
                orderStatus = PayConstants.ORDER_STATUS_FAILURE;
            }

            // 更新订单支付状态
            CustomerUnionpayOrderEntity orderEntity = new CustomerUnionpayOrderEntity();
            orderEntity.setId(customerUnionPayOrderEntity.getId());
            orderEntity.setQueryCount(customerUnionPayOrderEntity.getQueryCount() + 1);
            orderEntity.setOrigRespCode(origRespCode);
            orderEntity.setOrderStatus(orderStatus);
            this.baseMapper.updateById(orderEntity);

            // 通知账单服务：更改应收账单支付状态
            JSONObject feignReqJson = new JSONObject();
            feignReqJson.put("billNo", customerUnionPayOrderEntity.getBillId());
            feignReqJson.put("payState", orderStatus);
            try {
                log.info("Feign请求[账单服务]传参：{}", feignReqJson.toJSONString());
                JSONObject feignResJson = freightBillService.updatePayState(feignReqJson);
                log.info("Feign请求[账单服务]返回：{}", feignResJson.toJSONString());
            } catch (Exception e) {
                log.error("Feign请求[账单服务]失败：{}", e.getMessage(), e);
            }
        }

        return orderStatus;
    }
}