package com.esv.freight.customer.module.goodsowner.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.esv.freight.customer.common.component.ErrorMessageComponent;
import com.esv.freight.customer.common.exception.EException;
import com.esv.freight.customer.common.util.FeignUtils;
import com.esv.freight.customer.feign.FeignBaseService;
import com.esv.freight.customer.module.goodsowner.dao.DeliveryAddressDao;
import com.esv.freight.customer.module.goodsowner.entity.AuditHistoryEntity;
import com.esv.freight.customer.module.goodsowner.entity.DeliveryAddressEntity;
import com.esv.freight.customer.module.goodsowner.form.DeleteAddressForm;
import com.esv.freight.customer.module.goodsowner.form.DeliveryAddressForm;
import com.esv.freight.customer.module.goodsowner.service.DeliveryAddressService;
import com.esv.freight.customer.module.goodsowner.vo.DeliveryAddressVO;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.List;


@Service("deliveryAddressService")
public class DeliveryAddressServiceImpl extends ServiceImpl<DeliveryAddressDao, DeliveryAddressEntity> implements DeliveryAddressService {

    private FeignBaseService feignBaseService;

    private ErrorMessageComponent errorMessageComponent;

    @Autowired
    public DeliveryAddressServiceImpl(FeignBaseService feignBaseService, ErrorMessageComponent errorMessageComponent) {
        this.feignBaseService = feignBaseService;
        this.errorMessageComponent = errorMessageComponent;
    }

    @Override
    public Long addAddress(DeliveryAddressForm form) {
        // 1.判断地址是否重复
        List<DeliveryAddressEntity> deliveryAddressEntityList = this.getAddressByName(form.getOwnerId(), form.getAddressName());
        if (0 < deliveryAddressEntityList.size()) {
            throw new EException(1001, errorMessageComponent.getGoodsOwnerDeliveryAddressAdd1001());
        }

        // 2.获取地址编码
        JSONObject batchIdReqJson = new JSONObject();
        batchIdReqJson.put("prefix", "FH");
        batchIdReqJson.put("formatter", "yyyyMMdd");
        batchIdReqJson.put("length", 13);
        JSONObject batchIdResJson;
        try {
            batchIdResJson = FeignUtils.getFeignResultData(feignBaseService.getBatchId(batchIdReqJson));
        } catch (Exception e) {
            log.error("调用[基础服务]生成发货地址编号失败");
            throw new EException("生成发货地址编号时发生错误");
        }
        String addressNumber = batchIdResJson.getString("batchId");

        // 3.新增地址
        DeliveryAddressEntity addressEntity = new DeliveryAddressEntity();
        BeanUtils.copyProperties(form, addressEntity);
        addressEntity.setAddressNumber(addressNumber);
        if (StringUtils.isNotBlank(form.getLon())) {
            addressEntity.setLon(new BigDecimal(form.getLon()));
        }
        if (StringUtils.isNotBlank(form.getLat())) {
            addressEntity.setLat(new BigDecimal(form.getLat()));
        }
        this.baseMapper.insert(addressEntity);

        return addressEntity.getId();
    }

    @Override
    public int editAddress(DeliveryAddressForm form) {
        // 1.判断ID是否有效
        DeliveryAddressEntity entity = this.baseMapper.selectById(form.getId());
        if (null == entity) {
            throw new EException(1001, errorMessageComponent.getGoodsOwnerDeliveryAddressEdit1001());
        }

        // 2.判断地址是否重复
        QueryWrapper<DeliveryAddressEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.ne("id", form.getId());
        queryWrapper.eq("owner_id", form.getOwnerId());
        queryWrapper.eq("address_name", form.getAddressName());
        List<DeliveryAddressEntity> deliveryAddressEntityList = this.baseMapper.selectList(queryWrapper);
        if (0 < deliveryAddressEntityList.size()) {
            throw new EException(1002, errorMessageComponent.getGoodsOwnerDeliveryAddressEdit1002());
        }

        // 3.更新地址
        DeliveryAddressEntity updateEntity = new DeliveryAddressEntity();
        BeanUtils.copyProperties(form, updateEntity);
        if (StringUtils.isNotBlank(form.getLon())) {
            updateEntity.setLon(new BigDecimal(form.getLon()));
        }
        if (StringUtils.isNotBlank(form.getLat())) {
            updateEntity.setLat(new BigDecimal(form.getLat()));
        }
        int flag = this.baseMapper.updateById(updateEntity);

        return flag;
    }

    @Override
    public int deleteAddress(DeleteAddressForm form) {
        String[] ids = form.getId().split(",");
        int count = this.baseMapper.deleteBatchIds(Arrays.asList(ids));
        return count;
    }

    @Override
    public DeliveryAddressVO getAddressDetail(DeliveryAddressForm form) {
        // 1.判断ID是否有效
        DeliveryAddressEntity entity = this.baseMapper.selectById(form.getId());
        if (null == entity) {
            throw new EException(1001, errorMessageComponent.getGoodsOwnerDeliveryAddressDetail1001());
        }

        // 数据转换
        DeliveryAddressVO addressVO = new DeliveryAddressVO();
        BeanUtils.copyProperties(entity, addressVO);

        return addressVO;
    }

    @Override
    public List<DeliveryAddressEntity> getAddressByName(Long ownerId, String addressName) {
        QueryWrapper<DeliveryAddressEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("owner_id", ownerId);
        queryWrapper.eq("address_name", addressName);
        return this.getBaseMapper().selectList(queryWrapper);
    }
}