package com.esv.freight.customer.module.contract.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.aspose.words.Document;
import com.aspose.words.SaveFormat;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.deepoove.poi.XWPFTemplate;
import com.deepoove.poi.data.PictureRenderData;
import com.esv.freight.customer.common.component.ErrorMessageComponent;
import com.esv.freight.customer.common.component.RedisComponent;
import com.esv.freight.customer.common.constants.CommonConstants;
import com.esv.freight.customer.common.exception.EException;
import com.esv.freight.customer.common.response.ECode;
import com.esv.freight.customer.common.util.DateUtils;
import com.esv.freight.customer.common.util.FeignUtils;
import com.esv.freight.customer.common.util.InputStreamUtils;
import com.esv.freight.customer.common.util.ReqUtils;
import com.esv.freight.customer.feign.FeignBaseService;
import com.esv.freight.customer.feign.FeignFileService;
import com.esv.freight.customer.module.contract.ContractConstants;
import com.esv.freight.customer.module.contract.dao.ContractOnlineRecordDao;
import com.esv.freight.customer.module.contract.entity.ContractOnlinePlatformDataEntity;
import com.esv.freight.customer.module.contract.entity.ContractOnlineRecordEntity;
import com.esv.freight.customer.module.contract.entity.ContractOnlineTemplateEntity;
import com.esv.freight.customer.module.contract.form.ContractOnlineRecordSignDriverForm;
import com.esv.freight.customer.module.contract.form.ContractOnlineRecordSignGoodsOwnerForm;
import com.esv.freight.customer.module.contract.pojo.ContractOnlineDriverPojo;
import com.esv.freight.customer.module.contract.pojo.ContractOnlineGoodsOwnerPojo;
import com.esv.freight.customer.module.contract.pojo.ContractOnlinePlatformPojo;
import com.esv.freight.customer.module.contract.service.ContractOnlinePlatformDataService;
import com.esv.freight.customer.module.contract.service.ContractOnlineRecordService;
import com.esv.freight.customer.module.contract.service.ContractOnlineTemplateService;
import com.esv.freight.customer.module.contract.vo.ContractOnlineRecordSignDriverVO;
import com.esv.freight.customer.module.contract.vo.ContractOnlineRecordSignGoodsOwnerVO;
import com.esv.freight.customer.module.contract.vo.ContractOnlineRecordVO;
import com.esv.freight.customer.module.driver.dto.DriverDetailDto;
import com.esv.freight.customer.module.driver.service.DriverAccountService;
import com.esv.freight.customer.module.goodsowner.GoodsOwnerConstants;
import com.esv.freight.customer.module.goodsowner.dto.AccountInfoDto;
import com.esv.freight.customer.module.goodsowner.form.AccountForm;
import com.esv.freight.customer.module.goodsowner.service.GoodsOwnerAccountService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.MDC;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Base64;
import java.util.Date;
import java.util.UUID;


@Service("contractOnlineRecordService")
@Slf4j
public class ContractOnlineRecordServiceImpl extends ServiceImpl<ContractOnlineRecordDao, ContractOnlineRecordEntity> implements ContractOnlineRecordService {

    @Value("${spring.application.name}")
    private String applicationName;

    private static final Long CONTRACT_CACHE_TIME = 86400L;

    /**
     * 货主/司机签名图像宽度、长度
     **/
    private static final int CUSTOMER_SIGN_WIDTH = 150;
    private static final int CUSTOMER_SIGN_HEIGHT = 60;

    /**
     * 平台公章图像宽度、长度
     **/
    private static final int PLATFORM_SEAL_WIDTH = 200;
    private static final int PLATFORM_SEAL_HEIGHT = 200;

    private ErrorMessageComponent errorMessageComponent;
    private FeignFileService feignFileService;
    private FeignBaseService feignBaseService;
    private RedisComponent redisComponent;
    private ContractOnlineTemplateService contractOnlineTemplateService;
    private GoodsOwnerAccountService goodsOwnerAccountService;
    private DriverAccountService driverAccountService;
    private ContractOnlinePlatformDataService contractOnlinePlatformDataService;

    @Autowired
    public ContractOnlineRecordServiceImpl(ErrorMessageComponent errorMessageComponent, FeignFileService feignFileService,
                                           FeignBaseService feignBaseService, RedisComponent redisComponent,
                                           ContractOnlineTemplateService contractOnlineTemplateService, GoodsOwnerAccountService goodsOwnerAccountService,
                                           DriverAccountService driverAccountService, ContractOnlinePlatformDataService contractOnlinePlatformDataService) {
        this.errorMessageComponent = errorMessageComponent;
        this.feignFileService = feignFileService;
        this.feignBaseService = feignBaseService;
        this.redisComponent = redisComponent;
        this.contractOnlineTemplateService = contractOnlineTemplateService;
        this.goodsOwnerAccountService = goodsOwnerAccountService;
        this.driverAccountService = driverAccountService;
        this.contractOnlinePlatformDataService = contractOnlinePlatformDataService;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ContractOnlineRecordSignGoodsOwnerVO goodsOwnerSign(ContractOnlineRecordSignGoodsOwnerForm form) {
        // 1:获取货主信息
        AccountForm accountForm = new AccountForm();
        accountForm.setId(form.getGoodsOwnerId());
        AccountInfoDto accountInfo = goodsOwnerAccountService.getAccountInfo(accountForm);
        if (null == accountInfo) {
            throw new EException(1001, errorMessageComponent.getContractOnlineGoodsOwnerSign1001());
        }

        // 2:获取电子合同模板数据
        String contractTemplateData = this.getContractOnlineTemplateEntity(ContractConstants.CONTRACT_TYPE_GOODS_OWNER);
        byte[] contractTemplateBytes = Base64.getDecoder().decode(contractTemplateData);

        // 3:填充电子合同模板
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        ContractOnlineGoodsOwnerPojo pojo;
        byte[] signBytes;
        XWPFTemplate template = null;
        try {
            template = XWPFTemplate.compile(InputStreamUtils.byte2InputStream(contractTemplateBytes));
            pojo = this.getGoodsOwnerSignDataMap(form, accountInfo);
            template.render(pojo);
            template.write(byteArrayOutputStream);
            byteArrayOutputStream.flush();
            signBytes = byteArrayOutputStream.toByteArray();
        } catch (IOException e) {
            log.error("填充电子合同模板失败：" + e.getMessage());
            throw new EException(ECode.SERVER_ERROR.code(), "生成电子合同失败[填充电子合同模板失败]");
        } finally {
            if (null != template) {
                try {
                    template.close();
                } catch (IOException e) {
                    log.error(e.getMessage(), e);
                }
            }
            IOUtils.closeQuietly(byteArrayOutputStream);
        }

        // 4:保存货主签订后的电子合同文件
        String contractNumber = pojo.getContactNumber();
        JSONObject feignReqJson = new JSONObject();
        feignReqJson.put("fileType", "word");
        feignReqJson.put("fileName", "电子合同-货主签名-" + contractNumber + ".docx");
        feignReqJson.put("fileData", Base64.getEncoder().encodeToString(signBytes));
        JSONObject feignResJson;
        try {
            feignResJson = FeignUtils.getFeignResultData(feignFileService.uploadSingleFile(feignReqJson));
        } catch (Exception e) {
            log.error("调用[文件服务]上传货主签名合同文件数据失败：" + e.getMessage());
            throw new EException("生成电子合同失败[上传货主签名合同文件数据失败]");
        }

        // 5:保存货主签订后的电子合同记录
        ContractOnlineRecordEntity onlineRecordEntity = new ContractOnlineRecordEntity();
        onlineRecordEntity.setContractType(ContractConstants.CONTRACT_TYPE_GOODS_OWNER);
        onlineRecordEntity.setSourceFileId(feignResJson.getString("id"));
        onlineRecordEntity.setSourceFileUrl(feignResJson.getString("url"));
        onlineRecordEntity.setContractNumber(contractNumber);
        onlineRecordEntity.setBusinessNumber(form.getBusinessNumber());
        onlineRecordEntity.setCustomerId(form.getGoodsOwnerId());
        onlineRecordEntity.setCustomerSignTime(new Date());
        onlineRecordEntity.setSignComplete(ContractConstants.CONTRACT_SIGN_COMPLETE_NO);
        this.baseMapper.insert(onlineRecordEntity);

        // 6:异步保存线上合同目标文件（PDF）及信息
        this.asyncSaveTargetFileInfo(MDC.get("traceId"), onlineRecordEntity, signBytes);

        // 7:返回
        ContractOnlineRecordSignGoodsOwnerVO vo = new ContractOnlineRecordSignGoodsOwnerVO();
        vo.setContractNumber(contractNumber);
        return vo;
    }

    @Override
    public ContractOnlineRecordSignDriverVO driverSign(ContractOnlineRecordSignDriverForm form) {
        // 1:获取司机信息
        DriverDetailDto accountInfo = driverAccountService.getDriverDetailById(form.getDriverId());
        if (null == accountInfo) {
            throw new EException(1001, errorMessageComponent.getContractOnlineDriverSign1001());
        }

        // 2:获取电子合同模板数据
        String contractTemplateData = this.getContractOnlineTemplateEntity(ContractConstants.CONTRACT_TYPE_DRIVER);
        byte[] contractTemplateBytes = Base64.getDecoder().decode(contractTemplateData);

        // 3:填充电子合同模板
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        ContractOnlineDriverPojo pojo;
        byte[] signBytes;
        XWPFTemplate template = null;
        try {
            template = XWPFTemplate.compile(InputStreamUtils.byte2InputStream(contractTemplateBytes));
            pojo = this.getDriverSignDataMap(form, accountInfo);
            template.render(pojo);
            template.write(byteArrayOutputStream);
            byteArrayOutputStream.flush();
            signBytes = byteArrayOutputStream.toByteArray();
        } catch (IOException e) {
            log.error("填充电子合同模板失败：" + e.getMessage());
            throw new EException(ECode.SERVER_ERROR.code(), "生成电子合同失败[填充电子合同模板失败]");
        } finally {
            if (null != template) {
                try {
                    template.close();
                } catch (IOException e) {
                    log.error(e.getMessage(), e);
                }
            }
            IOUtils.closeQuietly(byteArrayOutputStream);
        }

        // 4:保存司机签订后的电子合同文件
        String contractNumber = pojo.getContactNumber();
        JSONObject feignReqJson = new JSONObject();
        feignReqJson.put("fileType", "word");
        feignReqJson.put("fileName", "电子合同-司机签名-" + contractNumber + ".docx");
        feignReqJson.put("fileData", Base64.getEncoder().encodeToString(signBytes));
        JSONObject feignResJson;
        try {
            feignResJson = FeignUtils.getFeignResultData(feignFileService.uploadSingleFile(feignReqJson));
        } catch (Exception e) {
            log.error("调用[文件服务]上传货主签名合同文件数据失败：" + e.getMessage());
            throw new EException("生成电子合同失败[上传司机签名合同文件数据失败]");
        }

        // 5:保存货主签订后的电子合同记录
        ContractOnlineRecordEntity onlineRecordEntity = new ContractOnlineRecordEntity();
        onlineRecordEntity.setContractType(ContractConstants.CONTRACT_TYPE_DRIVER);
        onlineRecordEntity.setSourceFileId(feignResJson.getString("id"));
        onlineRecordEntity.setSourceFileUrl(feignResJson.getString("url"));
        onlineRecordEntity.setContractNumber(contractNumber);
        onlineRecordEntity.setBusinessNumber(form.getBusinessNumber());
        onlineRecordEntity.setCustomerId(form.getDriverId());
        onlineRecordEntity.setCustomerSignTime(new Date());
        onlineRecordEntity.setSignComplete(ContractConstants.CONTRACT_SIGN_COMPLETE_NO);
        this.baseMapper.insert(onlineRecordEntity);

        // 6:异步保存线上合同目标文件（PDF）及信息
        this.asyncSaveTargetFileInfo(MDC.get("traceId"), onlineRecordEntity, signBytes);

        // 7:返回
        ContractOnlineRecordSignDriverVO vo = new ContractOnlineRecordSignDriverVO();
        vo.setContractNumber(contractNumber);
        return vo;
    }

    @Override
    public ContractOnlineRecordVO getContractInfoByNumber(String contractNumber) {
        ContractOnlineRecordEntity recordEntity = this.baseMapper.selectOne(new LambdaQueryWrapper<ContractOnlineRecordEntity>()
                .eq(ContractOnlineRecordEntity::getContractNumber, contractNumber));
        if (null == recordEntity) {
            throw new EException(1001, errorMessageComponent.getContractOnlineGetByNumber1001());
        }
        ContractOnlineRecordVO vo = new ContractOnlineRecordVO();
        BeanUtils.copyProperties(recordEntity, vo);

        // 判断是货主还是司机签订的合同，并获取客户的名称
        String customerName = null;
        if (ContractConstants.CONTRACT_TYPE_GOODS_OWNER.equals(recordEntity.getContractType())) {
            AccountForm accountForm = new AccountForm();
            accountForm.setId(recordEntity.getCustomerId());
            AccountInfoDto accountInfo = goodsOwnerAccountService.getAccountInfo(accountForm);
            if (null != accountInfo) {
                if (GoodsOwnerConstants.OWNER_TYPE_COMPANY.equals(accountInfo.getOwnerType())) {
                    customerName = accountInfo.getOwnerFullName();
                } else {
                    customerName = accountInfo.getContactor();
                }
            }
        } else {
            DriverDetailDto driverDetailDto = driverAccountService.getDriverDetailById(recordEntity.getCustomerId());
            customerName = driverDetailDto.getName();
        }
        vo.setCustomerName(customerName);
        vo.setContractFileId(recordEntity.getTargetFileId());
        vo.setContractFileUrl(recordEntity.getTargetFileUrl());
        vo.setCustomerSignTime(recordEntity.getCustomerSignTime().getTime());
        vo.setPlatformSignTime(null == recordEntity.getPlatformSignTime() ? null : recordEntity.getPlatformSignTime().getTime());
        vo.setEffectiveTime(null == recordEntity.getEffectiveTime() ? null : recordEntity.getEffectiveTime().getTime());

        return vo;
    }

    /**
     * description 获取货主签订合同时的数据
     * param [form, accountInfo]
     * return java.util.Map<java.lang.String,java.lang.Object>
     * author Administrator
     * createTime 2020/05/21 19:32
     **/
    private ContractOnlineGoodsOwnerPojo getGoodsOwnerSignDataMap(ContractOnlineRecordSignGoodsOwnerForm form, AccountInfoDto accountInfo) {
        ContractOnlineGoodsOwnerPojo pojo = new ContractOnlineGoodsOwnerPojo();
        BeanUtils.copyProperties(form, pojo);

        // 设置合同编号
        JSONObject feignReqJson = new JSONObject();
        feignReqJson.put("prefix", "HZHT");
        feignReqJson.put("formatter", "yyyyMMdd");
        feignReqJson.put("length", 18);
        JSONObject feignResJson;
        try {
            feignResJson = FeignUtils.getFeignResultData(feignBaseService.getBatchId(feignReqJson));
            pojo.setContactNumber(feignResJson.getString("batchId"));
        } catch (Exception e) {
            log.error("调用[基础服务]获取合同编号失败：" + e.getMessage());
            throw new EException("生成电子合同失败[获取合同编号失败]");
        }

        // 设置货主信息
        if (GoodsOwnerConstants.OWNER_TYPE_COMPANY.equals(accountInfo.getOwnerType())) {
            pojo.setGoodsOwnerName(accountInfo.getOwnerFullName());
            pojo.setGoodsOwnerId(accountInfo.getUniCreditCode());
        } else {
            pojo.setGoodsOwnerName(accountInfo.getContactor());
            pojo.setGoodsOwnerId(accountInfo.getIdCard());
        }
        pojo.setPlatformName("{{platformName}}");
        pojo.setPlatformId("{{platformId}}");
        pojo.setEffectiveTime("{{effectiveTime}}");
        pojo.setGoodsOwnerSignImg(new PictureRenderData(CUSTOMER_SIGN_WIDTH, CUSTOMER_SIGN_HEIGHT, ".png", Base64.getDecoder().decode(form.getSignData())));
        pojo.setPlatformFreightSealImg("{{@platformFreightSealImg}}");

        return pojo;
    }

    /**
     * description 获取司机签订合同时的数据
     * param [form, accountInfo]
     * return com.esv.freight.customer.module.contract.pojo.ContractOnlineDriverPojo
     * author Administrator
     * createTime 2020/05/26 15:16
     **/
    private ContractOnlineDriverPojo getDriverSignDataMap(ContractOnlineRecordSignDriverForm form, DriverDetailDto accountInfo) {
        ContractOnlineDriverPojo pojo = new ContractOnlineDriverPojo();
        BeanUtils.copyProperties(form, pojo);

        // 设置合同编号
        JSONObject feignReqJson = new JSONObject();
        feignReqJson.put("prefix", "SJHT");
        feignReqJson.put("formatter", "yyyyMMdd");
        feignReqJson.put("length", 18);
        JSONObject feignResJson;
        try {
            feignResJson = FeignUtils.getFeignResultData(feignBaseService.getBatchId(feignReqJson));
            pojo.setContactNumber(feignResJson.getString("batchId"));
        } catch (Exception e) {
            log.error("调用[基础服务]获取合同编号失败：" + e.getMessage());
            throw new EException("生成电子合同失败[获取合同编号失败]");
        }

        // 设置司机信息
        pojo.setDriverId(accountInfo.getIdCard());
        pojo.setDriverName(accountInfo.getName());
        pojo.setPlatformName("{{platformName}}");
        pojo.setPlatformId("{{platformId}}");
        pojo.setEffectiveTime("{{effectiveTime}}");
        pojo.setDriverSignImg(new PictureRenderData(CUSTOMER_SIGN_WIDTH, CUSTOMER_SIGN_HEIGHT, ".png", Base64.getDecoder().decode(form.getSignData())));
        pojo.setPlatformFreightSealImg("{{@platformFreightSealImg}}");

        return pojo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Integer platformSign(String contractNumber) {
        // 1:判断合同号是否有效
        ContractOnlineRecordEntity contractOnlineRecordEntity = this.baseMapper.selectOne(new LambdaQueryWrapper<ContractOnlineRecordEntity>()
                .eq(ContractOnlineRecordEntity::getContractNumber, contractNumber));
        if (null == contractOnlineRecordEntity) {
            throw new EException(1001, errorMessageComponent.getContractOnlinePlatformSign1001());
        }

        // 2:获取合同数据
        JSONObject feignReqJson = new JSONObject();
        feignReqJson.put("id", contractOnlineRecordEntity.getSourceFileId());
        JSONObject feignResJson;
        try {
            feignResJson = FeignUtils.getFeignResultData(feignFileService.getFileData(feignReqJson));
        } catch (Exception e) {
            log.error("调用[文件服务]获取合同数据失败：" + e.getMessage());
            throw new EException("平台签订电子合同失败[获取合同数据失败]");
        }
        String contractFileData = feignResJson.getString("fileData");
        byte[] contractFileBytes = Base64.getDecoder().decode(contractFileData);

        // 3:获取平台信息
        ContractOnlinePlatformPojo pojo = this.getContractOnlinePlatformData();
        Date effectiveTime = new Date();
        pojo.setEffectiveTime(DateUtils.format(effectiveTime));
        pojo.setPlatformFreightSealImg(new PictureRenderData(PLATFORM_SEAL_WIDTH, PLATFORM_SEAL_HEIGHT, ".png", Base64.getDecoder().decode((String) pojo.getPlatformFreightSealImg())));

        // 4:增加平台签名或盖章
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        byte[] signBytes;
        XWPFTemplate template = null;
        try {
            template = XWPFTemplate.compile(InputStreamUtils.byte2InputStream(contractFileBytes));
            template.render(pojo);
            template.write(byteArrayOutputStream);
            byteArrayOutputStream.flush();
            signBytes = byteArrayOutputStream.toByteArray();
        } catch (IOException e) {
            log.error("平台签订电子合同失败：" + e.getMessage());
            throw new EException(ECode.SERVER_ERROR.code(), "平台签订电子合同失败[填充平台签名失败]");
        } finally {
            if (null != template) {
                try {
                    template.close();
                } catch (IOException e) {
                    log.error(e.getMessage(), e);
                }
            }
            IOUtils.closeQuietly(byteArrayOutputStream);
        }

        // 5:保存平台签名文件
        feignReqJson = new JSONObject();
        feignReqJson.put("fileType", "word");
        feignReqJson.put("fileName", "电子合同-平台签名-" + contractNumber + ".docx");
        feignReqJson.put("fileData", Base64.getEncoder().encodeToString(signBytes));
        try {
            feignResJson = FeignUtils.getFeignResultData(feignFileService.uploadSingleFile(feignReqJson));
        } catch (Exception e) {
            log.error("调用[文件服务]上传平台签名合同文件数据失败：" + e.getMessage());
            throw new EException("平台签订电子合同失败[上传平台签名合同文件数据失败]");
        }

        // 6:更新合同记录
        ContractOnlineRecordEntity onlineRecordEntity = new ContractOnlineRecordEntity();
        onlineRecordEntity.setId(contractOnlineRecordEntity.getId());
        onlineRecordEntity.setSourceFileId(feignResJson.getString("id"));
        onlineRecordEntity.setSourceFileUrl(feignResJson.getString("url"));
        onlineRecordEntity.setPlatformSignTime(effectiveTime);
        onlineRecordEntity.setEffectiveTime(effectiveTime);
        onlineRecordEntity.setSignComplete(ContractConstants.CONTRACT_SIGN_COMPLETE_YES);
        int flag = this.baseMapper.updateById(onlineRecordEntity);

        // 7:异步保存线上合同目标文件（PDF）及信息
        this.asyncSaveTargetFileInfo(MDC.get("traceId"), contractOnlineRecordEntity, signBytes);

        return flag;
    }

    /**
     * 获取电子合同模板数据
     * contractType：合同类型：1-货主与平台合同、2-司机与平台合同
     **/
    private String getContractOnlineTemplateEntity(Integer contractType) {
        String contractTemplateData;
        String cacheKey = applicationName + "::online-contract::" + contractType + "::" + ReqUtils.getRequestTenantId();
        contractTemplateData = (String) redisComponent.get(cacheKey);
        if (StringUtils.isBlank(contractTemplateData)) {
            ContractOnlineTemplateEntity entity = this.contractOnlineTemplateService.getUsedTemplate(contractType);
            String fileId = entity.getTemplateFileId();

            JSONObject feignReqJson = new JSONObject();
            feignReqJson.put("id", fileId);
            JSONObject feignResJson;
            try {
                feignResJson = FeignUtils.getFeignResultData(feignFileService.getFileData(feignReqJson));
            } catch (Exception e) {
                log.error("调用[文件服务]获取合同模板数据失败：" + e.getMessage());
                throw new EException("生成电子合同失败[获取合同模板数据失败]");
            }
            contractTemplateData = feignResJson.getString("fileData");
            redisComponent.set(cacheKey, contractTemplateData, CONTRACT_CACHE_TIME);
        }

        return contractTemplateData;
    }

    /**
     * 获取线上电子合同平台数据
     **/
    private ContractOnlinePlatformPojo getContractOnlinePlatformData() {
        ContractOnlinePlatformPojo pojo;
        String cacheKey = applicationName + "::online-contract::platform-data::" + ReqUtils.getRequestTenantId();
        String cacheValue = (String) redisComponent.get(cacheKey);
        if (StringUtils.isBlank(cacheValue)) {
            ContractOnlinePlatformDataEntity entity = this.contractOnlinePlatformDataService.getUsedData();
            String fileId = entity.getSealFileId();

            JSONObject feignReqJson = new JSONObject();
            feignReqJson.put("id", fileId);
            JSONObject feignResJson;
            try {
                feignResJson = FeignUtils.getFeignResultData(feignFileService.getFileData(feignReqJson));
            } catch (Exception e) {
                log.error("调用[文件服务]获取合同平台数据失败：" + e.getMessage());
                throw new EException("生成电子合同失败[获取合同平台数据失败]");
            }
            String fileData = feignResJson.getString("fileData");

            pojo = new ContractOnlinePlatformPojo();
            pojo.setPlatformFreightSealImg(fileData);
            pojo.setPlatformName(entity.getPlatformName());
            pojo.setPlatformId(entity.getPlatformId());
            redisComponent.set(cacheKey, pojo.toString(), CONTRACT_CACHE_TIME);
        } else {
            pojo = JSONObject.toJavaObject(JSONObject.parseObject(cacheValue), ContractOnlinePlatformPojo.class);
        }

        return pojo;
    }

    /**
     * description 异步保存线上合同目标文件（PDF）及信息
     * param traceId：logback日志ID
     * param entity：线上合同记录
     * param fileBytes：线上合同源文件数据
     * return void
     * author Administrator
     * createTime 2020/05/26 9:34
     **/
    @Async
    public void asyncSaveTargetFileInfo(String traceId, ContractOnlineRecordEntity entity, byte[] fileBytes) {
        String contractNumber = entity.getContractNumber();
        if (StringUtils.isEmpty(traceId) || CommonConstants.NULL_STRING.equalsIgnoreCase(traceId)) {
            traceId = UUID.randomUUID().toString().replace("-", "");
        }
        MDC.put("traceId", traceId);
        log.info("------异步保存合同[{}]目标文件及信息开始------", contractNumber);

        // 转换源文件为PDF
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        InputStream inputStream = InputStreamUtils.byte2InputStream(fileBytes);
        byte[] targetFileBytes;
        try {
            Document document = new Document(inputStream);
            document.save(outputStream, SaveFormat.PDF);
            targetFileBytes = outputStream.toByteArray();
        } catch (Exception e) {
            log.error("合同[{}]转换为PDF时失败：{}", contractNumber, e.getMessage());
            log.info("------异步保存合同[{}]目标文件及信息结束------", contractNumber);
            return;
        } finally {
            IOUtils.closeQuietly(inputStream);
            IOUtils.closeQuietly(outputStream);
        }

        // 保存PDF文件
        JSONObject feignReqJson = new JSONObject();
        feignReqJson.put("fileType", "pdf");
        feignReqJson.put("fileName", "电子合同-" + contractNumber + ".pdf");
        feignReqJson.put("fileData", Base64.getEncoder().encodeToString(targetFileBytes));
        JSONObject feignResJson;
        try {
            feignResJson = FeignUtils.getFeignResultData(feignFileService.uploadSingleFile(feignReqJson));
        } catch (Exception e) {
            log.error("调用[文件服务]上传目标合同[{}]文件数据失败：{}", contractNumber, e.getMessage());
            log.info("------异步保存合同[{}]目标文件及信息结束------", contractNumber);
            return;
        }

        // 更新合同记录信息
        ContractOnlineRecordEntity onlineRecordEntity = new ContractOnlineRecordEntity();
        onlineRecordEntity.setId(entity.getId());
        onlineRecordEntity.setTargetFileId(feignResJson.getString("id"));
        onlineRecordEntity.setTargetFileUrl(feignResJson.getString("url"));
        this.baseMapper.updateById(onlineRecordEntity);

        log.info("------异步保存合同[{}]目标文件及信息结束------", contractNumber);
    }

}