package com.esv.freight.customer.module.driver.controller;

import com.alibaba.fastjson.JSONObject;
import com.esv.freight.customer.BaseTestController;
import com.esv.freight.customer.common.response.ECode;
import com.esv.freight.customer.module.driver.form.CarrierForm;
import com.esv.freight.customer.module.driver.form.DriverQueryForm;
import com.esv.freight.customer.module.driver.form.DriverVehicleForm;
import com.esv.freight.customer.module.vehicle.form.VehicleQueryForm;
import lombok.extern.slf4j.Slf4j;
import org.junit.Assert;
import org.junit.FixMethodOrder;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.MethodSorters;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.http.MediaType;
import org.springframework.test.annotation.Rollback;
import org.springframework.test.context.junit4.SpringRunner;
import org.springframework.test.web.servlet.MvcResult;
import org.springframework.test.web.servlet.request.MockMvcRequestBuilders;
import org.springframework.test.web.servlet.result.MockMvcResultHandlers;
import org.springframework.test.web.servlet.result.MockMvcResultMatchers;
import org.springframework.transaction.annotation.Transactional;

/**
 * @description:
 * @project: freight-customer-service
 * @name: com.esv.freight.customer.module.driver.controller.DriverVehicleControllerTest
 * @author: 黄朝斌
 * @email: huangchaobin@esvtek.com
 * @createTime: 2020/04/29 19:17
 * @version:1.0
 */
@RunWith(SpringRunner.class)
@SpringBootTest
@Slf4j
@FixMethodOrder(MethodSorters.NAME_ASCENDING)
@Transactional
@Rollback(false)
public class DriverVehicleControllerTest extends BaseTestController {

    /**
     * 查询指定司机绑定的车辆列表
     **/
    @Test
    public void a1_listByDriver_success_test() throws Exception {
        String url = "/carrier/driver/vehicle/listByDriver";

        // 构造数据
        DriverQueryForm form = new DriverQueryForm();
        form.setId(1L);

        MvcResult mvcResult = this.getMockMvc().perform(MockMvcRequestBuilders.post(url)
                .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
                .headers(this.getDefaultHttpHeaders())
                .content(form.toString()))
                .andDo(MockMvcResultHandlers.print())
                .andExpect(MockMvcResultMatchers.status().isOk())
                .andReturn();

        String responseStr = mvcResult.getResponse().getContentAsString();
        log.info(responseStr);

        JSONObject result = JSONObject.parseObject(responseStr);
        Assert.assertEquals(ECode.SUCCESS.code(), result.getIntValue("code"));
        Assert.assertTrue(null != result.getJSONArray("data"));
    }

    /**
     * 查询指定司机绑定的车辆列表：无效的司机ID
     **/
    @Test
    public void a2_listByDriver_wrong_id_failure_test() throws Exception {
        String url = "/carrier/driver/vehicle/listByDriver";

        // 构造数据
        DriverQueryForm form = new DriverQueryForm();
        form.setId(99999L);

        MvcResult mvcResult = this.getMockMvc().perform(MockMvcRequestBuilders.post(url)
                .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
                .headers(this.getDefaultHttpHeaders())
                .content(form.toString()))
                .andDo(MockMvcResultHandlers.print())
                .andExpect(MockMvcResultMatchers.status().isOk())
                .andReturn();

        String responseStr = mvcResult.getResponse().getContentAsString();
        log.info(responseStr);

        JSONObject result = JSONObject.parseObject(responseStr);
        Assert.assertEquals(1001, result.getIntValue("code"));
    }

    /**
     * 查询指定司机绑定的车辆列表：该司机未绑定车辆
     **/
    @Test
    public void a3_listByDriver_not_bind_vehicle_failure_test() throws Exception {
        String url = "/carrier/driver/vehicle/listByDriver";

        // 构造数据
        DriverQueryForm form = new DriverQueryForm();
        form.setId(2L);

        MvcResult mvcResult = this.getMockMvc().perform(MockMvcRequestBuilders.post(url)
                .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
                .headers(this.getDefaultHttpHeaders())
                .content(form.toString()))
                .andDo(MockMvcResultHandlers.print())
                .andExpect(MockMvcResultMatchers.status().isOk())
                .andReturn();

        String responseStr = mvcResult.getResponse().getContentAsString();
        log.info(responseStr);

        JSONObject result = JSONObject.parseObject(responseStr);
        Assert.assertEquals(1002, result.getIntValue("code"));
    }

    /**
     * 绑定司机与车辆
     **/
    @Test
    @Rollback
    public void b1_bind_success_test() throws Exception {
        String url = "/carrier/driver/vehicle/bind";

        // 构造数据
        DriverVehicleForm form = new DriverVehicleForm();
        form.setDriverId(3L);
        form.setVehicleId(4L);

        MvcResult mvcResult = this.getMockMvc().perform(MockMvcRequestBuilders.post(url)
                .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
                .headers(this.getDefaultHttpHeaders())
                .content(form.toString()))
                .andDo(MockMvcResultHandlers.print())
                .andExpect(MockMvcResultMatchers.status().isOk())
                .andReturn();

        String responseStr = mvcResult.getResponse().getContentAsString();
        log.info(responseStr);

        JSONObject result = JSONObject.parseObject(responseStr);
        Assert.assertEquals(ECode.SUCCESS.code(), result.getIntValue("code"));
    }

    /**
     * 解绑司机与车辆
     **/
    @Test
    @Rollback
    public void c1_unbind_success_test() throws Exception {
        String url = "/carrier/driver/vehicle/unbind";

        // 构造数据
        DriverVehicleForm form = new DriverVehicleForm();
        form.setDriverId(3L);
        form.setVehicleId(4L);

        MvcResult mvcResult = this.getMockMvc().perform(MockMvcRequestBuilders.post(url)
                .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
                .headers(this.getDefaultHttpHeaders())
                .content(form.toString()))
                .andDo(MockMvcResultHandlers.print())
                .andExpect(MockMvcResultMatchers.status().isOk())
                .andReturn();

        String responseStr = mvcResult.getResponse().getContentAsString();
        log.info(responseStr);

        JSONObject result = JSONObject.parseObject(responseStr);
        Assert.assertEquals(ECode.SUCCESS.code(), result.getIntValue("code"));
    }

    /**
     * 解绑司机与车辆：司机自主绑定的车辆，平台不可解绑
     **/
    @Test
    @Rollback
    public void c2_unbind_driver_add_vehicle_failure_test() throws Exception {
        String url = "/carrier/driver/vehicle/unbind";

        // 构造数据
        DriverVehicleForm form = new DriverVehicleForm();
        form.setDriverId(3L);
        form.setVehicleId(4L);

        MvcResult mvcResult = this.getMockMvc().perform(MockMvcRequestBuilders.post(url)
                .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
                .headers(this.getDefaultHttpHeaders())
                .content(form.toString()))
                .andDo(MockMvcResultHandlers.print())
                .andExpect(MockMvcResultMatchers.status().isOk())
                .andReturn();

        String responseStr = mvcResult.getResponse().getContentAsString();
        log.info(responseStr);

        JSONObject result = JSONObject.parseObject(responseStr);
        Assert.assertEquals(1021, result.getIntValue("code"));
    }

    /**
     * 解绑司机与车辆：车辆与司机未绑定​
     **/
    @Test
    @Rollback
    public void c3_unbind_not_bind_failure_test() throws Exception {
        String url = "/carrier/driver/vehicle/unbind";

        // 构造数据
        DriverVehicleForm form = new DriverVehicleForm();
        form.setDriverId(3L);
        form.setVehicleId(3L);

        MvcResult mvcResult = this.getMockMvc().perform(MockMvcRequestBuilders.post(url)
                .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
                .headers(this.getDefaultHttpHeaders())
                .content(form.toString()))
                .andDo(MockMvcResultHandlers.print())
                .andExpect(MockMvcResultMatchers.status().isOk())
                .andReturn();

        String responseStr = mvcResult.getResponse().getContentAsString();
        log.info(responseStr);

        JSONObject result = JSONObject.parseObject(responseStr);
        Assert.assertEquals(1020, result.getIntValue("code"));
    }

    /**
     * 验证司机与车辆是否绑定
     **/
    @Test
    @Rollback
    public void d1_checkBind_success_test() throws Exception {
        String url = "/carrier/driver/vehicle/checkBind";

        // 构造数据
        DriverVehicleForm form = new DriverVehicleForm();
        form.setDriverId(3L);
        form.setVehicleId(4L);

        MvcResult mvcResult = this.getMockMvc().perform(MockMvcRequestBuilders.post(url)
                .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
                .headers(this.getDefaultHttpHeaders())
                .content(form.toString()))
                .andDo(MockMvcResultHandlers.print())
                .andExpect(MockMvcResultMatchers.status().isOk())
                .andReturn();

        String responseStr = mvcResult.getResponse().getContentAsString();
        log.info(responseStr);

        JSONObject result = JSONObject.parseObject(responseStr);
        Assert.assertEquals(ECode.SUCCESS.code(), result.getIntValue("code"));
        Assert.assertTrue(result.getJSONObject("data").containsKey("driverInfo"));
        Assert.assertTrue(result.getJSONObject("data").containsKey("vehicleInfo"));
    }

    /**
     * 查询指定承运商的司机与车辆绑定列表：不指定承运商
     **/
    @Test
    public void e1_listByCarrier_success_test() throws Exception {
        String url = "/carrier/driver/vehicle/listByCarrier";

        // 构造数据
        CarrierForm form = new CarrierForm();

        MvcResult mvcResult = this.getMockMvc().perform(MockMvcRequestBuilders.post(url)
                .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
                .headers(this.getDefaultHttpHeaders())
                .content(form.toString()))
                .andDo(MockMvcResultHandlers.print())
                .andExpect(MockMvcResultMatchers.status().isOk())
                .andReturn();

        String responseStr = mvcResult.getResponse().getContentAsString();
        log.info(responseStr);

        JSONObject result = JSONObject.parseObject(responseStr);
        Assert.assertEquals(ECode.SUCCESS.code(), result.getIntValue("code"));
        Assert.assertTrue(null != result.getJSONArray("data"));
    }

    /**
     * 查询指定承运商的司机与车辆绑定列表：指定承运商ID存在
     **/
    @Test
    public void e2_listByCarrier_with_right_id_success_test() throws Exception {
        String url = "/carrier/driver/vehicle/listByCarrier";

        // 构造数据
        CarrierForm form = new CarrierForm();
        form.setId(2L);

        MvcResult mvcResult = this.getMockMvc().perform(MockMvcRequestBuilders.post(url)
                .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
                .headers(this.getDefaultHttpHeaders())
                .content(form.toString()))
                .andDo(MockMvcResultHandlers.print())
                .andExpect(MockMvcResultMatchers.status().isOk())
                .andReturn();

        String responseStr = mvcResult.getResponse().getContentAsString();
        log.info(responseStr);

        JSONObject result = JSONObject.parseObject(responseStr);
        Assert.assertEquals(ECode.SUCCESS.code(), result.getIntValue("code"));
        Assert.assertTrue(null != result.getJSONArray("data"));
    }

    /**
     * 查询指定承运商的司机与车辆绑定列表：指定承运商ID不存在
     **/
    @Test
    public void e3_listByCarrier_with_right_id_success_test() throws Exception {
        String url = "/carrier/driver/vehicle/listByCarrier";

        // 构造数据
        CarrierForm form = new CarrierForm();
        form.setId(99999L);

        MvcResult mvcResult = this.getMockMvc().perform(MockMvcRequestBuilders.post(url)
                .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
                .headers(this.getDefaultHttpHeaders())
                .content(form.toString()))
                .andDo(MockMvcResultHandlers.print())
                .andExpect(MockMvcResultMatchers.status().isOk())
                .andReturn();

        String responseStr = mvcResult.getResponse().getContentAsString();
        log.info(responseStr);

        JSONObject result = JSONObject.parseObject(responseStr);
        Assert.assertEquals(ECode.SUCCESS.code(), result.getIntValue("code"));
        Assert.assertTrue(0 == result.getJSONArray("data").size());
    }

    /**
     * 查询指定车辆绑定的司机列表
     **/
    @Test
    public void f1_listByVehicle_success_test() throws Exception {
        String url = "/carrier/driver/vehicle/listByVehicle";

        // 构造数据
        VehicleQueryForm form = new VehicleQueryForm();
        form.setId(3L);

        MvcResult mvcResult = this.getMockMvc().perform(MockMvcRequestBuilders.post(url)
                .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
                .headers(this.getDefaultHttpHeaders())
                .content(form.toString()))
                .andDo(MockMvcResultHandlers.print())
                .andExpect(MockMvcResultMatchers.status().isOk())
                .andReturn();

        String responseStr = mvcResult.getResponse().getContentAsString();
        log.info(responseStr);

        JSONObject result = JSONObject.parseObject(responseStr);
        Assert.assertEquals(ECode.SUCCESS.code(), result.getIntValue("code"));
        Assert.assertTrue(null != result.getJSONArray("data"));
    }

    /**
     * 查询指定车辆绑定的司机列表：无效的车辆ID
     **/
    @Test
    public void f2_listByVehicle_wrong_id_failure_test() throws Exception {
        String url = "/carrier/driver/vehicle/listByVehicle";

        // 构造数据
        VehicleQueryForm form = new VehicleQueryForm();
        form.setId(99999L);

        MvcResult mvcResult = this.getMockMvc().perform(MockMvcRequestBuilders.post(url)
                .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
                .headers(this.getDefaultHttpHeaders())
                .content(form.toString()))
                .andDo(MockMvcResultHandlers.print())
                .andExpect(MockMvcResultMatchers.status().isOk())
                .andReturn();

        String responseStr = mvcResult.getResponse().getContentAsString();
        log.info(responseStr);

        JSONObject result = JSONObject.parseObject(responseStr);
        Assert.assertEquals(1001, result.getIntValue("code"));
    }

    /**
     * 查询指定车辆绑定的司机列表：该车辆未绑定司机​
     **/
    @Test
    public void f3_listByVehicle_not_bind_driver_failure_test() throws Exception {
        String url = "/carrier/driver/vehicle/listByVehicle";

        // 构造数据
        VehicleQueryForm form = new VehicleQueryForm();
        form.setId(2L);

        MvcResult mvcResult = this.getMockMvc().perform(MockMvcRequestBuilders.post(url)
                .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
                .headers(this.getDefaultHttpHeaders())
                .content(form.toString()))
                .andDo(MockMvcResultHandlers.print())
                .andExpect(MockMvcResultMatchers.status().isOk())
                .andReturn();

        String responseStr = mvcResult.getResponse().getContentAsString();
        log.info(responseStr);

        JSONObject result = JSONObject.parseObject(responseStr);
        Assert.assertEquals(1002, result.getIntValue("code"));
    }
}
