package com.esv.freight.customer.module.driver.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.esv.freight.customer.common.component.ErrorMessageComponent;
import com.esv.freight.customer.common.component.PasswordComponent;
import com.esv.freight.customer.common.exception.EException;
import com.esv.freight.customer.common.util.ReqUtils;
import com.esv.freight.customer.common.vo.PageResultVO;
import com.esv.freight.customer.module.carrier.CarrierConstants;
import com.esv.freight.customer.module.carrier.entity.CarrierAccountEntity;
import com.esv.freight.customer.module.carrier.service.CarrierAccountService;
import com.esv.freight.customer.module.driver.DriverConstants;
import com.esv.freight.customer.module.driver.dao.DriverAccountDao;
import com.esv.freight.customer.module.driver.dto.DriverDetailDto;
import com.esv.freight.customer.module.driver.entity.DriverAccountEntity;
import com.esv.freight.customer.module.driver.entity.DriverAuditHistoryEntity;
import com.esv.freight.customer.module.driver.entity.DriverInfoEntity;
import com.esv.freight.customer.module.driver.form.DriverAuditForm;
import com.esv.freight.customer.module.driver.form.DriverInfoForm;
import com.esv.freight.customer.module.driver.form.DriverQueryForm;
import com.esv.freight.customer.module.driver.service.DriverAccountService;
import com.esv.freight.customer.module.driver.service.DriverAuditHistoryService;
import com.esv.freight.customer.module.driver.service.DriverInfoService;
import com.esv.freight.customer.module.driver.vo.DriverListVO;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;


@Service("driverAccountService")
public class DriverAccountServiceImpl extends ServiceImpl<DriverAccountDao, DriverAccountEntity> implements DriverAccountService {

    private ErrorMessageComponent errorMessageComponent;

    private PasswordComponent passwordComponent;

    private DriverInfoService driverInfoService;

    private DriverAuditHistoryService driverAuditHistoryService;

    private CarrierAccountService carrierAccountService;

    @Autowired
    public DriverAccountServiceImpl(ErrorMessageComponent errorMessageComponent, PasswordComponent passwordComponent,
                                    DriverInfoService driverInfoService, DriverAuditHistoryService driverAuditHistoryService,
                                    CarrierAccountService carrierAccountService) {
        this.errorMessageComponent = errorMessageComponent;
        this.passwordComponent = passwordComponent;
        this.driverInfoService = driverInfoService;
        this.driverAuditHistoryService = driverAuditHistoryService;
        this.carrierAccountService = carrierAccountService;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long insertDriver(DriverInfoForm form) {
        // 1:校验
        // 1.1:校验账号是否已存在
        Integer count = this.baseMapper.selectCount(new QueryWrapper<DriverAccountEntity>().lambda()
                .eq(DriverAccountEntity::getAccount, form.getAccount()));
        if (0 < count) {
            throw new EException(1001, errorMessageComponent.getCarrierDriverAdd1001());
        }
        // 1.2:校验身份证号码是否已存在
        count = this.driverInfoService.getBaseMapper().selectCount(new QueryWrapper<DriverInfoEntity>().lambda()
                .eq(DriverInfoEntity::getIdCard, form.getIdCard()));
        if (0 < count) {
            throw new EException(1002, errorMessageComponent.getCarrierDriverAdd1002());
        }
        // 1.3:校验驾驶证号码是否已存在
        count = this.driverInfoService.getBaseMapper().selectCount(new QueryWrapper<DriverInfoEntity>().lambda()
                .eq(DriverInfoEntity::getDrivingLicense, form.getDrivingLicense()));
        if (0 < count) {
            throw new EException(1003, errorMessageComponent.getCarrierDriverAdd1003());
        }
        // 1.4:校验从业资格证号是否已存在
        if (StringUtils.isNotBlank(form.getCertificateNumber())) {
            count = this.driverInfoService.getBaseMapper().selectCount(new QueryWrapper<DriverInfoEntity>().lambda()
                    .eq(DriverInfoEntity::getCertificateNumber, form.getCertificateNumber()));
            if (0 < count) {
                throw new EException(1004, errorMessageComponent.getCarrierDriverAdd1004());
            }
        }
        // 1.5:校验承运商帐号ID是否有效
        CarrierAccountEntity carrierAccountEntity = this.carrierAccountService.getCarrierById(form.getCarrierId());
        if (null == carrierAccountEntity) {
            throw new EException(1005, errorMessageComponent.getCarrierDriverAdd1005());
        }

        // 2:新增司机帐号
        DriverAccountEntity driverAccountEntity = new DriverAccountEntity();
        BeanUtils.copyProperties(form, driverAccountEntity);
        String salt = passwordComponent.generateAccountPwdSalt();
        driverAccountEntity.setSalt(salt);
        driverAccountEntity.setPassword(passwordComponent.generatePwd4Salt(form.getPassword(), salt));
        driverAccountEntity.setAccountStatus(DriverConstants.ACCOUNT_STATUS_UNBLOCK);
        driverAccountEntity.setSourceType(DriverConstants.ACCOUNT_SOURCE_TYPE_PLATFORM);
        driverAccountEntity.setAuditStatus(DriverConstants.ACCOUNT_AUDIT_STATUS_SUCCESS);
        this.baseMapper.insert(driverAccountEntity);
        Long driverId = driverAccountEntity.getId();

        // 3:新增司机信息
        DriverInfoEntity driverInfoEntity = new DriverInfoEntity();
        BeanUtils.copyProperties(form, driverInfoEntity);
        driverInfoEntity.setDriverId(driverId);
        this.driverInfoService.getBaseMapper().insert(driverInfoEntity);

        // 4:新增审核记录
        DriverAuditHistoryEntity driverAuditHistoryEntity = new DriverAuditHistoryEntity();
        driverAuditHistoryEntity.setCarrierId(form.getCarrierId());
        driverAuditHistoryEntity.setDriverId(driverId);
        driverAuditHistoryEntity.setAuditStatus(DriverConstants.ACCOUNT_AUDIT_STATUS_SUCCESS);
        driverAuditHistoryEntity.setRemark("平台创建自动审核");
        driverAuditHistoryEntity.setOperateUser(ReqUtils.getRequestUserAccount());
        this.driverAuditHistoryService.getBaseMapper().insert(driverAuditHistoryEntity);

        return driverId;
    }

    @Override
    public Integer updateDriver(DriverInfoForm form) {
        // 1:校验
        // 1.1:校验账号ID是否有效
        DriverAccountEntity driverAccountEntity = this.baseMapper.selectById(form.getId());
        if (null == driverAccountEntity) {
            throw new EException(1001, errorMessageComponent.getCarrierDriverEdit1001());
        }
        // 1.2:校验身份证号码是否已存在
        Integer count = this.driverInfoService.getBaseMapper().selectCount(new QueryWrapper<DriverInfoEntity>().lambda()
                .ne(DriverInfoEntity::getDriverId, form.getId())
                .eq(DriverInfoEntity::getIdCard, form.getIdCard()));
        if (0 < count) {
            throw new EException(1002, errorMessageComponent.getCarrierDriverEdit1002());
        }
        // 1.3:校验驾驶证号码是否已存在
        count = this.driverInfoService.getBaseMapper().selectCount(new QueryWrapper<DriverInfoEntity>().lambda()
                .ne(DriverInfoEntity::getDriverId, form.getId())
                .eq(DriverInfoEntity::getDrivingLicense, form.getDrivingLicense()));
        if (0 < count) {
            throw new EException(1003, errorMessageComponent.getCarrierDriverEdit1003());
        }
        // 1.4:校验从业资格证号是否已存在
        if (StringUtils.isNotBlank(form.getCertificateNumber())) {
            count = this.driverInfoService.getBaseMapper().selectCount(new QueryWrapper<DriverInfoEntity>().lambda()
                    .ne(DriverInfoEntity::getDriverId, form.getId())
                    .eq(DriverInfoEntity::getCertificateNumber, form.getCertificateNumber()));
            if (0 < count) {
                throw new EException(1004, errorMessageComponent.getCarrierDriverEdit1004());
            }
        }

        // 2:更新密码
        int flag = 0;
        if (StringUtils.isNotBlank(form.getPassword())) {
            DriverAccountEntity accountEntity = new DriverAccountEntity();
            accountEntity.setId(form.getId());
            accountEntity.setPassword(passwordComponent.generatePwd4Salt(form.getPassword(), driverAccountEntity.getSalt()));
            flag += this.baseMapper.updateById(accountEntity);
        }

        // 3:更新帐号信息
        DriverInfoEntity driverInfoEntity = new DriverInfoEntity();
        BeanUtils.copyProperties(form, driverInfoEntity);
        flag += this.driverInfoService.getBaseMapper().update(driverInfoEntity, new UpdateWrapper<DriverInfoEntity>().lambda()
                .eq(DriverInfoEntity::getDriverId, form.getId()));

        return flag;
    }

    @Override
    public DriverDetailDto getDriverDetailById(Long id) {
        return this.baseMapper.selectDetailById(id);
    }

    @Override
    public DriverDetailDto getDriverDetailByAccount(String account) {
        return this.baseMapper.selectDetailByAccount(account);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Integer auditDriver(DriverAuditForm form) {
        // 1:校验帐号ID是否有效
        DriverAccountEntity accountEntity = this.baseMapper.selectById(form.getId());
        if (null == accountEntity) {
            throw new EException(1001, errorMessageComponent.getCarrierDriverAudit1001());
        }
        // 2:校验帐号是否审核通过
        if (DriverConstants.ACCOUNT_AUDIT_STATUS_SUCCESS.equals(accountEntity.getAuditStatus())) {
            throw new EException(1002, errorMessageComponent.getCarrierDriverAudit1002());
        }

        // 3:审核帐号
        DriverAccountEntity driverAccountEntity = new DriverAccountEntity();
        BeanUtils.copyProperties(form, driverAccountEntity);
        int flag = this.baseMapper.updateById(driverAccountEntity);

        // 4:增加审核记录
        DriverAuditHistoryEntity driverAuditHistoryEntity = new DriverAuditHistoryEntity();
        driverAuditHistoryEntity.setCarrierId(accountEntity.getCarrierId());
        driverAuditHistoryEntity.setDriverId(form.getId());
        driverAuditHistoryEntity.setAuditStatus(form.getAuditStatus());
        driverAuditHistoryEntity.setOperateUser(ReqUtils.getRequestUserAccount());
        driverAuditHistoryEntity.setRemark(form.getRemark());
        flag += this.driverAuditHistoryService.getBaseMapper().insert(driverAuditHistoryEntity);

        return flag;
    }

    @Override
    public Integer blockDriver(Long id) {
        // 1:校验帐号ID是否有效
        DriverAccountEntity accountEntity = this.baseMapper.selectById(id);
        if (null == accountEntity) {
            throw new EException(1001, errorMessageComponent.getCarrierDriverBlock1001());
        }
        // 2:校验帐号是否已停用
        if (DriverConstants.ACCOUNT_STATUS_BLOCK.equals(accountEntity.getAccountStatus())) {
            throw new EException(1002, errorMessageComponent.getCarrierDriverBlock1002());
        }

        // 3:停用帐号
        DriverAccountEntity driverAccountEntity = new DriverAccountEntity();
        driverAccountEntity.setId(id);
        driverAccountEntity.setAccountStatus(DriverConstants.ACCOUNT_STATUS_BLOCK);
        return this.baseMapper.updateById(driverAccountEntity);
    }

    @Override
    public Integer unblockDriver(Long id) {
        // 1:校验帐号ID是否有效
        DriverAccountEntity accountEntity = this.baseMapper.selectById(id);
        if (null == accountEntity) {
            throw new EException(1001, errorMessageComponent.getCarrierDriverUnblock1001());
        }
        // 2:校验帐号是否已启用
        if (DriverConstants.ACCOUNT_STATUS_UNBLOCK.equals(accountEntity.getAccountStatus())) {
            throw new EException(1002, errorMessageComponent.getCarrierDriverUnblock1002());
        }

        // 3:启用帐号
        DriverAccountEntity driverAccountEntity = new DriverAccountEntity();
        driverAccountEntity.setId(id);
        driverAccountEntity.setAccountStatus(DriverConstants.ACCOUNT_STATUS_UNBLOCK);
        return this.baseMapper.updateById(driverAccountEntity);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long registerDriver(DriverInfoForm form) {
        // 1:校验账号是否已存在
        Integer count = this.baseMapper.selectCount(new QueryWrapper<DriverAccountEntity>().lambda()
                .eq(DriverAccountEntity::getAccount, form.getAccount()));
        if (0 < count) {
            throw new EException(1001, errorMessageComponent.getCarrierDriverRegister1001());
        }

        // 2:校验承运商帐号ID是否有效
        CarrierAccountEntity carrierAccountEntity = this.carrierAccountService.getCarrierById(form.getCarrierId());
        if (null == carrierAccountEntity) {
            throw new EException(1002, errorMessageComponent.getCarrierDriverRegister1002());
        }

        // 3:新增司机帐号
        DriverAccountEntity driverAccountEntity = new DriverAccountEntity();
        driverAccountEntity.setAccount(form.getAccount());
        driverAccountEntity.setCarrierId(form.getCarrierId());
        driverAccountEntity.setSalt(passwordComponent.generateAccountPwdSalt());
        driverAccountEntity.setAccountStatus(DriverConstants.ACCOUNT_STATUS_BLOCK);
        driverAccountEntity.setSourceType(DriverConstants.ACCOUNT_SOURCE_TYPE_REGISTER);
        driverAccountEntity.setAuditStatus(DriverConstants.ACCOUNT_AUDIT_STATUS_UNAUDITED);
        this.baseMapper.insert(driverAccountEntity);
        Long driverId = driverAccountEntity.getId();

        // 4:新增司机信息
        DriverInfoEntity driverInfoEntity = new DriverInfoEntity();
        driverInfoEntity.setDriverId(driverId);
        driverInfoEntity.setCarrierId(form.getCarrierId());
        this.driverInfoService.getBaseMapper().insert(driverInfoEntity);

        // 5:新增审核记录
        DriverAuditHistoryEntity driverAuditHistoryEntity = new DriverAuditHistoryEntity();
        driverAuditHistoryEntity.setCarrierId(form.getCarrierId());
        driverAuditHistoryEntity.setDriverId(driverId);
        driverAuditHistoryEntity.setAuditStatus(DriverConstants.ACCOUNT_AUDIT_STATUS_UNAUDITED);
        driverAuditHistoryEntity.setRemark("用户通过终端APP注册帐号");
        driverAuditHistoryEntity.setOperateUser(ReqUtils.getRequestUserAccount());
        this.driverAuditHistoryService.getBaseMapper().insert(driverAuditHistoryEntity);

        return driverId;
    }

    @Override
    public List<DriverAuditHistoryEntity> getAuditHistory(Long id) {
        DriverAccountEntity driverAccountEntity = this.baseMapper.selectById(id);
        if (null == driverAccountEntity) {
            throw new EException(1001, errorMessageComponent.getCarrierDriverAuditHistory1001());
        }

        return this.driverAuditHistoryService.getBaseMapper().selectList(new QueryWrapper<DriverAuditHistoryEntity>().lambda()
                .eq(DriverAuditHistoryEntity::getDriverId, id).orderByAsc(DriverAuditHistoryEntity::getOperateTime));
    }

    @Override
    public PageResultVO getDriver4Page(DriverQueryForm queryObj) {
        IPage<DriverDetailDto> page = new Page<>(queryObj.getPageNum(), queryObj.getPageSize());
        this.baseMapper.selectDriver4Page(page, queryObj);

        // 数据转换
        List<DriverDetailDto> dtoList = page.getRecords();
        List<DriverListVO> targetRecordList = new ArrayList<>();
        for (DriverDetailDto dto : dtoList) {
            DriverListVO vo = new DriverListVO();
            BeanUtils.copyProperties(dto, vo);
            if (CarrierConstants.CARRIER_TYPE_COMPANY.equals(dto.getCarrierType())) {
                vo.setCarrierName(dto.getCarrierFullName());
            } else {
                vo.setCarrierName(dto.getCarrierContactor());
            }
            vo.setCreateTime(dto.getCreateTime().getTime());
            targetRecordList.add(vo);
        }

        return new PageResultVO(page, targetRecordList);
    }

    @Override
    public DriverDetailDto checkAccount(DriverInfoForm form) {
        return null;
    }
}