package com.esv.freight.customer.module.goodsowner.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.esv.freight.customer.common.component.ErrorMessageComponent;
import com.esv.freight.customer.common.exception.EException;
import com.esv.freight.customer.common.util.FeignUtils;
import com.esv.freight.customer.common.vo.PageResultVO;
import com.esv.freight.customer.feign.FeignBaseService;
import com.esv.freight.customer.module.goodsowner.GoodsOwnerConstants;
import com.esv.freight.customer.module.goodsowner.dao.ReceiveAddressDao;
import com.esv.freight.customer.module.goodsowner.dto.ReceiveAddressDto;
import com.esv.freight.customer.module.goodsowner.entity.ReceiveAddressEntity;
import com.esv.freight.customer.module.goodsowner.form.DeleteAddressForm;
import com.esv.freight.customer.module.goodsowner.form.ReceiveAddressForm;
import com.esv.freight.customer.module.goodsowner.service.ReceiveAddressService;
import com.esv.freight.customer.module.goodsowner.vo.AddressBriefVO;
import com.esv.freight.customer.module.goodsowner.vo.ReceiveAddressListVO;
import com.esv.freight.customer.module.goodsowner.vo.ReceiveAddressVO;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.*;


@Service("receiveAddressService")
public class ReceiveAddressServiceImpl extends ServiceImpl<ReceiveAddressDao, ReceiveAddressEntity> implements ReceiveAddressService {

    private FeignBaseService feignBaseService;

    private ErrorMessageComponent errorMessageComponent;

    @Autowired
    public ReceiveAddressServiceImpl(FeignBaseService feignBaseService, ErrorMessageComponent errorMessageComponent) {
        this.feignBaseService = feignBaseService;
        this.errorMessageComponent = errorMessageComponent;
    }

    @Override
    public Long addAddress(ReceiveAddressForm form) {
        // 1.判断地址是否重复
        List<ReceiveAddressEntity> receiveAddressEntityList = this.getAddressByName(form.getOwnerId(), form.getAddressName());
        if (0 < receiveAddressEntityList.size()) {
            throw new EException(1001, errorMessageComponent.getGoodsOwnerReceiveAddressAdd1001());
        }

        // 2.获取地址编码
        JSONObject batchIdReqJson = new JSONObject();
        batchIdReqJson.put("prefix", "SH");
        batchIdReqJson.put("formatter", "yyyyMMdd");
        batchIdReqJson.put("length", 13);
        JSONObject batchIdResJson;
        try {
            batchIdResJson = FeignUtils.getFeignResultData(feignBaseService.getBatchId(batchIdReqJson));
        } catch (Exception e) {
            log.error("调用[基础服务]生成收货地址编号失败");
            throw new EException("生成收货地址编号时发生错误");
        }
        String addressNumber = batchIdResJson.getString("batchId");

        // 3.新增地址
        ReceiveAddressEntity addressEntity = new ReceiveAddressEntity();
        BeanUtils.copyProperties(form, addressEntity);
        addressEntity.setAddressNumber(addressNumber);
        if (StringUtils.isNotBlank(form.getLon())) {
            addressEntity.setLon(new BigDecimal(form.getLon()));
        }
        if (StringUtils.isNotBlank(form.getLat())) {
            addressEntity.setLat(new BigDecimal(form.getLat()));
        }
        this.baseMapper.insert(addressEntity);

        return addressEntity.getId();
    }

    @Override
    public int editAddress(ReceiveAddressForm form) {
        // 1.判断ID是否有效
        ReceiveAddressEntity entity = this.baseMapper.selectById(form.getId());
        if (null == entity) {
            throw new EException(1001, errorMessageComponent.getGoodsOwnerReceiveAddressEdit1001());
        }

        // 2.判断地址是否重复
        QueryWrapper<ReceiveAddressEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.ne("id", form.getId());
        queryWrapper.eq("owner_id", form.getOwnerId());
        queryWrapper.eq("address_name", form.getAddressName());
        List<ReceiveAddressEntity> receiveAddressEntityList = this.baseMapper.selectList(queryWrapper);
        if (0 < receiveAddressEntityList.size()) {
            throw new EException(1002, errorMessageComponent.getGoodsOwnerReceiveAddressEdit1002());
        }

        // 3.更新地址
        ReceiveAddressEntity updateEntity = new ReceiveAddressEntity();
        BeanUtils.copyProperties(form, updateEntity);
        if (StringUtils.isNotBlank(form.getLon())) {
            updateEntity.setLon(new BigDecimal(form.getLon()));
        }
        if (StringUtils.isNotBlank(form.getLat())) {
            updateEntity.setLat(new BigDecimal(form.getLat()));
        }
        int flag = this.baseMapper.updateById(updateEntity);

        return flag;
    }

    @Override
    public int deleteAddress(DeleteAddressForm form) {
        String[] ids = form.getId().split(",");
        int count = this.baseMapper.deleteBatchIds(Arrays.asList(ids));
        return count;
    }

    @Override
    public ReceiveAddressVO getAddressDetail(ReceiveAddressForm form) {
        // 1.判断ID是否有效
        ReceiveAddressEntity entity = this.baseMapper.selectById(form.getId());
        if (null == entity) {
            throw new EException(1001, errorMessageComponent.getGoodsOwnerReceiveAddressDetail1001());
        }

        // 数据转换
        ReceiveAddressVO addressVO = new ReceiveAddressVO();
        BeanUtils.copyProperties(entity, addressVO);

        return addressVO;
    }

    @Override
    public List<AddressBriefVO> getAllAddress(ReceiveAddressForm form) {
        // 查询
        QueryWrapper<ReceiveAddressEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("owner_id", form.getOwnerId());
        List<ReceiveAddressEntity> receiveAddressEntityList = this.baseMapper.selectList(queryWrapper);

        // 数据转换
        List<AddressBriefVO> addressBriefVOList = new ArrayList<>();
        receiveAddressEntityList.forEach(entity -> {
            AddressBriefVO vo = new AddressBriefVO();
            BeanUtils.copyProperties(entity, vo);
            addressBriefVOList.add(vo);
        });

        return addressBriefVOList;
    }

    @Override
    public List<ReceiveAddressEntity> getAddressByName(Long ownerId, String addressName) {
        QueryWrapper<ReceiveAddressEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("owner_id", ownerId);
        queryWrapper.eq("address_name", addressName);
        return this.getBaseMapper().selectList(queryWrapper);
    }

    @Override
    public PageResultVO selectAddressList(ReceiveAddressForm form) {
        // 分页查询
        Map<String, Object> queryObj = new HashMap<>(3);
        if (StringUtils.isNotBlank(form.getKeywords())) {
            queryObj.put("keywords", form.getKeywords());
        }
        if (StringUtils.isNotBlank(form.getProvinceCode())) {
            queryObj.put("provinceCode", form.getProvinceCode());
        }
        if (StringUtils.isNotBlank(form.getCityCode())) {
            queryObj.put("cityCode", form.getCityCode());
        }
        IPage<ReceiveAddressDto> page = new Page<>(form.getPageNum(), form.getPageSize());
        this.baseMapper.selectAddressList(page, queryObj);

        // 数据转换
        List<ReceiveAddressDto> addressDtoList = page.getRecords();
        List<ReceiveAddressListVO> addressListVOList = new ArrayList<>();
        addressDtoList.forEach(entity -> {
            ReceiveAddressListVO vo = new ReceiveAddressListVO();
            BeanUtils.copyProperties(entity, vo);
            vo.setCreateTime(entity.getCreateTime().getTime());
            if (GoodsOwnerConstants.OWNER_TYPE_PERSONAL.equals(entity.getOwnerType())) {
                vo.setGoodsOwnerName(entity.getContactor());
            } else {
                vo.setGoodsOwnerName(entity.getOwnerFullName());
            }
            addressListVOList.add(vo);
        });

        return new PageResultVO(page, addressListVOList);
    }
}