package com.esv.freight.customer.module.goodsowner.controller;

import com.alibaba.fastjson.JSONObject;
import com.esv.freight.customer.common.exception.EException;
import com.esv.freight.customer.common.response.EResponse;
import com.esv.freight.customer.common.validator.groups.*;
import com.esv.freight.customer.module.goodsowner.GoodsOwnerConstants;
import com.esv.freight.customer.module.goodsowner.form.DeleteAddressForm;
import com.esv.freight.customer.module.goodsowner.form.DeliveryAddressForm;
import com.esv.freight.customer.module.goodsowner.form.ReceiveAddressForm;
import com.esv.freight.customer.module.goodsowner.service.DeliveryAddressService;
import com.esv.freight.customer.module.goodsowner.service.ReceiveAddressService;
import com.esv.freight.customer.module.goodsowner.validator.groups.AddressBrief;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

/**
 * @description: 承运商发货地址Controller
 * @project: freight-customer-service
 * @name: com.esv.freight.customer.module.goodsowner.controller.DeliveryAddressController
 * @author: 黄朝斌
 * @email: huangchaobin@esvtek.com
 * @createTime: 2020/04/21 15:40
 * @version:1.0
 */
@Slf4j
@RestController
@RequestMapping("/goodsowner/delivery/address")
@Validated
public class DeliveryAddressController {

    private DeliveryAddressService deliveryAddressService;

    private ReceiveAddressService receiveAddressService;

    @Autowired
    public DeliveryAddressController(DeliveryAddressService deliveryAddressService, ReceiveAddressService receiveAddressService) {
        this.deliveryAddressService = deliveryAddressService;
        this.receiveAddressService = receiveAddressService;
    }

    /**
     * description 新增发货地址
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/21 16:56
     **/
    @PostMapping("/add")
    public EResponse add(@RequestBody @Validated(ValidatorInsert.class) DeliveryAddressForm form) throws EException {
        Long id = deliveryAddressService.addAddress(form);
        JSONObject data = new JSONObject();
        data.put("id", id);

        // 判断是否复制地址
        if (GoodsOwnerConstants.ADDRESS_COPY_YES.equals(form.getAddressCopy())) {
            ReceiveAddressForm receiveAddressForm = new ReceiveAddressForm();
            BeanUtils.copyProperties(form, receiveAddressForm);
            receiveAddressForm.setReceiver(form.getDeliverer());
            receiveAddressForm.setReceiverPhone(form.getDelivererPhone());
            receiveAddressForm.setReceiverTelephone(form.getDelivererTelephone());
            try {
                receiveAddressService.addAddress(receiveAddressForm);
            } catch (Exception e) {
                log.error("复制为收货地址时发生错误");
                log.error(e.getMessage(), e);
            }
        }

        return EResponse.ok(data);
    }

    /**
     * description 编辑发货地址
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/21 18:56
     **/
    @PostMapping("/edit")
    public EResponse edit(@RequestBody @Validated(ValidatorUpdate.class) DeliveryAddressForm form) throws EException {
        deliveryAddressService.editAddress(form);
        return EResponse.ok();
    }

    /**
     * description 查看发货地址详情
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/21 19:55
     **/
    @PostMapping("/detail")
    public EResponse detail(@RequestBody @Validated(ValidatorDetail.class) DeliveryAddressForm form) throws EException {
        return EResponse.ok(deliveryAddressService.getAddressDetail(form));
    }

    /**
     * description 获取所有发货地址列表
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/21 20:08
     **/
    @PostMapping("/all")
    public EResponse all(@RequestBody @Validated(AddressBrief.class) DeliveryAddressForm form) throws EException {
        return EResponse.ok(deliveryAddressService.getAllAddress(form));
    }

    /**
     * description 删除发货地址
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/21 19:27
     **/
    @PostMapping("/delete")
    public EResponse delete(@RequestBody @Validated(ValidatorDelete.class) DeleteAddressForm form) throws EException {
        int count = deliveryAddressService.deleteAddress(form);
        JSONObject data = new JSONObject();
        data.put("count", count);
        return EResponse.ok(data);
    }

    /**
     * description 分页查询发货地址列表
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/22 9:28
     **/
    @PostMapping("/list")
    public EResponse list(@RequestBody @Validated(ValidatorList.class) DeliveryAddressForm form) throws EException {
        return EResponse.ok(deliveryAddressService.selectAddressList(form));
    }
}
