package com.esv.freight.customer.module.vehicle.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.esv.freight.customer.common.component.ErrorMessageComponent;
import com.esv.freight.customer.common.constants.CommonConstants;
import com.esv.freight.customer.common.exception.EException;
import com.esv.freight.customer.common.response.ECode;
import com.esv.freight.customer.common.util.ReqUtils;
import com.esv.freight.customer.common.vo.PageResultVO;
import com.esv.freight.customer.module.carrier.CarrierConstants;
import com.esv.freight.customer.module.carrier.entity.CarrierAccountEntity;
import com.esv.freight.customer.module.carrier.service.CarrierAccountService;
import com.esv.freight.customer.module.vehicle.VehicleConstants;
import com.esv.freight.customer.module.vehicle.dao.VehicleDao;
import com.esv.freight.customer.module.vehicle.dto.VehicleDetailDto;
import com.esv.freight.customer.module.vehicle.entity.AttachmentEntity;
import com.esv.freight.customer.module.vehicle.entity.VehicleAuditHistoryEntity;
import com.esv.freight.customer.module.vehicle.entity.VehicleDriverEntity;
import com.esv.freight.customer.module.vehicle.entity.VehicleEntity;
import com.esv.freight.customer.module.vehicle.form.VehicleAuditForm;
import com.esv.freight.customer.module.vehicle.form.VehicleInfoForm;
import com.esv.freight.customer.module.vehicle.form.VehicleQueryForm;
import com.esv.freight.customer.module.vehicle.service.AttachmentService;
import com.esv.freight.customer.module.vehicle.service.VehicleAuditHistoryService;
import com.esv.freight.customer.module.vehicle.service.VehicleDriverService;
import com.esv.freight.customer.module.vehicle.service.VehicleService;
import com.esv.freight.customer.module.vehicle.vo.VehicleAllVO;
import com.esv.freight.customer.module.vehicle.vo.VehicleListVO;
import com.esv.gateway.common.GatewayHeaders;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;


@Service("vehicleService")
public class VehicleServiceImpl extends ServiceImpl<VehicleDao, VehicleEntity> implements VehicleService {

    private ErrorMessageComponent errorMessageComponent;

    private CarrierAccountService carrierAccountService;

    private AttachmentService attachmentService;

    private VehicleAuditHistoryService vehicleAuditHistoryService;

    private VehicleDriverService vehicleDriverService;

    @Autowired
    public VehicleServiceImpl(ErrorMessageComponent errorMessageComponent, CarrierAccountService carrierAccountService,
                              AttachmentService attachmentService, VehicleAuditHistoryService vehicleAuditHistoryService,
                              VehicleDriverService vehicleDriverService) {
        this.errorMessageComponent = errorMessageComponent;
        this.carrierAccountService = carrierAccountService;
        this.attachmentService = attachmentService;
        this.vehicleAuditHistoryService = vehicleAuditHistoryService;
        this.vehicleDriverService = vehicleDriverService;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long insertVehicle(VehicleInfoForm form) {
        // 1:校验
        // 1.1校验承运商ID是否有效
        CarrierAccountEntity carrierAccountEntity = this.carrierAccountService.getBaseMapper().selectById(form.getCarrierId());
        if (null == carrierAccountEntity) {
            throw new EException(1001, errorMessageComponent.getCarrierVehicleAdd1001());
        }
        // 1.2校验车牌号是否存在
        int count = this.baseMapper.selectCount(new QueryWrapper<VehicleEntity>().lambda().eq(VehicleEntity::getLicenseNumber, form.getLicenseNumber()));
        if (0 < count) {
            throw new EException(1002, errorMessageComponent.getCarrierVehicleAdd1002());
        }
        // 1.3校验行驶证档案编号是否存在
        count = this.baseMapper.selectCount(new QueryWrapper<VehicleEntity>().lambda().eq(VehicleEntity::getVehicleLicenseNumber, form.getVehicleLicenseNumber()));
        if (0 < count) {
            throw new EException(1003, errorMessageComponent.getCarrierVehicleAdd1003());
        }
        // 1.4校验道路运输证号是否存在
        count = this.baseMapper.selectCount(new QueryWrapper<VehicleEntity>().lambda().eq(VehicleEntity::getRoadCertificateNumber, form.getRoadCertificateNumber()));
        if (0 < count) {
            throw new EException(1004, errorMessageComponent.getCarrierVehicleAdd1004());
        }

        // 2:新增车辆
        VehicleEntity vehicleEntity = new VehicleEntity();
        BeanUtils.copyProperties(form, vehicleEntity);
        vehicleEntity.setId(null);
        // 判断创建来源
        String sourceType = ReqUtils.getRequestHeader(CommonConstants.REQ_SOURCE_TYPE_KEY);
        if (CommonConstants.REQ_SOURCE_TYPE_ANDROID.equals(sourceType)
                || CommonConstants.REQ_SOURCE_TYPE_IOS.equals(sourceType) ) {
            // 用户通过终端APP添加车辆
            vehicleEntity.setAuditStatus(VehicleConstants.VEHICLE_AUDIT_STATUS_UNAUDITED);
            vehicleEntity.setSourceType(VehicleConstants.VEHICLE_SOURCE_TYPE_REGISTER);
            vehicleEntity.setVehicleStatus(VehicleConstants.VEHICLE_ACCOUNT_STATUS_BLOCK);
        } else {
            vehicleEntity.setAuditStatus(VehicleConstants.VEHICLE_AUDIT_STATUS_SUCCESS);
            vehicleEntity.setSourceType(VehicleConstants.VEHICLE_SOURCE_TYPE_PLATFORM);
            vehicleEntity.setVehicleStatus(VehicleConstants.VEHICLE_ACCOUNT_STATUS_UNBLOCK);
        }
        this.baseMapper.insert(vehicleEntity);
        Long vehicleId = vehicleEntity.getId();

        // 3:新增车辆附件
        AttachmentEntity attachmentEntity = new AttachmentEntity();
        BeanUtils.copyProperties(form, attachmentEntity);
        attachmentEntity.setCarrierId(form.getCarrierId());
        attachmentEntity.setVehicleId(vehicleId);
        this.attachmentService.getBaseMapper().insert(attachmentEntity);

        // 4:新增审核记录
        VehicleAuditHistoryEntity vehicleAuditHistoryEntity = new VehicleAuditHistoryEntity();
        vehicleAuditHistoryEntity.setCarrierId(form.getCarrierId());
        vehicleAuditHistoryEntity.setVehicleId(vehicleId);
        if (CommonConstants.REQ_SOURCE_TYPE_ANDROID.equals(sourceType)
                || CommonConstants.REQ_SOURCE_TYPE_IOS.equals(sourceType) ) {
            vehicleAuditHistoryEntity.setAuditStatus(VehicleConstants.VEHICLE_AUDIT_STATUS_UNAUDITED);
            vehicleAuditHistoryEntity.setRemark("用户通过终端APP添加车辆");
        } else {
            vehicleAuditHistoryEntity.setAuditStatus(VehicleConstants.VEHICLE_AUDIT_STATUS_SUCCESS);
            vehicleAuditHistoryEntity.setRemark("平台创建自动审核");
        }
        vehicleAuditHistoryEntity.setOperateUser(ReqUtils.getRequestHeader(GatewayHeaders.USER_ACCOUNT));
        this.vehicleAuditHistoryService.getBaseMapper().insert(vehicleAuditHistoryEntity);

        // 5:新增车辆-司机绑定关系
        if (CommonConstants.REQ_SOURCE_TYPE_ANDROID.equals(sourceType)
                || CommonConstants.REQ_SOURCE_TYPE_IOS.equals(sourceType) ) {
            VehicleDriverEntity vehicleDriverEntity = new VehicleDriverEntity();
            vehicleDriverEntity.setDriverId(ReqUtils.getRequestUserId());
            vehicleDriverEntity.setVehicleId(vehicleId);
            vehicleDriverEntity.setCarrierId(form.getCarrierId());
            vehicleDriverEntity.setSelected(VehicleConstants.VEHICLE_SELECTED_NO);
            vehicleDriverEntity.setOperateUser(ReqUtils.getRequestUserAccount());
            this.vehicleDriverService.getBaseMapper().insert(vehicleDriverEntity);
        }

        return vehicleId;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Integer updateVehicle(VehicleInfoForm form) {
        // 1:校验
        // 1.1校验车辆ID是否有效
        VehicleEntity vehicleEntity = this.baseMapper.selectById(form.getId());
        if (null == vehicleEntity) {
            throw new EException(1001, errorMessageComponent.getCarrierVehicleEdit1001());
        }
        // 1.2校验车牌号是否存在
        int count = 0;
        if (StringUtils.isNotBlank(form.getLicenseNumber())) {
            this.baseMapper.selectCount(new QueryWrapper<VehicleEntity>().lambda()
                    .ne(VehicleEntity::getId, form.getId())
                    .eq(VehicleEntity::getLicenseNumber, form.getLicenseNumber()));
            if (0 < count) {
                throw new EException(1002, errorMessageComponent.getCarrierVehicleEdit1002());
            }
        }
        // 1.3校验行驶证档案编号是否存在
        if (StringUtils.isNotBlank(form.getVehicleLicenseNumber())) {
            count = this.baseMapper.selectCount(new QueryWrapper<VehicleEntity>().lambda()
                    .ne(VehicleEntity::getId, form.getId())
                    .eq(VehicleEntity::getVehicleLicenseNumber, form.getVehicleLicenseNumber()));
            if (0 < count) {
                throw new EException(1003, errorMessageComponent.getCarrierVehicleEdit1003());
            }
        }
        // 1.4校验道路运输证号是否存在
        if (StringUtils.isNotBlank(form.getRoadCertificateNumber())) {
            count = this.baseMapper.selectCount(new QueryWrapper<VehicleEntity>().lambda()
                    .ne(VehicleEntity::getId, form.getId())
                    .eq(VehicleEntity::getRoadCertificateNumber, form.getRoadCertificateNumber()));
            if (0 < count) {
                throw new EException(1004, errorMessageComponent.getCarrierVehicleEdit1004());
            }
        }
        // 道路运输证号，4.5吨及以下普货车辆可以为空
        BigDecimal bigDecimal4_5 = new BigDecimal("4.5");
        if (-1 == bigDecimal4_5.compareTo(form.getLoadCapacity())
                && StringUtils.isBlank(form.getRoadCertificateNumber())
                && StringUtils.isBlank(vehicleEntity.getRoadCertificateNumber())) {
            throw new EException(ECode.PARAM_ERROR.code(), "4.5顿以上载重的车辆，道路运输证号不能为空");
        }

        // 2:更新车辆信息
        int flag = 0;
        vehicleEntity = new VehicleEntity();
        BeanUtils.copyProperties(form, vehicleEntity);
        flag += this.baseMapper.updateById(vehicleEntity);

        // 3:更新车辆附件
        AttachmentEntity attachmentEntity = new AttachmentEntity();
        BeanUtils.copyProperties(form, attachmentEntity);
        flag += this.attachmentService.getBaseMapper().update(attachmentEntity,
                new UpdateWrapper<AttachmentEntity>().lambda().eq(AttachmentEntity::getVehicleId, form.getId()));

        return flag;
    }

    @Override
    public VehicleDetailDto getVehicleDetail(Long id) {
        return this.baseMapper.selectVehicleDetail(id);
    }

    @Override
    public void auditVehicle(VehicleAuditForm form) {
        // 1:校验
        // 1.1校验车辆ID是否有效
        VehicleEntity vehicleEntity = this.baseMapper.selectById(form.getId());
        if (null == vehicleEntity) {
            throw new EException(1001, errorMessageComponent.getCarrierVehicleAudit1001());
        }
        // 1.2校验车辆是否审核通过
        if (VehicleConstants.VEHICLE_AUDIT_STATUS_SUCCESS.equals(vehicleEntity.getAuditStatus())) {
            throw new EException(1002, errorMessageComponent.getCarrierVehicleAudit1002());
        }

        // 2:更新车辆审核状态
        VehicleEntity updateEntity = new VehicleEntity();
        updateEntity.setId(form.getId());
        updateEntity.setAuditStatus(form.getAuditState());
        this.baseMapper.updateById(updateEntity);

        // 3:更新车辆审核记录
        VehicleAuditHistoryEntity vehicleAuditHistoryEntity = new VehicleAuditHistoryEntity();
        vehicleAuditHistoryEntity.setCarrierId(vehicleEntity.getCarrierId());
        vehicleAuditHistoryEntity.setVehicleId(form.getId());
        vehicleAuditHistoryEntity.setAuditStatus(form.getAuditState());
        vehicleAuditHistoryEntity.setRemark(form.getRemark());
        this.vehicleAuditHistoryService.getBaseMapper().insert(vehicleAuditHistoryEntity);
    }

    @Override
    public Boolean vehicleExits(Long id) {
        int count = this.baseMapper.selectCount(new QueryWrapper<VehicleEntity>().lambda().eq(VehicleEntity::getId, id));
        if (0 == count) {
            return false;
        } else {
            return true;
        }
    }

    @Override
    public Integer blockVehicle(Long id) {
        VehicleEntity vehicleEntity = this.baseMapper.selectById(id);
        if (null == vehicleEntity) {
            throw new EException(1001, errorMessageComponent.getCarrierVehicleBlock1001());
        }
        if (VehicleConstants.VEHICLE_ACCOUNT_STATUS_BLOCK.equals(vehicleEntity.getVehicleStatus())) {
            throw new EException(1002, errorMessageComponent.getCarrierVehicleBlock1002());
        }

        VehicleEntity updateEntity = new VehicleEntity();
        updateEntity.setId(id);
        updateEntity.setVehicleStatus(VehicleConstants.VEHICLE_ACCOUNT_STATUS_BLOCK);
        return this.baseMapper.updateById(updateEntity);
    }

    @Override
    public Integer unBlockVehicle(Long id) {
        VehicleEntity vehicleEntity = this.baseMapper.selectById(id);
        if (null == vehicleEntity) {
            throw new EException(1001, errorMessageComponent.getCarrierVehicleUnBlock1001());
        }
        if (VehicleConstants.VEHICLE_ACCOUNT_STATUS_UNBLOCK.equals(vehicleEntity.getVehicleStatus())) {
            throw new EException(1002, errorMessageComponent.getCarrierVehicleUnBlock1002());
        }

        VehicleEntity updateEntity = new VehicleEntity();
        updateEntity.setId(id);
        updateEntity.setVehicleStatus(VehicleConstants.VEHICLE_ACCOUNT_STATUS_UNBLOCK);
        return this.baseMapper.updateById(updateEntity);
    }

    @Override
    public PageResultVO getVehicle4Page(VehicleQueryForm queryObj) {
        IPage<VehicleDetailDto> page = new Page<>(queryObj.getPageNum(), queryObj.getPageSize());
        this.baseMapper.selectVehicle4Page(page, queryObj);

        // 数据转换
        List<VehicleDetailDto> dtoList = page.getRecords();
        List<VehicleListVO> targetRecordList = new ArrayList<>();
        for (VehicleDetailDto dto : dtoList) {
            VehicleListVO vo = new VehicleListVO();
            BeanUtils.copyProperties(dto, vo);
            if (CarrierConstants.CARRIER_TYPE_COMPANY.equals(dto.getCarrierType())) {
                vo.setCarrierName(dto.getCarrierFullName());
            } else {
                vo.setCarrierName(dto.getCarrierContactor());
            }
            vo.setCreateTime(dto.getCreateTime().getTime());
            targetRecordList.add(vo);
        }

        return new PageResultVO(page, targetRecordList);
    }

    @Override
    public List<VehicleAllVO> getVehicle4List(VehicleQueryForm queryObj) {
        // 1:查询
        List<VehicleDetailDto> dtoList = this.baseMapper.selectVehicle4List(queryObj);

        // 2:数据转换
        List<VehicleAllVO> voList = new ArrayList<>();
        for (VehicleDetailDto dto : dtoList) {
            VehicleAllVO vo = new VehicleAllVO();
            BeanUtils.copyProperties(dto, vo);
            if (CarrierConstants.CARRIER_TYPE_COMPANY.equals(dto.getCarrierType())) {
                vo.setCarrierName(dto.getCarrierFullName());
            } else {
                vo.setCarrierName(dto.getCarrierContactor());
            }
            vo.setVehicleId(dto.getId());
            voList.add(vo);
        }

        return voList;
    }
}