package com.esv.freight.customer.module.vehicle.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.esv.freight.customer.common.component.ErrorMessageComponent;
import com.esv.freight.customer.common.exception.EException;
import com.esv.freight.customer.common.response.ECode;
import com.esv.freight.customer.common.util.ReqUtils;
import com.esv.freight.customer.module.carrier.entity.CarrierAccountEntity;
import com.esv.freight.customer.module.carrier.service.CarrierAccountService;
import com.esv.freight.customer.module.vehicle.VehicleConstants;
import com.esv.freight.customer.module.vehicle.dao.VehicleDao;
import com.esv.freight.customer.module.vehicle.dto.VehicleDetailDto;
import com.esv.freight.customer.module.vehicle.entity.AttachmentEntity;
import com.esv.freight.customer.module.vehicle.entity.VehicleAuditHistoryEntity;
import com.esv.freight.customer.module.vehicle.entity.VehicleEntity;
import com.esv.freight.customer.module.vehicle.form.VehicleInfoForm;
import com.esv.freight.customer.module.vehicle.service.AttachmentService;
import com.esv.freight.customer.module.vehicle.service.VehicleAuditHistoryService;
import com.esv.freight.customer.module.vehicle.service.VehicleService;
import com.esv.gateway.common.GatewayHeaders;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;


@Service("vehicleService")
public class VehicleServiceImpl extends ServiceImpl<VehicleDao, VehicleEntity> implements VehicleService {

    private ErrorMessageComponent errorMessageComponent;

    private CarrierAccountService carrierAccountService;

    private AttachmentService attachmentService;

    private VehicleAuditHistoryService vehicleAuditHistoryService;

    @Autowired
    public VehicleServiceImpl(ErrorMessageComponent errorMessageComponent, CarrierAccountService carrierAccountService,
                              AttachmentService attachmentService, VehicleAuditHistoryService vehicleAuditHistoryService) {
        this.errorMessageComponent = errorMessageComponent;
        this.carrierAccountService = carrierAccountService;
        this.attachmentService = attachmentService;
        this.vehicleAuditHistoryService = vehicleAuditHistoryService;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long insertVehicle(VehicleInfoForm form) {
        // 1:校验
        // 1.1校验承运商ID是否有效
        CarrierAccountEntity carrierAccountEntity = this.carrierAccountService.getBaseMapper().selectById(form.getCarrierId());
        if (null == carrierAccountEntity) {
            throw new EException(1001, errorMessageComponent.getCarrierVehicleAdd1001());
        }
        // 1.2校验车牌号是否存在
        int count = this.baseMapper.selectCount(new QueryWrapper<VehicleEntity>().lambda().eq(VehicleEntity::getLicenseNumber, form.getLicenseNumber()));
        if (0 < count) {
            throw new EException(1002, errorMessageComponent.getCarrierVehicleAdd1002());
        }
        // 1.3校验行驶证档案编号是否存在
        count = this.baseMapper.selectCount(new QueryWrapper<VehicleEntity>().lambda().eq(VehicleEntity::getVehicleLicenseNumber, form.getVehicleLicenseNumber()));
        if (0 < count) {
            throw new EException(1003, errorMessageComponent.getCarrierVehicleAdd1003());
        }
        // 1.4校验道路运输证号是否存在
        count = this.baseMapper.selectCount(new QueryWrapper<VehicleEntity>().lambda().eq(VehicleEntity::getRoadCertificateNumber, form.getRoadCertificateNumber()));
        if (0 < count) {
            throw new EException(1004, errorMessageComponent.getCarrierVehicleAdd1004());
        }

        // 2:新增车辆
        VehicleEntity vehicleEntity = new VehicleEntity();
        BeanUtils.copyProperties(form, vehicleEntity);
        vehicleEntity.setId(null);
        vehicleEntity.setVehicleStatus(VehicleConstants.VEHICLE_ACCOUNT_STATUS_UNBLOCK);
        vehicleEntity.setAuditStatus(VehicleConstants.VEHICLE_AUDIT_STATUS_SUCCESS);
        vehicleEntity.setSourceType(VehicleConstants.VEHICLE_SOURCE_TYPE_PLATFORM);
        this.baseMapper.insert(vehicleEntity);
        Long vehicleId = vehicleEntity.getId();

        // 3:新增车辆附件
        AttachmentEntity attachmentEntity = new AttachmentEntity();
        BeanUtils.copyProperties(form, attachmentEntity);
        attachmentEntity.setCarrierId(form.getCarrierId());
        attachmentEntity.setVehicleId(vehicleId);
        this.attachmentService.getBaseMapper().insert(attachmentEntity);

        // 4:新增审核记录
        VehicleAuditHistoryEntity vehicleAuditHistoryEntity = new VehicleAuditHistoryEntity();
        vehicleAuditHistoryEntity.setCarrierId(form.getCarrierId());
        vehicleAuditHistoryEntity.setVehicleId(vehicleId);
        vehicleAuditHistoryEntity.setAuditStatus(VehicleConstants.VEHICLE_AUDIT_STATUS_SUCCESS);
        vehicleAuditHistoryEntity.setRemark("平台创建自动审核");
        vehicleAuditHistoryEntity.setOperateUser(ReqUtils.getRequestHeader(GatewayHeaders.USER_ACCOUNT));
        this.vehicleAuditHistoryService.getBaseMapper().insert(vehicleAuditHistoryEntity);

        return vehicleId;
    }

    @Override
    public Integer updateVehicle(VehicleInfoForm form) {
        // 1:校验
        // 1.1校验车辆ID是否有效
        VehicleEntity vehicleEntity = this.baseMapper.selectById(form.getId());
        if (null == vehicleEntity) {
            throw new EException(1001, errorMessageComponent.getCarrierVehicleEdit1001());
        }
        // 1.2校验车牌号是否存在
        int count = 0;
        if (StringUtils.isNotBlank(form.getLicenseNumber())) {
            this.baseMapper.selectCount(new QueryWrapper<VehicleEntity>().lambda()
                    .ne(VehicleEntity::getId, form.getId())
                    .eq(VehicleEntity::getLicenseNumber, form.getLicenseNumber()));
            if (0 < count) {
                throw new EException(1002, errorMessageComponent.getCarrierVehicleEdit1002());
            }
        }
        // 1.3校验行驶证档案编号是否存在
        if (StringUtils.isNotBlank(form.getVehicleLicenseNumber())) {
            count = this.baseMapper.selectCount(new QueryWrapper<VehicleEntity>().lambda()
                    .ne(VehicleEntity::getId, form.getId())
                    .eq(VehicleEntity::getVehicleLicenseNumber, form.getVehicleLicenseNumber()));
            if (0 < count) {
                throw new EException(1003, errorMessageComponent.getCarrierVehicleEdit1003());
            }
        }
        // 1.4校验道路运输证号是否存在
        if (StringUtils.isNotBlank(form.getRoadCertificateNumber())) {
            count = this.baseMapper.selectCount(new QueryWrapper<VehicleEntity>().lambda()
                    .ne(VehicleEntity::getId, form.getId())
                    .eq(VehicleEntity::getRoadCertificateNumber, form.getRoadCertificateNumber()));
            if (0 < count) {
                throw new EException(1004, errorMessageComponent.getCarrierVehicleEdit1004());
            }
        }
        // 道路运输证号，4.5吨及以下普货车辆可以为空
        BigDecimal bigDecimal4_5 = new BigDecimal("4.5");
        if (-1 == bigDecimal4_5.compareTo(form.getLoadCapacity())
                && StringUtils.isBlank(form.getRoadCertificateNumber())
                && StringUtils.isBlank(vehicleEntity.getRoadCertificateNumber())) {
            throw new EException(ECode.PARAM_ERROR.code(), "4.5顿以上载重的车辆，道路运输证号不能为空");
        }

        // 2:更新车辆信息
        int flag = 0;
        vehicleEntity = new VehicleEntity();
        BeanUtils.copyProperties(form, vehicleEntity);
        flag += this.baseMapper.updateById(vehicleEntity);

        // 3:更新车辆附件
        AttachmentEntity attachmentEntity = new AttachmentEntity();
        BeanUtils.copyProperties(form, attachmentEntity);
        flag += this.attachmentService.getBaseMapper().update(attachmentEntity,
                new UpdateWrapper<AttachmentEntity>().lambda().eq(AttachmentEntity::getVehicleId, form.getId()));

        return flag;
    }

    @Override
    public VehicleDetailDto getVehicleDetail(Long id) {
        return this.baseMapper.selectVehicleDetail(id);
    }
}