package com.esv.freight.customer.module.goodsowner.controller;

import com.alibaba.fastjson.JSONObject;
import com.esv.freight.customer.common.component.ErrorMessageComponent;
import com.esv.freight.customer.common.component.PasswordComponent;
import com.esv.freight.customer.common.exception.EException;
import com.esv.freight.customer.common.response.ECode;
import com.esv.freight.customer.common.response.EResponse;
import com.esv.freight.customer.common.util.ReqUtils;
import com.esv.freight.customer.common.util.VerifyUtils;
import com.esv.freight.customer.common.validator.groups.ValidatorDetail;
import com.esv.freight.customer.common.validator.groups.ValidatorInsert;
import com.esv.freight.customer.common.validator.groups.ValidatorUpdate;
import com.esv.freight.customer.module.goodsowner.GoodsOwnerConstants;
import com.esv.freight.customer.module.goodsowner.dto.AccountInfoDto;
import com.esv.freight.customer.module.goodsowner.entity.AuditHistoryEntity;
import com.esv.freight.customer.module.goodsowner.form.AccountForm;
import com.esv.freight.customer.module.goodsowner.form.AccountInfoForm;
import com.esv.freight.customer.module.goodsowner.service.AccountService;
import com.esv.freight.customer.module.goodsowner.validator.groups.*;
import com.esv.freight.customer.module.goodsowner.vo.AccountInfoVO;
import com.esv.freight.customer.module.goodsowner.vo.AuditHistoryVO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.ArrayList;
import java.util.List;

/**
 * @description:
 * @project: freight-customer-service
 * @name: com.esv.freight.customer.module.goodsowner.controller.GoodsOwnerController
 * @author: 黄朝斌
 * @email: huangchaobin@esvtek.com
 * @createTime: 2020/04/17 14:33
 * @version:1.0
 */
@Slf4j
@RestController
@RequestMapping("/goodsowner/account")
@Validated
public class GoodsOwnerController {

    private AccountService accountService;

    private ErrorMessageComponent errorMessageComponent;

    private PasswordComponent passwordComponent;

    @Autowired
    public GoodsOwnerController(AccountService accountService, ErrorMessageComponent errorMessageComponent, PasswordComponent passwordComponent) {
        this.accountService = accountService;
        this.errorMessageComponent = errorMessageComponent;
        this.passwordComponent = passwordComponent;
    }

    @PostMapping("/list")
    public EResponse getAccountList() {
        String account = "test";
        return EResponse.ok(accountService.getAccountRecordByAccount(account));
    }

    /**
     * description 平台用户创建帐号
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/17 17:39
     **/
    @PostMapping("/add")
    public EResponse createAccountByManage(@RequestBody @Validated(ValidatorInsert.class) AccountInfoForm form) throws EException {
        /****************************** 参数校验 ******************************/
        // 货主为企业
        if (GoodsOwnerConstants.OWNER_TYPE_COMPANY.equals(form.getOwnerType())) {
            String[] notBlankParams = new String[] {"ownerFullName", "uniCreditCode", "creditCopyUrl", "legalPerson"};
            ReqUtils.checkParamsNotBlank(JSONObject.parseObject(form.toString()), notBlankParams);
        }
        // 校验身份证号码
        if (!VerifyUtils.isValidIdCard18(form.getIdCard())) {
            throw new EException(ECode.PARAM_ERROR.code(), "无效的身份证号码");
        }
        /****************************** 参数校验 ******************************/

        return EResponse.ok(accountService.createAccountByManager(form));
    }

    /**
     * description 更新帐号信息
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/18 14:06
     **/
    @PostMapping("/edit")
    public EResponse updateAccountInfo(@RequestBody @Validated(ValidatorUpdate.class) AccountInfoForm form) throws EException {
        /****************************** 参数校验 ******************************/
        // 校验身份证号码
        if (StringUtils.isNotBlank(form.getIdCard()) && !VerifyUtils.isValidIdCard18(form.getIdCard())) {
            throw new EException(ECode.PARAM_ERROR.code(), "无效的身份证号码");
        }
        /****************************** 参数校验 ******************************/

        accountService.updateAccountInfo(form);
        return EResponse.ok();
    }

    /**
     * description 停用帐号
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/18 15:11
     **/
    @PostMapping("/block")
    public EResponse blockAccount(@RequestBody @Validated(ValidatorBlock.class) AccountForm form) throws EException {
        accountService.blockAccount(form.getId());
        return EResponse.ok();
    }

    /**
     * description 启用帐号
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/18 15:11
     **/
    @PostMapping("/unblock")
    public EResponse unblockAccount(@RequestBody @Validated(ValidatorUnblock.class) AccountForm form) throws EException {
        accountService.unblockAccount(form.getId());
        return EResponse.ok();
    }

    /**
     * description 审核帐号
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/18 16:08
     **/
    @PostMapping("/audit")
    public EResponse auditAccount(@RequestBody @Validated(ValidatorAudit.class) AccountForm form) throws EException {
        /****************************** 参数校验 ******************************/
        // 校验备注
        if (GoodsOwnerConstants.OWNER_AUDIT_STATUS_FAILURE.equals(form.getAuditStatus()) && StringUtils.isBlank(form.getRemark())) {
            throw new EException(ECode.PARAM_ERROR.code(), "参数remark不能为空");
        }
        /****************************** 参数校验 ******************************/

        accountService.auditAccount(form);
        return EResponse.ok();
    }

    /**
     * description 查看帐号的审核历史
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/18 16:50
     **/
    @PostMapping("/audit/history")
    public EResponse getAccountAuditHistory(@RequestBody @Validated(ValidatorAuditHistory.class) AccountForm form) throws EException {
        // 查询审核历史
        List<AuditHistoryEntity> auditHistoryEntityList = accountService.getAuditHistory(form.getId());

        // 数据转换
        List<AuditHistoryVO> auditHistoryVOList = new ArrayList<>();
        auditHistoryEntityList.forEach(auditHistoryEntity -> {
            AuditHistoryVO auditHistoryVO = new AuditHistoryVO();
            BeanUtils.copyProperties(auditHistoryEntity, auditHistoryVO);
            auditHistoryVO.setOperateTime(auditHistoryEntity.getOperateTime().getTime());
            auditHistoryVOList.add(auditHistoryVO);
        });

        return EResponse.ok(auditHistoryVOList);
    }

    /**
     * description 获取帐号详细信息
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/20 10:12
     **/
    @PostMapping("/detail")
    public EResponse getAccountDetail(@RequestBody @Validated(ValidatorDetail.class) AccountForm form) throws EException {
        /****************************** 参数校验 ******************************/
        if (null == form.getId() && null == form.getAccount()) {
            throw new EException(ECode.PARAM_ERROR.code(), "参数id、account不能同时为空");
        }
        /****************************** 参数校验 ******************************/

        AccountInfoDto accountInfoDto = accountService.getAccountInfo(form);
        // 查询结果处理
        if (null != form.getId()) {
            if (null == accountInfoDto) {
                throw new EException(1001, errorMessageComponent.getGoodsOwnerAccountDetail1001());
            }
        } else if (null != form.getAccount()) {
            if (null == accountInfoDto) {
                throw new EException(1002, errorMessageComponent.getGoodsOwnerAccountDetail1002());
            }
        } else {

        }

        AccountInfoVO accountInfoVO = new AccountInfoVO();
        BeanUtils.copyProperties(accountInfoDto, accountInfoVO);
        return EResponse.ok(accountInfoVO);
    }

    /**
     * description 帐号密码校验
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/20 11:22
     **/
    @PostMapping("/password/check")
    public EResponse checkAccountPwd(@RequestBody @Validated(ValidatorPassword.class) AccountForm form) throws EException {
        AccountInfoDto accountInfoDto = accountService.getAccountInfo(form);
        // 校验帐号是否存在
        if (null == accountInfoDto) {
            throw new EException(1001, errorMessageComponent.getGoodsOwnerAccountPasswordCheck1001());
        }
        // 校验密码是否正确
        if (!passwordComponent.checkPwd4Salt(form.getPassword(), accountInfoDto.getSalt(), accountInfoDto.getPassword())) {
            throw new EException(1002, errorMessageComponent.getGoodsOwnerAccountPasswordCheck1001());
        }

        AccountInfoVO accountInfoVO = new AccountInfoVO();
        BeanUtils.copyProperties(accountInfoDto, accountInfoVO);
        return EResponse.ok(accountInfoVO);
    }

    /**
     * description 帐号注册
     * param [form]
     * return com.esv.freight.customer.common.response.EResponse
     * author Administrator
     * createTime 2020/04/20 14:56
     **/
    @PostMapping("/register")
    public EResponse accountRegister(@RequestBody @Validated(ValidatorRegister.class) AccountForm form) throws EException {
        Long id = accountService.registerAccount(form.getAccount());
        JSONObject data = new JSONObject();
        data.put("id", id);
        return EResponse.ok(data);
    }
}
