package com.esv.freight.customer.module.driver.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.esv.freight.customer.common.component.ErrorMessageComponent;
import com.esv.freight.customer.common.component.PasswordComponent;
import com.esv.freight.customer.common.exception.EException;
import com.esv.freight.customer.common.util.ReqUtils;
import com.esv.freight.customer.module.driver.DriverConstants;
import com.esv.freight.customer.module.driver.dao.DriverAccountDao;
import com.esv.freight.customer.module.driver.entity.DriverAccountEntity;
import com.esv.freight.customer.module.driver.entity.DriverAuditHistoryEntity;
import com.esv.freight.customer.module.driver.entity.DriverInfoEntity;
import com.esv.freight.customer.module.driver.service.DriverAccountService;
import com.esv.freight.customer.module.driver.service.DriverAuditHistoryService;
import com.esv.freight.customer.module.driver.service.DriverInfoService;
import com.esv.freight.customer.module.driver.form.DriverInfoForm;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;


@Service("driverAccountService")
public class DriverAccountServiceImpl extends ServiceImpl<DriverAccountDao, DriverAccountEntity> implements DriverAccountService {

    private ErrorMessageComponent errorMessageComponent;

    private PasswordComponent passwordComponent;

    private DriverInfoService driverInfoService;

    private DriverAuditHistoryService driverAuditHistoryService;

    @Autowired
    public DriverAccountServiceImpl(ErrorMessageComponent errorMessageComponent, PasswordComponent passwordComponent,
                                    DriverInfoService driverInfoService, DriverAuditHistoryService driverAuditHistoryService) {
        this.errorMessageComponent = errorMessageComponent;
        this.passwordComponent = passwordComponent;
        this.driverInfoService = driverInfoService;
        this.driverAuditHistoryService = driverAuditHistoryService;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long insertDriver(DriverInfoForm form) {
        // 1:校验
        // 1.1:校验账号是否已存在
        Integer count = this.baseMapper.selectCount(new QueryWrapper<DriverAccountEntity>().lambda()
                .eq(DriverAccountEntity::getAccount, form.getAccount()));
        if (0 < count) {
            throw new EException(1001, errorMessageComponent.getCarrierDriverAdd1001());
        }
        // 1.2:校验身份证号码是否已存在
        count = this.driverInfoService.getBaseMapper().selectCount(new QueryWrapper<DriverInfoEntity>().lambda()
                .eq(DriverInfoEntity::getIdCard, form.getIdCard()));
        if (0 < count) {
            throw new EException(1002, errorMessageComponent.getCarrierDriverAdd1002());
        }
        // 1.3:校验驾驶证号码是否已存在
        count = this.driverInfoService.getBaseMapper().selectCount(new QueryWrapper<DriverInfoEntity>().lambda()
                .eq(DriverInfoEntity::getDrivingLicense, form.getDrivingLicense()));
        if (0 < count) {
            throw new EException(1003, errorMessageComponent.getCarrierDriverAdd1003());
        }
        // 1.4:校验从业资格证号是否已存在
        if (StringUtils.isNotBlank(form.getCertificateNumber())) {
            count = this.driverInfoService.getBaseMapper().selectCount(new QueryWrapper<DriverInfoEntity>().lambda()
                    .eq(DriverInfoEntity::getCertificateNumber, form.getCertificateNumber()));
            if (0 < count) {
                throw new EException(1004, errorMessageComponent.getCarrierDriverAdd1004());
            }
        }

        // 2:新增司机帐号
        DriverAccountEntity driverAccountEntity = new DriverAccountEntity();
        BeanUtils.copyProperties(form, driverAccountEntity);
        String salt = passwordComponent.generateAccountPwdSalt();
        driverAccountEntity.setSalt(salt);
        driverAccountEntity.setPassword(passwordComponent.generatePwd4Salt(form.getPassword(), salt));
        driverAccountEntity.setAccountStatus(DriverConstants.ACCOUNT_STATUS_UNBLOCK);
        driverAccountEntity.setSourceType(DriverConstants.ACCOUNT_SOURCE_TYPE_PLATFORM);
        driverAccountEntity.setAuthenticateStatus(DriverConstants.ACCOUNT_AUTHENTICATE_STATUS_YES);
        driverAccountEntity.setAuditStatus(DriverConstants.ACCOUNT_AUDIT_STATUS_SUCCESS);
        this.baseMapper.insert(driverAccountEntity);
        Long driverId = driverAccountEntity.getId();

        // 3:新增司机信息
        DriverInfoEntity driverInfoEntity = new DriverInfoEntity();
        BeanUtils.copyProperties(form, driverInfoEntity);
        driverInfoEntity.setDriverId(driverId);
        this.driverInfoService.getBaseMapper().insert(driverInfoEntity);

        // 4:新增审核记录
        DriverAuditHistoryEntity driverAuditHistoryEntity = new DriverAuditHistoryEntity();
        driverAuditHistoryEntity.setCarrierId(form.getCarrierId());
        driverAuditHistoryEntity.setDriverId(driverId);
        driverAuditHistoryEntity.setAuditStatus(DriverConstants.ACCOUNT_AUDIT_STATUS_SUCCESS);
        driverAuditHistoryEntity.setRemark("平台创建自动审核");
        driverAuditHistoryEntity.setOperateUser(ReqUtils.getRequestUserAccount());
        this.driverAuditHistoryService.getBaseMapper().insert(driverAuditHistoryEntity);

        return driverId;
    }

    @Override
    public Integer updateDriver(DriverInfoForm form) {
        // 1:校验
        // 1.1:校验账号ID是否有效
        DriverAccountEntity driverAccountEntity = this.baseMapper.selectById(form.getId());
        if (null == driverAccountEntity) {
            throw new EException(1001, errorMessageComponent.getCarrierDriverEdit1001());
        }
        // 1.2:校验身份证号码是否已存在
        Integer count = this.driverInfoService.getBaseMapper().selectCount(new QueryWrapper<DriverInfoEntity>().lambda()
                .ne(DriverInfoEntity::getDriverId, form.getId())
                .eq(DriverInfoEntity::getIdCard, form.getIdCard()));
        if (0 < count) {
            throw new EException(1002, errorMessageComponent.getCarrierDriverEdit1002());
        }
        // 1.3:校验驾驶证号码是否已存在
        count = this.driverInfoService.getBaseMapper().selectCount(new QueryWrapper<DriverInfoEntity>().lambda()
                .ne(DriverInfoEntity::getDriverId, form.getId())
                .eq(DriverInfoEntity::getDrivingLicense, form.getDrivingLicense()));
        if (0 < count) {
            throw new EException(1003, errorMessageComponent.getCarrierDriverEdit1003());
        }
        // 1.4:校验从业资格证号是否已存在
        if (StringUtils.isNotBlank(form.getCertificateNumber())) {
            count = this.driverInfoService.getBaseMapper().selectCount(new QueryWrapper<DriverInfoEntity>().lambda()
                    .ne(DriverInfoEntity::getDriverId, form.getId())
                    .eq(DriverInfoEntity::getCertificateNumber, form.getCertificateNumber()));
            if (0 < count) {
                throw new EException(1004, errorMessageComponent.getCarrierDriverEdit1004());
            }
        }

        // 2:更新密码
        int flag = 0;
        if (StringUtils.isNotBlank(form.getPassword())) {
            DriverAccountEntity accountEntity = new DriverAccountEntity();
            accountEntity.setId(form.getId());
            accountEntity.setPassword(passwordComponent.generatePwd4Salt(form.getPassword(), driverAccountEntity.getSalt()));
            flag += this.baseMapper.updateById(accountEntity);
        }

        // 3:更新帐号信息
        DriverInfoEntity driverInfoEntity = new DriverInfoEntity();
        BeanUtils.copyProperties(form, driverInfoEntity);
        flag += this.driverInfoService.getBaseMapper().update(driverInfoEntity, new UpdateWrapper<DriverInfoEntity>().lambda()
                .eq(DriverInfoEntity::getDriverId, form.getId()));

        return flag;
    }
}