package com.esv.freight.customer.module.goodsowner.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.esv.freight.customer.common.component.ErrorMessageComponent;
import com.esv.freight.customer.common.exception.EException;
import com.esv.freight.customer.common.util.FeignUtils;
import com.esv.freight.customer.common.util.ReqUtils;
import com.esv.freight.customer.feign.FeignBaseService;
import com.esv.freight.customer.module.goodsowner.GoodsOwnerConstants;
import com.esv.freight.customer.module.goodsowner.dao.AccountDao;
import com.esv.freight.customer.module.goodsowner.entity.AccountEntity;
import com.esv.freight.customer.module.goodsowner.entity.AuditHistoryEntity;
import com.esv.freight.customer.module.goodsowner.entity.InfoEntity;
import com.esv.freight.customer.module.goodsowner.form.AccountInfoForm;
import com.esv.freight.customer.module.goodsowner.service.AccountService;
import com.esv.freight.customer.module.goodsowner.service.AuditHistoryService;
import com.esv.freight.customer.module.goodsowner.service.InfoService;
import com.esv.gateway.common.GatewayHeaders;
import org.apache.commons.codec.digest.DigestUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.UUID;


@Service("accountService")
public class AccountServiceImpl extends ServiceImpl<AccountDao, AccountEntity> implements AccountService {

    private FeignBaseService feignBaseService;

    private ErrorMessageComponent errorMessageComponent;

    private InfoService infoService;

    private AuditHistoryService auditHistoryService;

    @Autowired
    public AccountServiceImpl(FeignBaseService feignBaseService, ErrorMessageComponent errorMessageComponent, InfoService infoService, AuditHistoryService auditHistoryService) {
        this.feignBaseService = feignBaseService;
        this.errorMessageComponent = errorMessageComponent;
        this.infoService = infoService;
        this.auditHistoryService = auditHistoryService;
    }

    @Override
    public AccountEntity getAccountRecordByAccount(String account) {
        AccountEntity queryEntity = new AccountEntity();
        queryEntity.setAccount(account);
        QueryWrapper<AccountEntity> queryWrapper = new QueryWrapper<>(queryEntity);
        return this.baseMapper.selectOne(queryWrapper);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public int createAccountByManager(AccountInfoForm form) {
        // 1.判断帐号是否已存在
        AccountEntity accountEntity = this.getAccountRecordByAccount(form.getAccount());
        if (null != accountEntity) {
            throw new EException(1001, errorMessageComponent.goodsOwnerAccountAdd1001);
        } else {
            accountEntity = new AccountEntity();
        }

        // 2.获取客户编码
        JSONObject batchIdReqJson = new JSONObject();
        batchIdReqJson.put("prefix", "HZ");
        batchIdReqJson.put("formatter", "yyyyMMdd");
        batchIdReqJson.put("length", 13);
        JSONObject batchIdResJson;
        try {
            batchIdResJson = FeignUtils.getFeignResultData(feignBaseService.getBatchId(batchIdReqJson));
        } catch (Exception e) {
            log.error("调用[基础服务]生成客户编号失败");
            throw new EException("生成客户编号时发生错误");
        }
        String ownerNumber = batchIdResJson.getString("batchId");

        // 3.新增帐号
        BeanUtils.copyProperties(form, accountEntity);
        String salt = UUID.randomUUID().toString().replace("-", "");
        String password = DigestUtils.md5Hex(form.getPassword() + salt);
        accountEntity.setSalt(salt);
        accountEntity.setPassword(password);
        accountEntity.setSourceType(GoodsOwnerConstants.OWNER_SOURCE_TYPE_PLATFORM);
        accountEntity.setAuditStatus(GoodsOwnerConstants.OWNER_AUDIT_STATUS_SUCCESS);
        int accountId = this.baseMapper.insert(accountEntity);

        // 4.新增帐号信息
        InfoEntity infoEntity = new InfoEntity();
        BeanUtils.copyProperties(form, infoEntity);
        // 货主为个人
        if (GoodsOwnerConstants.OWNER_TYPE_PERSONAL.equals(form.getOwnerType())) {
            infoEntity.setOwnerFullName(null);
            infoEntity.setOwnerBriefName(null);
            infoEntity.setUniCreditCode(null);
            infoEntity.setCreditOriginalUrl(null);
            infoEntity.setCreditCopyUrl(null);
            infoEntity.setCreditExpireDate(null);
            infoEntity.setLegalPerson(null);
            infoEntity.setLegalPhone(null);
        }
        infoEntity.setOwnerNumber(ownerNumber);
        infoEntity.setAccountId(Long.parseLong(String.valueOf(accountId)));
        infoService.getBaseMapper().insert(infoEntity);

        // 5.新增帐号审核记录
        AuditHistoryEntity auditHistoryEntity = new AuditHistoryEntity();
        auditHistoryEntity.setAccountId(Long.parseLong(String.valueOf(accountId)));
        auditHistoryEntity.setAuditStatus(GoodsOwnerConstants.OWNER_AUDIT_STATUS_SUCCESS);
        auditHistoryEntity.setOperateUser(ReqUtils.getRequestHeader(GatewayHeaders.USER_ACCOUNT));
        auditHistoryService.getBaseMapper().insert(auditHistoryEntity);

        return accountId;
    }
}