package com.esv.freight.customer.module.goodsowner.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.Wrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.esv.freight.customer.common.component.ErrorMessageComponent;
import com.esv.freight.customer.common.component.PasswordComponent;
import com.esv.freight.customer.common.exception.EException;
import com.esv.freight.customer.common.util.FeignUtils;
import com.esv.freight.customer.common.util.ReqUtils;
import com.esv.freight.customer.common.vo.PageResultVO;
import com.esv.freight.customer.feign.FeignBaseService;
import com.esv.freight.customer.module.goodsowner.GoodsOwnerConstants;
import com.esv.freight.customer.module.goodsowner.dao.AccountDao;
import com.esv.freight.customer.module.goodsowner.dto.AccountInfoDto;
import com.esv.freight.customer.module.goodsowner.entity.GoodsOwnerAccountEntity;
import com.esv.freight.customer.module.goodsowner.entity.AuditHistoryEntity;
import com.esv.freight.customer.module.goodsowner.entity.InfoEntity;
import com.esv.freight.customer.module.goodsowner.form.AccountForm;
import com.esv.freight.customer.module.goodsowner.form.AccountInfoForm;
import com.esv.freight.customer.module.goodsowner.form.AccountQueryForm;
import com.esv.freight.customer.module.goodsowner.service.GoodsOwnerAccountService;
import com.esv.freight.customer.module.goodsowner.service.AuditHistoryService;
import com.esv.freight.customer.module.goodsowner.service.InfoService;
import com.esv.freight.customer.module.goodsowner.vo.AccountAvailableVO;
import com.esv.freight.customer.module.goodsowner.vo.AccountListVO;
import com.esv.gateway.common.GatewayHeaders;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;


@Service("goodsOwnerAccountService")
@Slf4j
public class GoodsOwnerAccountServiceImpl extends ServiceImpl<AccountDao, GoodsOwnerAccountEntity> implements GoodsOwnerAccountService {

    private FeignBaseService feignBaseService;

    private PasswordComponent passwordComponent;

    private ErrorMessageComponent errorMessageComponent;

    private InfoService infoService;

    private AuditHistoryService auditHistoryService;

    @Autowired
    public GoodsOwnerAccountServiceImpl(FeignBaseService feignBaseService, PasswordComponent passwordComponent,
                                        ErrorMessageComponent errorMessageComponent, InfoService infoService, AuditHistoryService auditHistoryService) {
        this.feignBaseService = feignBaseService;
        this.passwordComponent = passwordComponent;
        this.errorMessageComponent = errorMessageComponent;
        this.infoService = infoService;
        this.auditHistoryService = auditHistoryService;
    }

    @Override
    public GoodsOwnerAccountEntity getAccountRecordByAccount(String account) {
        GoodsOwnerAccountEntity queryEntity = new GoodsOwnerAccountEntity();
        queryEntity.setAccount(account);
        QueryWrapper<GoodsOwnerAccountEntity> queryWrapper = new QueryWrapper<>(queryEntity);
        return this.baseMapper.selectOne(queryWrapper);
    }

    @Override
    public GoodsOwnerAccountEntity getAccountRecordById(Long id) {
        GoodsOwnerAccountEntity queryEntity = new GoodsOwnerAccountEntity();
        queryEntity.setId(id);
        QueryWrapper<GoodsOwnerAccountEntity> queryWrapper = new QueryWrapper<>(queryEntity);
        return this.baseMapper.selectOne(queryWrapper);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long createAccountByManager(AccountInfoForm form) {
        // 1.判断帐号是否已存在
        GoodsOwnerAccountEntity goodsOwnerAccountEntity = this.getAccountRecordByAccount(form.getAccount());
        if (null != goodsOwnerAccountEntity) {
            throw new EException(1001, errorMessageComponent.getGoodsOwnerAccountAdd1001());
        } else {
            goodsOwnerAccountEntity = new GoodsOwnerAccountEntity();
        }

        // 2.获取客户编码
        JSONObject batchIdReqJson = new JSONObject();
        batchIdReqJson.put("prefix", "HZ");
        batchIdReqJson.put("formatter", "yyyyMMdd");
        batchIdReqJson.put("length", 13);
        JSONObject batchIdResJson;
        try {
            batchIdResJson = FeignUtils.getFeignResultData(feignBaseService.getBatchId(batchIdReqJson));
        } catch (Exception e) {
            log.error("调用[基础服务]生成客户编号失败：{}", e.getMessage(), e);
            throw new EException("生成客户编号时发生错误");
        }
        String ownerNumber = batchIdResJson.getString("batchId");

        // 3.新增帐号
        BeanUtils.copyProperties(form, goodsOwnerAccountEntity);
        goodsOwnerAccountEntity.setId(null);
        String salt = passwordComponent.generateAccountPwdSalt();
        goodsOwnerAccountEntity.setSalt(salt);
        goodsOwnerAccountEntity.setPassword(passwordComponent.generatePwd4Salt(form.getPassword(), salt));
        goodsOwnerAccountEntity.setAccountStatus(GoodsOwnerConstants.OWNER_ACCOUNT_STATUS_UNBLOCK);
        goodsOwnerAccountEntity.setSourceType(GoodsOwnerConstants.OWNER_SOURCE_TYPE_PLATFORM);
        goodsOwnerAccountEntity.setAuditStatus(GoodsOwnerConstants.OWNER_AUDIT_STATUS_SUCCESS);
        this.baseMapper.insert(goodsOwnerAccountEntity);
        Long accountId = goodsOwnerAccountEntity.getId();

        // 4.新增帐号信息
        InfoEntity infoEntity = new InfoEntity();
        BeanUtils.copyProperties(form, infoEntity);
        // 货主为个人
        if (GoodsOwnerConstants.OWNER_TYPE_PERSONAL.equals(form.getOwnerType())) {
            infoEntity.setOwnerFullName(null);
            infoEntity.setOwnerBriefName(null);
            infoEntity.setUniCreditCode(null);
            infoEntity.setCreditOriginalUrl(null);
            infoEntity.setCreditCopyUrl(null);
            infoEntity.setCreditExpireDate(null);
            infoEntity.setLegalPerson(null);
            infoEntity.setLegalPhone(null);
        }
        infoEntity.setOwnerNumber(ownerNumber);
        infoEntity.setAccountId(accountId);
        infoService.getBaseMapper().insert(infoEntity);

        // 5.新增帐号审核记录
        AuditHistoryEntity auditHistoryEntity = new AuditHistoryEntity();
        auditHistoryEntity.setAccountId(accountId);
        auditHistoryEntity.setAuditStatus(GoodsOwnerConstants.OWNER_AUDIT_STATUS_SUCCESS);
        auditHistoryEntity.setOperateUser(ReqUtils.getRequestHeader(GatewayHeaders.USER_ACCOUNT));
        auditHistoryEntity.setRemark("平台创建自动审核");
        auditHistoryService.getBaseMapper().insert(auditHistoryEntity);

        return accountId;
    }

    @Override
    public void updateAccountInfo(AccountInfoForm form) {
        // 1.判断帐号是否存在
        Long accountId = form.getId();
        GoodsOwnerAccountEntity goodsOwnerAccountEntity = this.getAccountRecordById(accountId);
        if (null == goodsOwnerAccountEntity) {
            throw new EException(1001, errorMessageComponent.getGoodsOwnerAccountEdit1001());
        }

        // 2.更新表：goods_owner_account，更新密码
        String password = form.getPassword();
        if (StringUtils.isNotBlank(password)) {
            GoodsOwnerAccountEntity goodsOwnerAccountEntityUpdate = new GoodsOwnerAccountEntity();
            goodsOwnerAccountEntityUpdate.setId(accountId);
            password = DigestUtils.md5Hex(form.getPassword() + goodsOwnerAccountEntity.getSalt());
            goodsOwnerAccountEntityUpdate.setPassword(password);
            this.baseMapper.updateById(goodsOwnerAccountEntityUpdate);
        }

        // 3.更新表：goods_owner_info
        InfoEntity infoEntity = new InfoEntity();
        BeanUtils.copyProperties(form, infoEntity);
        // 货主类型不允许修改
        infoEntity.setOwnerType(null);
        // 货主为个人
        if (GoodsOwnerConstants.OWNER_TYPE_PERSONAL.equals(form.getOwnerType())) {
            infoEntity.setOwnerFullName(null);
            infoEntity.setOwnerBriefName(null);
            infoEntity.setUniCreditCode(null);
            infoEntity.setCreditOriginalUrl(null);
            infoEntity.setCreditCopyUrl(null);
            infoEntity.setCreditExpireDate(null);
            infoEntity.setLegalPerson(null);
            infoEntity.setLegalPhone(null);
        }
        InfoEntity infoEntityWrapper = new InfoEntity();
        infoEntityWrapper.setAccountId(accountId);
        Wrapper<InfoEntity> InfoUpdateWrapper = new UpdateWrapper<>(infoEntityWrapper);
        infoService.getBaseMapper().update(infoEntity, InfoUpdateWrapper);
    }

    @Override
    public void blockAccount(Long id) {
        // 判断帐号是否存在
        GoodsOwnerAccountEntity goodsOwnerAccountEntity = this.getAccountRecordById(id);
        if (null == goodsOwnerAccountEntity) {
            throw new EException(1001, errorMessageComponent.getGoodsOwnerAccountBlock1001());
        } else {
            if (GoodsOwnerConstants.OWNER_ACCOUNT_STATUS_BLOCK.equals(goodsOwnerAccountEntity.getAccountStatus())) {
                throw new EException(1002, errorMessageComponent.getGoodsOwnerAccountBlock1002());
            }
        }

        // 停用帐号
        GoodsOwnerAccountEntity goodsOwnerAccountEntityUpdate = new GoodsOwnerAccountEntity();
        goodsOwnerAccountEntityUpdate.setId(id);
        goodsOwnerAccountEntityUpdate.setAccountStatus(GoodsOwnerConstants.OWNER_ACCOUNT_STATUS_BLOCK);
        this.baseMapper.updateById(goodsOwnerAccountEntityUpdate);
    }

    @Override
    public void unblockAccount(Long id) {
        // 判断帐号是否存在
        GoodsOwnerAccountEntity goodsOwnerAccountEntity = this.getAccountRecordById(id);
        if (null == goodsOwnerAccountEntity) {
            throw new EException(1001, errorMessageComponent.getGoodsOwnerAccountUnblock1001());
        } else {
            if (GoodsOwnerConstants.OWNER_ACCOUNT_STATUS_UNBLOCK.equals(goodsOwnerAccountEntity.getAccountStatus())) {
                throw new EException(1002, errorMessageComponent.getGoodsOwnerAccountUnblock1002());
            }
        }

        // 启用帐号
        GoodsOwnerAccountEntity goodsOwnerAccountEntityUpdate = new GoodsOwnerAccountEntity();
        goodsOwnerAccountEntityUpdate.setId(id);
        goodsOwnerAccountEntityUpdate.setAccountStatus(GoodsOwnerConstants.OWNER_ACCOUNT_STATUS_UNBLOCK);
        this.baseMapper.updateById(goodsOwnerAccountEntityUpdate);
    }

    @Override
    public void auditAccount(AccountForm form) {
        // 1.判断帐号是否存在
        Long accountId = form.getId();
        GoodsOwnerAccountEntity goodsOwnerAccountEntity = this.getAccountRecordById(accountId);
        if (null == goodsOwnerAccountEntity) {
            throw new EException(1001, errorMessageComponent.getGoodsOwnerAccountAudit1001());
        } else {
            if (GoodsOwnerConstants.OWNER_AUDIT_STATUS_SUCCESS.equals(goodsOwnerAccountEntity.getAuditStatus())) {
                throw new EException(1002, errorMessageComponent.getGoodsOwnerAccountAudit1002());
            }
        }

        // 2.更新帐号审核状态
        GoodsOwnerAccountEntity goodsOwnerAccountEntityUpdate = new GoodsOwnerAccountEntity();
        goodsOwnerAccountEntityUpdate.setId(accountId);
        goodsOwnerAccountEntityUpdate.setAuditStatus(form.getAuditStatus());
        this.baseMapper.updateById(goodsOwnerAccountEntityUpdate);

        // 3.更新帐号审核历史
        AuditHistoryEntity auditHistoryEntity = new AuditHistoryEntity();
        auditHistoryEntity.setAccountId(accountId);
        auditHistoryEntity.setAuditStatus(form.getAuditStatus());
        auditHistoryEntity.setRemark(form.getRemark());
        auditHistoryEntity.setOperateUser(ReqUtils.getRequestHeader(GatewayHeaders.USER_ACCOUNT));
        auditHistoryService.getBaseMapper().insert(auditHistoryEntity);
    }

    @Override
    public List<AuditHistoryEntity> getAuditHistory(Long id) {
        // 判断帐号是否存在
        GoodsOwnerAccountEntity goodsOwnerAccountEntity = this.getAccountRecordById(id);
        if (null == goodsOwnerAccountEntity) {
            throw new EException(1001, errorMessageComponent.getGoodsOwnerAccountAuditHistory1001());
        }

        // 查询审核历史
        QueryWrapper<AuditHistoryEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("account_id", id);
        queryWrapper.orderByAsc("operate_time");
        return this.auditHistoryService.getBaseMapper().selectList(queryWrapper);
    }

    @Override
    public AccountInfoDto getAccountInfo(AccountForm form) {
        AccountInfoDto dto = null;
        if (null != form.getId()) {
            dto = this.baseMapper.getAccountInfoById(form.getId());
        } else if (null != form.getAccount()) {
            dto = this.baseMapper.getAccountInfoByAccount(form.getAccount());
        } else {

        }

        return dto;
    }

    @Override
    public Long registerAccount(String account) {
        // 1.判断帐号是否已存在
        GoodsOwnerAccountEntity goodsOwnerAccountEntity = this.getAccountRecordByAccount(account);
        if (null != goodsOwnerAccountEntity) {
            throw new EException(1001, errorMessageComponent.getGoodsOwnerAccountRegister1001());
        } else {
            goodsOwnerAccountEntity = new GoodsOwnerAccountEntity();
        }

        // 2.获取客户编码
        JSONObject batchIdReqJson = new JSONObject();
        batchIdReqJson.put("prefix", "HZ");
        batchIdReqJson.put("formatter", "yyyyMMdd");
        batchIdReqJson.put("length", 13);
        JSONObject batchIdResJson;
        try {
            batchIdResJson = FeignUtils.getFeignResultData(feignBaseService.getBatchId(batchIdReqJson));
        } catch (Exception e) {
            log.error("调用[基础服务]生成客户编号失败：{}", e.getMessage(), e);
            throw new EException("生成客户编号时发生错误");
        }
        String ownerNumber = batchIdResJson.getString("batchId");

        // 3.新增帐号
        goodsOwnerAccountEntity.setAccount(account);
        goodsOwnerAccountEntity.setSalt(passwordComponent.generateAccountPwdSalt());
        goodsOwnerAccountEntity.setAccountStatus(GoodsOwnerConstants.OWNER_ACCOUNT_STATUS_UNBLOCK);
        goodsOwnerAccountEntity.setSourceType(GoodsOwnerConstants.OWNER_SOURCE_TYPE_REGISTER);
        goodsOwnerAccountEntity.setAuditStatus(GoodsOwnerConstants.OWNER_AUDIT_STATUS_UNAUDITED);
        this.baseMapper.insert(goodsOwnerAccountEntity);
        Long accountId = goodsOwnerAccountEntity.getId();

        // 4.新增帐号信息
        InfoEntity infoEntity = new InfoEntity();
        infoEntity.setOwnerNumber(ownerNumber);
        infoEntity.setAccountId(accountId);
        infoService.getBaseMapper().insert(infoEntity);

        // 5.新增帐号审核记录
        AuditHistoryEntity auditHistoryEntity = new AuditHistoryEntity();
        auditHistoryEntity.setAccountId(accountId);
        auditHistoryEntity.setAuditStatus(GoodsOwnerConstants.OWNER_AUDIT_STATUS_UNAUDITED);
        auditHistoryEntity.setOperateUser(account);
        auditHistoryEntity.setRemark("用户通过终端APP注册帐号");
        auditHistoryService.getBaseMapper().insert(auditHistoryEntity);

        return accountId;
    }

    @Override
    public PageResultVO selectAccountInfoList(AccountQueryForm queryForm) {
        IPage<AccountInfoDto> page = new Page<>(queryForm.getPageNum(), queryForm.getPageSize());
        this.baseMapper.selectAccountInfoList(page, queryForm);

        // 数据转换
        List<AccountInfoDto> dtoList = page.getRecords();
        List<AccountListVO> targetRecordList = new ArrayList<>();
        for (AccountInfoDto dto : dtoList) {
            AccountListVO vo = new AccountListVO();
            BeanUtils.copyProperties(dto, vo);
            if (GoodsOwnerConstants.OWNER_TYPE_PERSONAL.equals(dto.getOwnerType())) {
                vo.setOwnerName(dto.getContactor());
            } else {
                vo.setOwnerName(dto.getOwnerFullName());
            }
            vo.setCreateTime(dto.getCreateTime().getTime());
            targetRecordList.add(vo);
        }

        return new PageResultVO(page, targetRecordList);
    }

    @Override
    public List<AccountAvailableVO> selectAllAvailableAccount() {
        List<AccountInfoDto> dtoList = this.baseMapper.selectAllAvailableAccount();
        List<AccountAvailableVO> targetRecordList = new ArrayList<>();
        for (AccountInfoDto dto : dtoList) {
            AccountAvailableVO vo = new AccountAvailableVO();
            BeanUtils.copyProperties(dto, vo);
            if (GoodsOwnerConstants.OWNER_TYPE_PERSONAL.equals(dto.getOwnerType())) {
                vo.setOwnerName(dto.getContactor());
            } else {
                vo.setOwnerName(dto.getOwnerFullName());
            }
            targetRecordList.add(vo);
        }

        return targetRecordList;
    }

    @Override
    public Integer resetAccountPwd(AccountForm form) {
        GoodsOwnerAccountEntity record = this.getAccountRecordById(form.getId());
        if (null == record) {
            throw new EException(1001, errorMessageComponent.getGoodsOwnerAccountPasswordReset1001());
        }
        GoodsOwnerAccountEntity goodsOwnerAccountEntity = new GoodsOwnerAccountEntity();
        goodsOwnerAccountEntity.setId(form.getId());
        goodsOwnerAccountEntity.setPassword(passwordComponent.generatePwd4Salt(form.getPassword(), record.getSalt()));

        return this.baseMapper.updateById(goodsOwnerAccountEntity);
    }
}